#pragma once
#include "Core/Fn.h"
#include "Core/Io/Stream.h"
#include "Core/Io/Url.h"
#include "Core/Graphics/Image.h"
#include "Core/Graphics/ImageSet.h"
#include "Core/Exception.h"
#include "Options.h"

namespace graphics {

	/**
	 * Description of a supported file format. Contains pointers to load- and save functionality.
	 *
	 * The idea is that this class contains functions that produce the relevant data types.
	 */
	class ImageFormat : public Object {
		STORM_CLASS;
	public:
		// Create.
		STORM_CTOR ImageFormat(Str *name, Array<Str *> *extensions,
							Fn<Bool, IStream *> *applicable,
							Fn<FormatOptions *> *options);

		// Create from C++. 'extensions' is null-terminated.
		typedef Bool (CODECALL *CppIsApplicable)(IStream *);
		typedef FormatOptions *(CODECALL *CppOptions)(ImageFormat *);
		ImageFormat(const wchar *name, const wchar **extensions,
					CppIsApplicable applicable, CppOptions options);

		// Get the name of this format.
		Str *STORM_FN name();

		// Get extensions used for this format.
		Array<Str *> *STORM_FN extensions();

		// Check if the provided extension is appropriate for this format.
		Bool STORM_FN hasExtension(Str *ext);

		// Check if the current format is relevant for this file.
		Bool STORM_FN applicable(IStream *in);

		// Create an instance of the format options for this format. Supplies default options.
		FormatOptions *STORM_FN options();

		// Load an image using this format. Equivalent to `options()->load()`. This does not allow
		// reading properties of the decoding process.
		Image *STORM_FN load(IStream *from);
		Image *STORM_FN load(Url *from);

		// Load a set of images using this format. Equivalent to `options()->loadSet()`. This does
		// not allow reading properties of the decoding process.
		ImageSet *STORM_FN loadSet(IStream *from);
		ImageSet *STORM_FN loadSet(Url *from);

		// Save an image using this format. Equivalent to `options()->save()`, but does not allow
		// specifying options.
		void STORM_FN save(Image *image, OStream *to);
		void STORM_FN save(Image *image, Url *to);

		// Print a string representation.
		void STORM_FN toS(StrBuf *to) const;

	private:
		// Name of this format.
		Str *fmtName;

		// Extensions supported by this format.
		Array<Str *> *fmtExtensions;

		// Function pointers for creating property data:
		Fn<Bool, IStream *> *isApplicable;
		Fn<FormatOptions *> *createOptions;
	};


	// Get a list of all supported formats.
	Array<ImageFormat *> *STORM_FN supportedImageFormats(EnginePtr e);

	// Register support for a new image format. Used from other libraries if necessary.
	void STORM_FN registerImageFormat(EnginePtr e, ImageFormat *format);


	// Find an appropriate image format based on the header at the current start of the stream.
	MAYBE(ImageFormat *) STORM_FN findFormat(IStream *from);

	// Load an image. Automatically detect the format based on the header of the file.
	// Run `supportedImageFormats` to get a list of supported image formats.
	Image *STORM_FN loadImage(IStream *from);
	Image *STORM_FN loadImage(Url *file);

	// Load a set of images. Like 'loadImage', detects the format based on the header of the file.
	ImageSet *STORM_FN loadImageSet(IStream *from);
	ImageSet *STORM_FN loadImageSet(Url *file);


	// Find an appropriate image format based on a file extension.
	MAYBE(ImageFormat *) STORM_FN findFormat(Str *fileExt);

	// Save an image. Use the file extension to find an appropriate image format. If you wish to
	// have more control of the save process, you can get a list of supported formats by calling
	// `supportedImageFormats`, or simply by creating the appropriate format directly yourself.
	void STORM_FN saveImage(Image *image, Url *file);

}
