///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// gnuplot output
//
// authors:
//      Jocelyn.Etienne@ujf-grenoble.fr
//      Pierre.Saramito@ujf-grenoble.fr
//
// date: 23 september 2008
//
#include "rheolef/field.h"
#include "rheolef/iorheo.h"
#include "rheolef/rheostream.h"
#include "rheolef/tiny_matvec.h"
using namespace rheolef;
using namespace std;

// ============================================================================
// 1D output
// ============================================================================
// output with continuous line path on 1D mesh:
//
// motivation : by default elements are not supposed to be sorted from
//  left to right or, for 2d isoline, not sorted at all.
// by a brutal element-per-element output, lines element could very small
// and thus dotted and discontinuous styles in gnuplot seems continuous.
// Thus, compute the connected components of the mesh and output it in a
// continuous line style. The connected components are computed by hazel_1d,
// the localizator of 1d meshes.
//
void
field::extract_line_path (const vector<size_t>& mask, vector<point>& x, vector<Float>& y) const
// draw one path (x,y) from the graph (x,u(x)) by using mask on geometry
// assume masked geometry is one connex component and with increasing x(i)
{
  geo::size_type           n  = mask.size();
  geo::const_iterator      K  = get_geo().begin();
  geo::const_iterator_node gx = get_geo().begin_node();
  if (get_approx() == "P0") {
      x.resize(2*n);
      y.resize(2*n);
      for (size_t i = 0; i < n; i++) {
	size_t pi  = mask[i];
	x[2*i]   = gx[K[pi][0]];
        x[2*i+1] = gx[K[pi][1]];
        y[2*i]   = at(pi);
	y[2*i+1] = at(pi);
      }
  } else if (get_approx() == "P1d") {
      x.resize(2*n);
      y.resize(2*n);
      for (size_t i = 0; i < n; i++) {
	size_t pi  = mask[i];
	x[2*i]   = gx[K[pi][0]];
	x[2*i+1] = gx[K[pi][1]];
        y[2*i]   = at(2*pi);
        y[2*i+1] = at(2*pi+1);
      }
  } else if (get_approx() == "P1") {
      x.resize(n+1);
      y.resize(n+1);
      const geo& g = get_geo();
      const space& V = get_space();
      tiny_vector<size_t> dof;
      for (size_t i = 0; i < n; i++) {
	size_t pi  = mask[i];
        const geo_element& Ki = K[pi];
        V.set_dof(Ki, dof) ;
	x[i] = V.x_dof(Ki, 0);
	y[i] = at(dof(0));
	if (i == n-1) {
	  x[n] = V.x_dof(Ki, 1);
	  y[n] = at(dof(1));
	}
      }
  } else if (get_approx() == "P2") {
      const space& V = get_space();
      tiny_vector<size_t> dof;
      x.resize(2*n+1);
      y.resize(2*n+1);
      for (size_t i = 0; i < n; i++) {
	size_t pi  = mask[i];
        const geo_element& Ki = K[pi];
        V.set_dof(Ki, dof) ;
	x[2*i]   = V.x_dof(Ki, 0);
	x[2*i+1] = V.x_dof(Ki, 2);
	y[2*i]   = at(dof(0));
	y[2*i+1] = at(dof(2));
	if (i == n-1) {
	  x[2*n] = V.x_dof(Ki, 1);
	  y[2*n] = at(dof(1));
	}
      }
  } else if (get_approx() == "H3") {
      const space& V = get_space();
      tiny_vector<size_t> dof;
      x.resize(3*n+1);
      y.resize(3*n+1);
      point a,b;
      for (size_t i = 0; i < n; i++) {
	size_t pi  = mask[i];
        const geo_element& Ki = K[pi];
        V.set_dof(Ki, dof) ;
	a = V.x_dof(Ki, 0);
	b = V.x_dof(Ki, 1);
	x[3*i]   = a;
	x[3*i+1] = (2.*a+b)/3.;
	x[3*i+2] = (a+2.*b)/3.;
	y[3*i]   = at(dof(0));
	y[3*i+1] = evaluate(meshpoint(point(1/3.),pi));
	y[3*i+2] = evaluate(meshpoint(point(2/3.),pi));
	if (i == n-1) {
	  x[3*n] = b;
	  y[3*n] = at(dof(1));
	}
      }
  } else {
      error_macro("unrecognized approximation " << get_approx()
	   << " for 1D line path computation");
  }
}
void
field::put_gnuplot1d_data (ostream& gdat) const
{
  int digits10 = numeric_limits<Float>::digits10;
  gdat << setprecision(digits10);
  vector<point> x;
  vector<Float> y;
  hazel_1d path (get_geo().data());
  for (size_t i = 0; i < path.size(); i++) {
    extract_line_path (path[i], x, y);
    for (size_t j = 0; j < x.size(); j++)
      gdat << x[j][0] << " " << y[j] << endl;
    gdat << endl;
  }
  gdat << endl;
}
int
field::gnuplot1d (
    const string& basename,
    bool execute, bool clean, bool verbose
    ) const 
{
  //
  // header
  //
  string plot_name = basename + ".plot";
  ofstream plot (plot_name.c_str());
  if (verbose) clog << "! file \"" << plot_name << "\" created.\n";
  
  string gdat_name = basename + ".gdat";
  ofstream gdat (gdat_name.c_str());
  if (verbose) clog << "! file \"" << gdat_name << "\" created.\n";
  put_gnuplot1d_data (gdat);
  gdat.close();

  plot << "#!gnuplot" << endl << endl;
  plot << "#set logscale xy" << endl;
  plot << "#set size square" << endl << endl;
  plot << "plot \"" << gdat_name << "\" w lines linetype 1" << endl;
  plot << "pause -1 \"<return>\"" << endl << endl;
  plot <<  endl;
  plot.close();
  //
  // run gnuplot
  //
  int status = 0;
  if (execute) {
      string command = "gnuplot -persist " + basename + ".plot";
      if (verbose) {
          clog << "! " << command << endl;
	  command += " 1>&2";
      } else {
	  command += " >/dev/null";
      }
      cin.sync();
      status = system (command.c_str());
  }
  //
  // clear gnuplot data
  //
  if (clean) {
      string command 
	= "/bin/rm -f " + basename + ".plot " + basename + ".gdat" ;
      if (verbose) clog << "! " << command << endl;
      status |= system (command.c_str());
  }
  return status;
}
// ============================================================================
// 2D output: isolines or elevation
// ============================================================================
int
field::gnuplot2d_elevation (const string& basename, plot_options& opt) const
{
  string plot_name = basename + ".plot";
  ofstream plot (plot_name.c_str());
  if (opt.verbose) clog << "! file \"" << plot_name << "\" created.\n";
  const geo& g = get_geo();

  plot << "#!gnuplot\n";
  if (!opt.bare)
  	plot << "set title \"" << basename << "\"" << endl;
  else
  	plot << "set nokey" << endl;
  if (opt.empty) 
   {
   	plot << "set noborder" << endl;
   	plot << "set noxtics" << endl;
   	plot << "set noytics" << endl;
   }
  if (opt.xmax==opt.xmin) 
   { 
   	opt.xmin=g.xmin(); 
	opt.xmax=g.xmax(); 
	point diag = opt.xmax-opt.xmin;
	Float w=opt.border*::sqrt(sqr(diag[0])+sqr(diag[1]));
	if (opt.xmirror) opt.xmin[0]=-opt.xmax[0];
	if (opt.ymirror) opt.xmin[1]=-opt.xmax[1];
   }
  if (opt.map)
   {
   	plot << "set view map" << endl;
	plot << "set lmargin at screen 0.10" << endl
	     << "set rmargin at screen 0.85" << endl
	     << "set bmargin at screen 0.15" << endl
	     << "set tmargin at screen 0.95" << endl;
	if (opt.equal_scales) plot << "set size ratio -1" << endl;
   }
  if (opt.postscript && opt.noxplot)  
   {
	plot << "set output '" << basename << ".eps'" << endl;
	g.write_gnuplot_postscript_options(plot,opt);
   }
  ofstream gdat;
  string coma=",";
  plot << "f(c)=column(c) #can be modified to plot anything, e.g. column(4)-column(1) yields f(x,y)-x" << endl;
  point spacer(1e-5,1e-5); spacer=norm(opt.xmax-opt.xmin)*spacer; 
  if (opt.meshplot) plot << "load \"" << basename << ".mesh.plot\"" << endl << "replot "; 
  else if (opt.keep_user_range) plot << "splot ";
  else plot << "splot ["<<opt.xmin[0]-spacer[0]<<":"<<opt.xmax[0]+spacer[0]<<"]["
  			<<opt.xmin[1]-spacer[1]<<":"<<opt.xmax[1]+spacer[1]<<"] ";
  plot << "\"" << basename 
  		<< ".gdat\" using 1:2:(f(4)) title \"" 
		<< basename << "\" with lines palette \\\n";
  if (opt.xmirror) plot << ", \"" << basename 
		<< ".gdat\" using (-$1):2:(f(4)) notitle with lines palette \\\n";
  if (opt.ymirror) plot << ", \"" << basename 
  	<< ".gdat\" using 1:(-$2):(f(4)) notitle with lines palette \\\n";
  if (opt.xmirror && opt.ymirror) plot << ", \"" << basename 
  	<< ".gdat\" using (-$1):(-$2):(f(4)) notitle with lines palette \\\n";
  plot << endl;
   
  string gdat_name = basename + ".gdat";
  gdat.open (gdat_name.c_str());
  if (opt.verbose) clog << "! file \"" << gdat_name << "\" created.\n";
  //
  // plot values
  //
  tiny_vector<size_type> dof0;
  geo::const_iterator last_K = g.end();
  vector<bool> done(get_space().size(), false);
  if (get_space().get_basis().family() == element_constant::Lagrange)
  for (geo::const_iterator iter_K = g.begin(); iter_K != last_K; iter_K++) {
      const geo_element& K = *iter_K;
      get_space().set_dof(K, dof0);
      for (geo::size_type i = 0; i <= K.size(); i++) {
          point x_i = get_space().x_dof(K, i % K.size());
          Float u0_i = at(dof0(i % K.size()));
          gdat << x_i << "\t" << u0_i << endl;
      }
      gdat  << endl;
      for (geo::size_type i = K.size(); i < dof0.size(); i++) {
          point x_i = get_space().x_dof(K, i );
          Float u0_i = at(dof0(i));
          gdat << x_i << "\t" << u0_i << endl << endl;
      }
      gdat  << endl;
  }
  else if (get_space().get_basis().family() == element_constant::Hermite)
   {
	for (geo::const_iterator iter_K = g.begin(); iter_K != last_K; iter_K++) {
	      const geo_element& K = *iter_K;
	      get_space().set_dof(K, dof0);
	      // P1-iso-P3 in 1D, missing middle points in 2D/3D.
	      vector<point> hat_x;
	      get_space().get_basis().hat_node(K, hat_x);
	      point x_p = get_space().x_dof(K, 0);
	      point hat_x_p = hat_x[0];
	      Float u0_0 = at(dof0(0));
	      gdat << x_p << "\t" << u0_0 << endl;
	      for (geo::size_type i = 1; i <= K.size(); i++) {
		  point x_n = get_space().x_dof(K, i % K.size());
	          point hat_x_n = hat_x[i % K.size()];
		  point x_1 = (2.*x_p+x_n)/3.; 
		  point x_2 = (x_p+2.*x_n)/3.; 
		  Float u0_1 = evaluate((2.*hat_x_p+hat_x_n)/3., K.index());
		  Float u0_2 = evaluate((hat_x_p+2.*hat_x_n)/3., K.index());
		  Float u0_n = at(dof0(i % K.size()));
		  gdat << x_1 << "\t" << u0_1 << endl;
		  gdat << x_2 << "\t" << u0_2 << endl;
		  gdat << x_n << "\t" << u0_n << endl;
		  x_p=x_n;
		  hat_x_p=hat_x_n;
	      }
	      gdat  << endl;
	  }
   }
  else error_macro("Gnuplot output of " << get_space().get_approx() << " fields not implemented");
  //
  // end of plot
  //
  plot << endl; // because of the trailing "\"
  plot << endl; 
  if (opt.pause) plot << "pause -1 \"<return>\"\n";
  if (opt.postscript && (!opt.noxplot)) {
	plot << "set output '" << basename << ".eps'" << endl;
   	g.write_gnuplot_postscript_options(plot,opt);
	plot << "replot";
  }
  plot.close();
  //
  // run gnuplot
  //
  if (opt.meshplot) {
	plot_options optm(opt);
	optm.execute=false; 
	optm.postscript=false;
	optm.bare=true; 
	optm.clean=false;
	g.gnuplot2d (basename+".mesh", optm);
   }
  int status = 0;
  char command [1000];
  if (opt.execute) {
      sprintf (command, "gnuplot -persist %s.plot", basename.c_str());
      if (opt.verbose) clog << "! " << command << endl;
      cin.sync();
      status = system (command);
  }
  //
  // clear gnuplot data
  //
  if (opt.clean) {
      sprintf (command, "/bin/rm -f %s.plot %s.gdat", basename.c_str(), basename.c_str());
      if (opt.verbose) clog << "! " << command << endl;
      status = system (command);
  }
  return status;
}
