# inspect a fitted lavaan object

# backward compatibility -- wrapper around lavInspect
inspect.lavaan <- function(object, what = "free", ...) {
    lavInspect.lavaan(object              = object,
                      what                   = what,
                      add.labels             = TRUE,
                      add.class              = TRUE,
                      drop.list.single.group = TRUE)
}

# the `tech' version: no labels, full matrices, ... for further processing
lavTech.lavaan <- function(object, 
                           what                   = "free",
                           add.labels             = FALSE,
                           add.class              = FALSE,
                           list.by.group          = FALSE,  
                           drop.list.single.group = FALSE) {

    lavInspect.lavaan(object, what = what,
                      add.labels = add.labels, add.class = add.class,
                      list.by.group = list.by.group,
                      drop.list.single.group =  drop.list.single.group)
}

# the `user' version: with defaults for display only
lavInspect.lavaan <- function(object,
                              what                   = "free",
                              add.labels             = TRUE,
                              add.class              = TRUE,
                              list.by.group          = TRUE,
                              drop.list.single.group = TRUE) {

    # object must inherit from class lavaan
    stopifnot(inherits(object, "lavaan"))

    # only a single argument
    if(length(what) > 1) {
        stop("`what' arguments contains multiple arguments; only one is allowed")
    }

    # be case insensitive
    what <- tolower(what)


    #### model matrices, with different contents ####
    if(what == "free") {
        lav_object_inspect_modelmatrices(object, what = "free", 
            type = "free", add.labels = add.labels, add.class = add.class,
            list.by.group = list.by.group, 
            drop.list.single.group = drop.list.single.group)
    } else if(what == "impute" ||
              what == "imputed") { # just to ease the transition for semTools!
        object@imputed
    } else if(what == "partable" || what == "user") {
        lav_object_inspect_modelmatrices(object, what = "free", 
            type="partable", add.labels = add.labels, add.class = add.class,
            list.by.group = list.by.group, 
            drop.list.single.group = drop.list.single.group)
    } else if(what == "se" ||
              what == "std.err" ||
              what == "standard.errors") {
        lav_object_inspect_modelmatrices(object, what = "se",
            add.labels = add.labels, add.class = add.class,
            list.by.group = list.by.group, 
            drop.list.single.group = drop.list.single.group)
    } else if(what == "start" || what == "starting.values") {
        lav_object_inspect_modelmatrices(object, what = "start",
            add.labels = add.labels, add.class = add.class,
            list.by.group = list.by.group, 
            drop.list.single.group = drop.list.single.group)
    } else if(what == "est"  || what == "estimates" ||
              what == "coef" || what == "coefficients" ||
              what == "x") {
        lav_object_inspect_modelmatrices(object, what = "est",
            add.labels = add.labels, add.class = add.class,
            list.by.group = list.by.group, 
            #list.by.group = FALSE, for semTools only
            drop.list.single.group = drop.list.single.group)
    } else if(what == "dx.free") {
        lav_object_inspect_modelmatrices(object, what = "dx.free",
            add.labels = add.labels, add.class = add.class,
            list.by.group = list.by.group, 
            drop.list.single.group = drop.list.single.group)
    } else if(what == "dx.all") {
        lav_object_inspect_modelmatrices(object, what = "dx.all",
            add.labels = add.labels, add.class = add.class,
            list.by.group = list.by.group, 
            drop.list.single.group = drop.list.single.group)
    } else if(what == "std" || what == "std.all" || what == "standardized") {
        lav_object_inspect_modelmatrices(object, what = "std.all",
           add.labels = add.labels, add.class = add.class,
           list.by.group = list.by.group, 
           drop.list.single.group = drop.list.single.group)
    } else if(what == "std.lv") {
        lav_object_inspect_modelmatrices(object, what = "std.lv",
           add.labels = add.labels, add.class = add.class,
           list.by.group = list.by.group, 
           drop.list.single.group = drop.list.single.group)
    } else if(what == "std.nox") {
        lav_object_inspect_modelmatrices(object, what = "std.nox",
           add.labels = add.labels, add.class = add.class,
           list.by.group = list.by.group, 
           drop.list.single.group = drop.list.single.group)


    #### parameter table ####
    } else if(what == "list") {
        parTable(object)

    #### fit indices ####
    } else if(what == "fit" ||
              what == "fitmeasures" ||
              what == "fit.measures" ||
              what == "fit.indices") {
        fitMeasures(object)


    #### modification indices ####
    } else if(what == "mi" ||
              what == "modindices" ||
              what == "modification.indices") {
        modificationIndices(object)


    #### sample statistics #####
    } else if(what == "obs" ||
              what == "observed" ||
              what == "sampstat" ||
              what == "sampstats" ||
              what == "samplestats" ||
              what == "samp" ||
              what == "sample" ||
              what == "samplestatistics") {
        lav_object_inspect_sampstat(object, h1 = FALSE,
            add.labels = add.labels, add.class = add.class, 
            drop.list.single.group = drop.list.single.group)
    } else if(what == "h1" || what == "missing.h1" || what == "sampstat.h1") {
        lav_object_inspect_sampstat(object, h1 = TRUE,
            add.labels = add.labels, add.class = add.class,
            drop.list.single.group = drop.list.single.group)

    #### wls.est - wls.obs - wls.v ####
    } else if(what == "wls.est") {
        lav_object_inspect_wls_est(object,
            add.labels = add.labels, add.class = add.class,
            drop.list.single.group = drop.list.single.group)
    } else if(what == "wls.obs") {
        lav_object_inspect_wls_obs(object,
            add.labels = add.labels, add.class = add.class,
            drop.list.single.group = drop.list.single.group)
    } else if(what == "wls.v") {
        lav_object_inspect_wls_v(object,
            add.labels = add.labels, add.class = add.class,
            drop.list.single.group = drop.list.single.group)



    #### data + missingness ####
    } else if(what == "data") {
        lav_object_inspect_data(object, 
            drop.list.single.group = drop.list.single.group)
    } else if(what == "case.idx") {
        lav_object_inspect_case_idx(object,
            drop.list.single.group = drop.list.single.group) 
    } else if(what == "ngroups") {
        object@Data@ngroups
    } else if(what == "group") {
        object@Data@group
    } else if(what == "cluster") {
        object@Data@cluster
    } else if(what == "ordered") {
        object@Data@ordered
    } else if(what == "group.label") {
        object@Data@group.label
    } else if(what == "nobs") {
        unlist( object@Data@nobs )
    } else if(what == "norig") {
        unlist( object@Data@norig )
    } else if(what == "ntotal") {
        sum(unlist( object@Data@nobs ))
    } else if(what == "coverage") {
        lav_object_inspect_missing_coverage(object,
            add.labels = add.labels, add.class = add.class,
            drop.list.single.group = drop.list.single.group)
    } else if(what %in% c("patterns", "pattern")) {
        lav_object_inspect_missing_patterns(object,
            add.labels = add.labels, add.class = add.class,
            drop.list.single.group = drop.list.single.group)
    } else if(what == "empty.idx") {
        lav_object_inspect_empty_idx(object,
            drop.list.single.group = drop.list.single.group)


    #### rsquare ####
    } else if(what == "rsquare" || what == "r-square" || what == "r2") {
         lav_object_inspect_rsquare(object, 
             add.labels = add.labels, add.class = add.class,
             drop.list.single.group = drop.list.single.group)


    #### model-implied sample statistics ####
    } else if(what == "implied" || what == "fitted") {
        lav_object_inspect_implied(object,
            add.labels = add.labels, add.class = add.class,
            drop.list.single.group = drop.list.single.group)
    } else if(what == "resid" || what == "res" || what == "residual" ||
              what == "residuals") {
        lav_object_inspect_residuals(object, h1 = TRUE,
            add.labels = add.labels, add.class = add.class,
            drop.list.single.group = drop.list.single.group)
    } else if(what == "cov.lv" || what == "veta") {
        lav_object_inspect_cov_lv(object,
            correlation.metric = FALSE,
            add.labels = add.labels, add.class = add.class,
            drop.list.single.group = drop.list.single.group)
    } else if(what == "cor.lv") {
        lav_object_inspect_cov_lv(object,
            correlation.metric = TRUE,
            add.labels = add.labels, add.class = add.class,
            drop.list.single.group = drop.list.single.group)
    } else if(what == "mean.lv" || what == "eeta") {
        lav_object_inspect_mean_lv(object,
            add.labels = add.labels, add.class = add.class,
            drop.list.single.group = drop.list.single.group)
    } else if(what == "cov.all") {
        lav_object_inspect_cov_all(object,
            correlation.metric = FALSE,
            add.labels = add.labels, add.class = add.class,
            drop.list.single.group = drop.list.single.group)
    } else if(what == "cor.all") {
        lav_object_inspect_cov_all(object,
            correlation.metric = TRUE,
            add.labels = add.labels, add.class = add.class,
            drop.list.single.group = drop.list.single.group)
    } else if(what == "cov.ov" || what == "sigma" || what == "sigma.hat") {
        lav_object_inspect_cov_ov(object,
            correlation.metric = FALSE,
            add.labels = add.labels, add.class = add.class,
            drop.list.single.group = drop.list.single.group)
    } else if(what == "cor.ov") {
        lav_object_inspect_cov_ov(object,
            correlation.metric = TRUE,
            add.labels = add.labels, add.class = add.class,
            drop.list.single.group = drop.list.single.group)
    } else if(what == "mean.ov" || what == "mu" || what == "mu.hat") {
        lav_object_inspect_mean_ov(object,
            add.labels = add.labels, add.class = add.class,
            drop.list.single.group = drop.list.single.group)
    } else if(what == "th" || what == "thresholds") {
        lav_object_inspect_th(object,
            add.labels = add.labels, add.class = add.class,
            drop.list.single.group = drop.list.single.group)
    } else if(what == "vy") {
        lav_object_inspect_vy(object,
            add.labels = add.labels, add.class = add.class,
            drop.list.single.group = drop.list.single.group)


    #### specific model matrices? ####
    } else if(what == "theta" || what == "theta.cov") {
        lav_object_inspect_theta(object,  correlation.metric = FALSE,
            add.labels = add.labels, add.class = add.class,
            drop.list.single.group = drop.list.single.group)
    } else if(what == "theta.cor") {
        lav_object_inspect_theta(object,  correlation.metric = TRUE,
            add.labels = add.labels, add.class = add.class,
            drop.list.single.group = drop.list.single.group)
 


    #### convergence, meanstructure, categorical ####
    } else if(what == "converged") {
        object@optim$converged
    } else if(what == "iterations" ||
              what == "iter" ||
              what == "niter") {
        object@optim$iterations
    } else if(what == "meanstructure") {
        object@Model@meanstructure
    } else if(what == "categorical") {
        object@Model@categorical
    } else if(what == "fixed.x") {
        object@Model@fixed.x
    } else if(what == "parameterization") {
        object@Model@parameterization
    


    #### NACOV samplestats ####
    } else if(what == "gamma") {
        lav_object_inspect_sampstat_gamma(object,
            add.labels = add.labels, add.class = add.class,
            drop.list.single.group = drop.list.single.group)


    #### gradient, Hessian, information, first.order, vcov ####
    } else if(what == "gradient") {
        lav_object_inspect_gradient(object,
            add.labels = add.labels, add.class = add.class)
    } else if(what == "hessian") {
        lav_object_inspect_hessian(object,
            add.labels = add.labels, add.class = add.class)

    } else if(what == "information") {
        lav_object_inspect_information(object, information = "default",
            augmented = FALSE, inverted = FALSE,
            add.labels = add.labels, add.class = add.class)
    } else if(what == "information.expected") {
        lav_object_inspect_information(object, information = "expected",
            augmented = FALSE, inverted = FALSE,
            add.labels = add.labels, add.class = add.class)
    } else if(what == "information.observed") {
        lav_object_inspect_information(object, information = "observed",
            augmented = FALSE, inverted = FALSE,
            add.labels = add.labels, add.class = add.class)
    } else if(what == "information.first.order" || what == "first.order") {
        lav_object_inspect_information(object, information = "first.order",
            augmented = FALSE, inverted = FALSE,
            add.labels = add.labels, add.class = add.class)

    } else if(what == "augmented.information") {
        lav_object_inspect_information(object, information = "default",
            augmented = TRUE, inverted = FALSE,
            add.labels = add.labels, add.class = add.class)
    } else if(what == "augmented.information.expected") {
        lav_object_inspect_information(object, information = "expected",
            augmented = TRUE, inverted = FALSE,
            add.labels = add.labels, add.class = add.class)
    } else if(what == "augmented.information.observed") {
        lav_object_inspect_information(object, information = "observed",
            augmented = TRUE, inverted = FALSE,
            add.labels = add.labels, add.class = add.class)
    } else if(what == "augmented.information.first.order" ||
              what == "augmented.first.order") {
        lav_object_inspect_information(object, information = "first.order",
            augmented = TRUE, inverted = FALSE,
            add.labels = add.labels, add.class = add.class)

    } else if(what == "inverted.information") {
        lav_object_inspect_information(object, information = "default",
            augmented = TRUE, inverted = TRUE,
            add.labels = add.labels, add.class = add.class)
    } else if(what == "inverted.information.expected") {
        lav_object_inspect_information(object, information = "expected",
            augmented = TRUE, inverted = TRUE,
            add.labels = add.labels, add.class = add.class)
    } else if(what == "inverted.information.observed") {
        lav_object_inspect_information(object, information = "observed",
            augmented = TRUE, inverted = TRUE,
            add.labels = add.labels, add.class = add.class)
    } else if(what == "inverted.information.first.order" || 
              what == "inverted.first.order") {
        lav_object_inspect_information(object, information = "first.order",
            augmented = TRUE, inverted = TRUE,
            add.labels = add.labels, add.class = add.class)

    } else if(what == "vcov") {
        lav_object_inspect_vcov(object,
            standardized = FALSE,
            add.labels = add.labels, add.class = add.class)
    } else if(what == "vcov.std.all" || what == "vcov.standardized" ||
              what == "vcov.std") {
        lav_object_inspect_vcov(object,
            standardized = TRUE, type = "std.all",
            add.labels = add.labels, add.class = add.class)
    } else if(what == "vcov.std.lv") {
        lav_object_inspect_vcov(object,
            standardized = TRUE, type = "std.lv",
            add.labels = add.labels, add.class = add.class)
    } else if(what == "vcov.std.nox") {
        lav_object_inspect_vcov(object,
            standardized = TRUE, type = "std.nox",
            add.labels = add.labels, add.class = add.class)
    } else if(what == "vcov.def") {
        lav_object_inspect_vcov_def(object,
            standardized = FALSE,
            add.labels = add.labels, add.class = add.class)
    } else if(what == "vcov.def.std.all" || what == "vcov.def.standardized" ||
              what == "vcov.def.std") {
        lav_object_inspect_vcov_def(object,
            standardized = TRUE, type = "std.all",
            add.labels = add.labels, add.class = add.class)
    } else if(what == "vcov.def.std.lv") {
        lav_object_inspect_vcov_def(object,
            standardized = TRUE, type = "std.lv",
            add.labels = add.labels, add.class = add.class)
    } else if(what == "vcov.def.std.nox") {
        lav_object_inspect_vcov_def(object,
            standardized = TRUE, type = "std.nox",
            add.labels = add.labels, add.class = add.class)
    } else if(what == "ugamma" || what == "ug" || what == "u.gamma") {
        lav_object_inspect_UGamma(object,
            add.labels = add.labels, add.class = add.class)

    ### jacobians ####
    } else if(what == "delta") {
        lav_object_inspect_delta(object,
            add.labels = add.labels, add.class = add.class,
            drop.list.single.group = drop.list.single.group)

    # post-checking
    } else if(what == "post.check" || what == "post") {
        lav_object_post_check(object)

    # options
    } else if(what == "options" || what == "lavoptions") {
        object@Options

    # call
    } else if(what == "call") {
        as.list( object@call )

    # timing
    } else if(what == "timing") {
        object@timing

    # optim
    } else if(what == "optim") {
        object@optim

    # test
    } else if(what == "test") {
        object@test

    # zero cell tables
    } else if(what == "zero.cell.tables") {
        lav_object_inspect_zero_cell_tables(object,
            add.labels = add.labels, add.class = add.class,
            drop.list.single.group = drop.list.single.group)

    #### not found ####
    } else {
        stop("unknown `what' argument in inspect function: `", what, "'")
    }

}


# helper functions (mostly to deal with older 'object' that may have
# been save somewhere)
lav_object_inspect_est <- function(object) {
    
    if(class(object) == "lavaan") {
        # from 0.5-19, they are in the partable
        if(!is.null(object@ParTable$est)) {
            OUT <- object@ParTable$est
        } else if("Fit" %in% slotNames(object)) {
            # in < 0.5-19, we should look in @Fit@est
            OUT <- object@Fit@est
        } else {
            PT <- parTable(object)
            OUT <- rep(as.numeric(NA), length(PT$lhs))
        }
    } else {
        # try coef()
        OUT <- coef(object, type = "user")
        if(is.matrix(OUT)) {
            # lavaanList?
            OUT <- rowMeans(OUT)
        }
    }

    OUT
}

lav_object_inspect_se <- function(object) {
    
    # from 0.5-19, they are in the partable
    if(!is.null(object@ParTable$se)) {
        OUT <- object@ParTable$se
    } else if("Fit" %in% slotNames(object)) {
        # in < 0.5-19, we should look in @Fit@se
        OUT <- object@Fit@se
    } else {
        PT <- parTable(object)
        OUT <- rep(as.numeric(NA), length(PT$lhs))
    }

    OUT
}

lav_object_inspect_start <- function(object) {

    # from 0.5-19, they are in the partable
    if(!is.null(object@ParTable$start)) {
        OUT <- object@ParTable$start
    } else {
        # in < 0.5-19, we should look in @Fit@start
        OUT <- object@Fit@start
    }

    OUT
}

lav_object_inspect_boot <- function(object) {

    # from 0.5-19. they are in a separate slot
    tmp <- try(slot(object,"boot"), silent = TRUE)
    if(inherits(tmp, "try-error")) {
        # older version of object?
        est <- lav_object_inspect_est(object)
        BOOT <- attr(est, "BOOT.COEF")
    } else {
        # 0.5-19 way
        BOOT <- object@boot$coef
    }

    BOOT
}


lav_object_inspect_modelmatrices <- function(object, what = "free",
    type = "free", add.labels = FALSE, add.class = FALSE,
    list.by.group = FALSE,
    drop.list.single.group = FALSE) {

    GLIST <- object@Model@GLIST

    if(what == "dx.free") {
        DX <- lav_model_gradient(lavmodel       = object@Model,
                                 GLIST          = NULL,
                                 lavsamplestats = object@SampleStats,
                                 lavdata        = object@Data,
                                 lavcache       = object@Cache,
                                 type           = "free",
                                 verbose        = FALSE,
                                 forcePD        = TRUE,
                                 group.weight   = TRUE,
                                 Delta          = NULL)
    } else if(what == "dx.all") {
        GLIST <- lav_model_gradient(lavmodel   = object@Model,
                                GLIST          = NULL,
                                lavsamplestats = object@SampleStats,
                                lavdata        = object@Data,
                                lavcache       = object@Cache,
                                type           = "allofthem",
                                verbose        = FALSE,
                                forcePD        = TRUE,
                                group.weight   = TRUE,
                                Delta          = NULL)
        names(GLIST) <- names(object@Model@GLIST)
    } else if(what == "std.all") {
        STD <- standardize.est.all(object)
    } else if(what == "std.lv") {
        STD <- standardize.est.lv(object)
    } else if(what == "std.nox") {
        STD <- standardize.est.all.nox(object)
    }

    for(mm in 1:length(GLIST)) {

        if(add.labels) {
            dimnames(GLIST[[mm]]) <- object@Model@dimNames[[mm]]
        }

        if(what == "free") {
            # fill in free parameter counts
            if(type == "free") {
                m.el.idx <- object@Model@m.free.idx[[mm]]
                x.el.idx <- object@Model@x.free.idx[[mm]]
            #} else if(type == "unco") {
            #    m.el.idx <- object@Model@m.unco.idx[[mm]]
            #    x.el.idx <- object@Model@x.unco.idx[[mm]]
            } else if(type == "partable") {
                m.el.idx <- object@Model@m.user.idx[[mm]]
                x.el.idx <- object@Model@x.user.idx[[mm]]
            } else {
                stop("lavaan ERROR: unknown type argument:", type, )
            }
            # erase everything
            GLIST[[mm]][,] <- 0.0
            GLIST[[mm]][m.el.idx] <- x.el.idx
        } else if(what == "se") {
            # fill in standard errors
            m.user.idx <- object@Model@m.user.idx[[mm]]
            x.user.idx <- object@Model@x.user.idx[[mm]]
            SE <- lav_object_inspect_se(object)
            # erase everything
            GLIST[[mm]][,] <- 0.0
            GLIST[[mm]][m.user.idx] <- SE[x.user.idx]
        } else if(what == "start") {
            # fill in starting values
            m.user.idx <- object@Model@m.user.idx[[mm]]
            x.user.idx <- object@Model@x.user.idx[[mm]]
            START <- lav_object_inspect_start(object)
            GLIST[[mm]][m.user.idx] <- START[x.user.idx]
        } else if(what == "est") {
            # fill in estimated parameter values
            m.user.idx <- object@Model@m.user.idx[[mm]]
            x.user.idx <- object@Model@x.user.idx[[mm]]
            EST <- lav_object_inspect_est(object)
            GLIST[[mm]][m.user.idx] <- EST[x.user.idx]
        } else if(what == "dx.free") {
            # fill in derivatives free parameters
            m.el.idx <- object@Model@m.free.idx[[mm]]
            x.el.idx <- object@Model@x.free.idx[[mm]]
            # erase everything
            GLIST[[mm]][,] <- 0.0
            GLIST[[mm]][m.el.idx] <- DX[x.el.idx]
        } else if(what %in% c("std.all", "std.lv", "std.nox")) {
            m.user.idx <- object@Model@m.user.idx[[mm]]
            x.user.idx <- object@Model@x.user.idx[[mm]]
            GLIST[[mm]][m.user.idx] <- STD[x.user.idx]
        } 

        # class
        if(add.class) {
            if(object@Model@isSymmetric[mm]) {
                class(GLIST[[mm]]) <- c("lavaan.matrix.symmetric", "matrix")
            } else {
                class(GLIST[[mm]]) <- c("lavaan.matrix", "matrix")
            }
        }
    }

    # try to reflect `equality constraints'
    con.flag <- FALSE
    if(what == "free" && object@Model@eq.constraints) {
        # extract constraints from parameter table
        PT <- parTable(object)
        CON <-  PT[PT$op %in% c("==","<",">") ,c("lhs","op","rhs")]
        rownames(CON) <- NULL

        # replace 'labels' by parameter numbers
        ID <- lav_partable_constraints_label_id(PT)
        LABEL <- names(ID)
        for(con in 1:nrow(CON)) {
            # lhs
            LHS.labels <- all.vars(as.formula(paste("~",CON[con,"lhs"]))) 

            if(length(LHS.labels) > 0L) {
                # par id
                LHS.freeid <- ID[match(LHS.labels, LABEL)]

                # substitute
                tmp <- CON[con,"lhs"]
                for(pat in 1:length(LHS.labels)) {
                    tmp <- sub(LHS.labels[pat], LHS.freeid[pat], tmp)
                }
                CON[con,"lhs"] <- tmp
            }

            # rhs
            RHS.labels <- all.vars(as.formula(paste("~",CON[con,"rhs"])))

            if(length(RHS.labels) > 0L) {
                # par id
                RHS.freeid <- ID[match(RHS.labels, LABEL)]
                # substitute
                tmp <- CON[con,"rhs"]
                for(pat in 1:length(RHS.labels)) {
                    tmp <- sub(RHS.labels[pat], RHS.freeid[pat], tmp)
                }
                CON[con,"rhs"] <- tmp
            }
        } # con

        # add this info at the top
        #GLIST <- c(constraints = list(CON), GLIST)
        #no, not a good idea, it does not work with list.by.group
  
        # add it as a 'header' attribute?
        attr(CON, "header") <- "Note: model contains equality constraints:"
        con.flag <- TRUE
    }

    # should we group them per group?
    if(list.by.group) {
        lavsamplestats <- object@SampleStats
        lavmodel       <- object@Model
        nmat           <- lavmodel@nmat

        OUT <- vector("list", length = lavsamplestats@ngroups)
        for(g in 1:lavsamplestats@ngroups) {
            # which mm belong to group g?
            mm.in.group <- 1:nmat[g] + cumsum(c(0,nmat))[g]
            mm.names <- names( GLIST[mm.in.group] )

            OUT[[g]] <- GLIST[mm.in.group]
        }

        if(lavsamplestats@ngroups == 1L && drop.list.single.group) {
            OUT <- OUT[[1]]
        } else {
            if(length(object@Data@group.label) > 0L) {
                names(OUT) <- unlist(object@Data@group.label)
            }
        }
    } else {
        OUT <- GLIST
    }

    # header
    if(con.flag) {
        attr(OUT, "header") <- CON
    }

    # lavaan.list
    if(add.class) {
        class(OUT) <- c("lavaan.list", "list")
    }

    OUT
}




# - fixme, should we export this function?
# - since 0.5-21, conditional.x = TRUE returns residual sample statistics
#    for ML, we have both joint and residual cov/var/...; but for 
#    categorical = TRUE, we only have residual cov/var...; so, we
#    only return residual in both cases, whenever residual
lav_object_inspect_sampstat <- function(object, h1 = FALSE,
    add.labels = FALSE, add.class = FALSE, drop.list.single.group = FALSE) {

    G <- object@Data@ngroups
    ov.names <- object@pta$vnames$ov
    ov.names.res <- object@pta$vnames$ov.nox
    ov.names.x   <- object@pta$vnames$ov.x
    lavsamplestats <- object@SampleStats

    OUT <- vector("list", length=G)
    for(g in 1:G) {

        if(!object@Model@conditional.x) {

            # covariance matrix
            if(h1 && !is.null(lavsamplestats@missing.h1[[g]])) {
                OUT[[g]]$cov  <- lavsamplestats@missing.h1[[g]]$sigma
            } else {
                OUT[[g]]$cov  <- lavsamplestats@cov[[g]]
            }
            if(add.labels && !is.null(OUT[[g]]$cov)) {
                rownames(OUT[[g]]$cov) <- colnames(OUT[[g]]$cov) <- 
                    ov.names[[g]]
            }
            if(add.class) {
                class(OUT[[g]]$cov) <- c("lavaan.matrix.symmetric", "matrix")
            }

            # mean vector
            if(h1 && !is.null(lavsamplestats@missing.h1[[g]])) {
                OUT[[g]]$mean <- lavsamplestats@missing.h1[[g]]$mu
            } else {
                OUT[[g]]$mean <- as.numeric(lavsamplestats@mean[[g]])
            }
            if(add.labels) {
                names(OUT[[g]]$mean) <- ov.names[[g]]
            }
            if(add.class) {
                class(OUT[[g]]$mean) <- c("lavaan.vector", "numeric")
            }

            # thresholds
            if(object@Model@categorical) {
                OUT[[g]]$th <- as.numeric(lavsamplestats@th[[g]])
                if(length(object@Model@num.idx[[g]]) > 0L) {
                    NUM.idx <- which(object@Model@th.idx[[g]] == 0)
                    OUT[[g]]$th <- OUT[[g]]$th[ -NUM.idx ]
                }
                if(add.labels) {
                    names(OUT[[g]]$th) <- object@pta$vnames$th[[g]]
                }
                if(add.class) {
                    class(OUT[[g]]$th) <- c("lavaan.vector", "numeric")
                }
            }
        } # !conditional.x

        else { # if conditional.x = TRUE

            # residual covariance matrix
            OUT[[g]]$res.cov  <- lavsamplestats@res.cov[[g]]
            if(add.labels) {
                rownames(OUT[[g]]$res.cov) <- colnames(OUT[[g]]$res.cov) <- 
                    ov.names.res[[g]]
            }
            if(add.class) {
                class(OUT[[g]]$res.cov) <- 
                    c("lavaan.matrix.symmetric", "matrix")
            }
   
            # intercepts
            if(object@Model@conditional.x) {
                OUT[[g]]$res.int <- as.numeric(lavsamplestats@res.int[[g]])
                if(add.labels) {
                    names(OUT[[g]]$res.int) <- ov.names.res[[g]]
                }
                if(add.class) {
                    class(OUT[[g]]$res.int) <- c("lavaan.vector", "numeric")
                }
            }

            # thresholds
            if(object@Model@categorical) {
                OUT[[g]]$res.th <- as.numeric(lavsamplestats@res.th[[g]])
                if(length(object@Model@num.idx[[g]]) > 0L) {
                    NUM.idx <- which(object@Model@th.idx[[g]] == 0)
                    OUT[[g]]$res.th <- OUT[[g]]$res.th[ -NUM.idx ]
                }
                if(add.labels) {
                    names(OUT[[g]]$res.th) <- object@pta$vnames$th[[g]]
                }
                if(add.class) {
                    class(OUT[[g]]$res.th) <- c("lavaan.vector", "numeric")
                }
            }

            # slopes
            if(object@Model@nexo > 0L) {
                OUT[[g]]$res.slopes  <- lavsamplestats@res.slopes[[g]]
                if(add.labels) {
                    rownames(OUT[[g]]$res.slopes) <- ov.names.res[[g]]
                    colnames(OUT[[g]]$res.slopes) <- ov.names.x[[g]]
                }
                if(add.class) {
                    class(OUT[[g]]$res.slopes) <- c("lavaan.matrix", "matrix")
                }
            }

            # cov.x
            if(object@Model@nexo > 0L) {
                OUT[[g]]$cov.x  <- lavsamplestats@cov.x[[g]]
                if(add.labels) {
                    rownames(OUT[[g]]$cov.x) <- ov.names.x[[g]]
                    colnames(OUT[[g]]$cov.x) <- ov.names.x[[g]]
                }
                if(add.class) {
                    class(OUT[[g]]$cov.x) <- 
                        c("lavaan.matrix.symmetric", "matrix")
                }
            }

        } # conditional.x

        # stochastic weights
        if(object@Model@group.w.free) {
            # to be consistent with the 'implied' values, 
            # transform so group.w is the 'log(group.freq)'
            OUT[[g]]$group.w <- 
                log(lavsamplestats@group.w[[g]] * lavsamplestats@ntotal)
            if(add.labels) {
                names(OUT[[g]]$group.w) <- "w"
            }
            if(add.class) {
                class(OUT[[g]]$group.w) <- c("lavaan.vector", "numeric")
            }
        }
    }

    if(G == 1L && drop.list.single.group) {
        OUT <- OUT[[1]]
    } else {
        if(length(object@Data@group.label) > 0L) {
            names(OUT) <- unlist(object@Data@group.label)
        }
    }

    OUT
}


lav_object_inspect_data <- function(object, add.labels = FALSE,
                                    drop.list.single.group = FALSE) {

    G <- object@Data@ngroups
    OUT <- object@Data@X

    if(add.labels) {
        for(g in 1:G) {
            colnames(OUT[[g]]) <- object@Data@ov.names[[g]]
        }
    }

    if(G == 1L && drop.list.single.group) {
        OUT <- OUT[[1]]
    } else {
        if(length(object@Data@group.label) > 0L) {
            names(OUT) <- unlist(object@Data@group.label)
        }
    }

    OUT
}

lav_object_inspect_case_idx <- function(object,
                                        drop.list.single.group = FALSE) {

    G <- object@Data@ngroups
    OUT <- object@Data@case.idx

    if(G == 1L && drop.list.single.group) {
        OUT <- OUT[[1]]
    } else {
        if(length(object@Data@group.label) > 0L) {
            names(OUT) <- unlist(object@Data@group.label)
        }
    }

    OUT
}


lav_object_inspect_rsquare <- function(object, est.std.all=NULL,
    add.labels = FALSE, add.class = FALSE, drop.list.single.group = FALSE) {

    G <- object@Data@ngroups
    OUT <- vector("list", length=G)

    if(is.null(est.std.all)) {
        est.std.all <- standardize.est.all(object)
    }

    partable <- object@ParTable
    partable$rsquare <- 1.0 - est.std.all
    # no values > 1.0
    partable$rsquare[partable$rsquare > 1.0] <- as.numeric(NA)

    for(g in 1:G) {
        ind.names <- partable$rhs[ which(partable$op == "=~" & 
                                         partable$group == g) ]
        eqs.y.names <- partable$lhs[ which(partable$op == "~"  & 
                                           partable$group == g) ]
        y.names <- unique( c(ind.names, eqs.y.names) )

        idx <- which(partable$op == "~~" & partable$lhs %in% y.names & 
                     partable$rhs == partable$lhs & partable$group == g)
        tmp <- partable$rsquare[idx]

        if(add.labels && length(tmp) > 0L) {
            names(tmp) <- partable$lhs[idx]
        }
        if(add.class) {
            class(tmp) <- c("lavaan.vector", "numeric")
        }

        OUT[[g]] <- tmp
    }

    if(G == 1L && drop.list.single.group) {
        OUT <- OUT[[1]]
    } else {
        if(length(object@Data@group.label) > 0L) {
            names(OUT) <- unlist(object@Data@group.label)
        }
    }

    OUT 
}

# model implied sample stats
lav_object_inspect_implied <- function(object,
    add.labels = FALSE, add.class = FALSE, drop.list.single.group = FALSE) {

    G <- object@Data@ngroups
    ov.names <- object@pta$vnames$ov
    ov.names.res <- object@pta$vnames$ov.nox
    ov.names.x   <- object@pta$vnames$ov.x
    lavimplied <- object@implied

    OUT <- vector("list", length=G)
    for(g in 1:G) {

        if(!object@Model@conditional.x) {

            # covariance matrix
            OUT[[g]]$cov  <- lavimplied$cov[[g]]
            if(add.labels && !is.null(OUT[[g]]$cov)) {
                rownames(OUT[[g]]$cov) <- colnames(OUT[[g]]$cov) <-
                    ov.names[[g]]
            }
            if(add.class) {
                class(OUT[[g]]$cov) <- c("lavaan.matrix.symmetric", "matrix")
            }

            # mean vector
            OUT[[g]]$mean <- as.numeric(lavimplied$mean[[g]])
            if(add.labels) {
                names(OUT[[g]]$mean) <- ov.names[[g]]
            }
            if(add.class) {
                class(OUT[[g]]$mean) <- c("lavaan.vector", "numeric")
            }

            # thresholds
            if(object@Model@categorical) {
                OUT[[g]]$th <- as.numeric(lavimplied$th[[g]])
                if(length(object@Model@num.idx[[g]]) > 0L) {
                    NUM.idx <- which(object@Model@th.idx[[g]] == 0)
                    OUT[[g]]$th <- OUT[[g]]$th[ -NUM.idx ]
                }
                if(add.labels) {
                    names(OUT[[g]]$th) <- object@pta$vnames$th[[g]]
                }
                if(add.class) {
                    class(OUT[[g]]$th) <- c("lavaan.vector", "numeric")
                }
            }
        } # !conditional.x

       else { # if conditional.x = TRUE

            # residual covariance matrix
            OUT[[g]]$res.cov  <- lavimplied$res.cov[[g]]
            if(add.labels) {
                rownames(OUT[[g]]$res.cov) <- colnames(OUT[[g]]$res.cov) <-
                    ov.names.res[[g]]
            }
            if(add.class) {
                class(OUT[[g]]$res.cov) <-
                    c("lavaan.matrix.symmetric", "matrix")
            }

            # intercepts
            if(object@Model@conditional.x) {
                OUT[[g]]$res.int <- as.numeric(lavimplied$res.int[[g]])
                if(add.labels) {
                    names(OUT[[g]]$res.int) <- ov.names.res[[g]]
                }
                if(add.class) {
                    class(OUT[[g]]$res.int) <- c("lavaan.vector", "numeric")
                }
            }

            # thresholds
            if(object@Model@categorical) {
                OUT[[g]]$res.th <- as.numeric(lavimplied$res.th[[g]])
                if(length(object@Model@num.idx[[g]]) > 0L) {
                    NUM.idx <- which(object@Model@th.idx[[g]] == 0)
                    OUT[[g]]$res.th <- OUT[[g]]$res.th[ -NUM.idx ]
                }
                if(add.labels) {
                    names(OUT[[g]]$res.th) <- object@pta$vnames$th[[g]]
                }
                if(add.class) {
                    class(OUT[[g]]$res.th) <- c("lavaan.vector", "numeric")
                }
            }

            # slopes
            if(object@Model@nexo > 0L) {
                OUT[[g]]$res.slopes  <- lavimplied$res.slopes[[g]]
                if(add.labels) {
                    rownames(OUT[[g]]$res.slopes) <- ov.names.res[[g]]
                    colnames(OUT[[g]]$res.slopes) <- ov.names.x[[g]]
                }
                if(add.class) {
                    class(OUT[[g]]$res.slopes) <- c("lavaan.matrix", "matrix")
                }
            }

            # cov.x
            if(object@Model@nexo > 0L) {
                OUT[[g]]$cov.x  <- object@SampleStats@cov.x[[g]]
                if(add.labels) {
                    rownames(OUT[[g]]$cov.x) <- ov.names.x[[g]]
                    colnames(OUT[[g]]$cov.x) <- ov.names.x[[g]]
                }
                if(add.class) {
                    class(OUT[[g]]$cov.x) <-
                        c("lavaan.matrix.symmetric", "matrix")
                }
            }

        } # conditional.x

        # stochastic weights
        if(object@Model@group.w.free) {
            OUT[[g]]$group.w <- lavimplied$group.w[[g]]
            if(add.labels) {
                names(OUT[[g]]$group.w) <- "w" # somewhat redundant
            }
            if(add.class) {
                class(OUT[[g]]$group.w) <- c("lavaan.vector", "numeric")
            }
        }
    }

    if(G == 1L && drop.list.single.group) {
        OUT <- OUT[[1]]
    } else {
        if(length(object@Data@group.label) > 0L) {
            names(OUT) <- unlist(object@Data@group.label)
        }
    }

    OUT
}


# residuals: _inspect_sampstat - _inspect_implied
lav_object_inspect_residuals <- function(object, h1 = TRUE,
    add.labels = FALSE, add.class = FALSE, drop.list.single.group = FALSE) {

   # unstandardized residuals
    obsList <- lav_object_inspect_sampstat(object, h1 = h1,
                                           add.labels = add.labels,
                                           add.class  = FALSE,
                                           drop.list.single.group = FALSE)
    estList <- lav_object_inspect_implied(object,
                                          add.labels = add.labels,
                                          add.class  = FALSE,
                                          drop.list.single.group = FALSE)
    # multiple groups
    ngroups <- length(obsList)

    resList <- vector("list", length = ngroups)
    for(g in 1:ngroups) {
        if(object@Model@conditional.x) {
            if(!is.null(estList[[g]]$res.cov)) {
                resList[[g]]$res.cov <- ( obsList[[g]]$res.cov -
                                          estList[[g]]$res.cov )
                if(add.class) {
                    class(resList[[g]]$res.cov) <- 
                        c("lavaan.matrix.symmetric", "matrix")
                }
            }
            if(!is.null(estList[[g]]$res.int)) {
                 resList[[g]]$res.int <- ( obsList[[g]]$res.int -
                                           estList[[g]]$res.int )
                if(add.class) {
                    class(resList[[g]]$res.int) <-
                        c("lavaan.vector", "numeric")
                }
            }
            if(!is.null(estList[[g]]$res.th)) {
                resList[[g]]$res.th  <- ( obsList[[g]]$res.th  -
                                          estList[[g]]$res.th )
                if(add.class) {
                    class(resList[[g]]$res.th) <-
                        c("lavaan.vector", "numeric")
                }
            }
            if(!is.null(estList[[g]]$res.slopes)) {
                resList[[g]]$res.slopes <- ( obsList[[g]]$res.slopes -
                                             estList[[g]]$res.slopes )
                if(add.class) {
                    class(resList[[g]]$res.slopes) <-
                        c("lavaan.matrix", "matrix")
                }
            }
            if(!is.null(estList[[g]]$cov.x)) {
                resList[[g]]$cov.x  <- ( obsList[[g]]$cov.x  -
                                         estList[[g]]$cov.x )
                if(add.class) {
                    class(resList[[g]]$cov.x) <- 
                        c("lavaan.matrix.symmetric", "matrix")
                }
            }

        # unconditional
        } else {
            if(!is.null(estList[[g]]$cov)) {
                resList[[g]]$cov <- ( obsList[[g]]$cov -
                                      estList[[g]]$cov )
                if(add.class) {
                    class(resList[[g]]$cov) <- 
                        c("lavaan.matrix.symmetric", "matrix")
                }
            }
            if(!is.null(estList[[g]]$mean)) {
                 resList[[g]]$mean <- ( obsList[[g]]$mean -
                                        estList[[g]]$mean )
                 if(add.class) {
                    class(resList[[g]]$mean) <-
                        c("lavaan.vector", "numeric")
                 }
            }
            if(!is.null(estList[[g]]$th)) {
                resList[[g]]$th  <- ( obsList[[g]]$th  -
                                      estList[[g]]$th )
                if(add.class) {
                    class(resList[[g]]$th) <-
                        c("lavaan.vector", "numeric")
                }
            }
        }

        # free group.w
        if(!is.null(estList[[g]]$group.w)) {
            resList[[g]]$group.w <- ( obsList[[g]]$group.w  - 
                                      estList[[g]]$group.w )
            if(add.class) {
                class(resList[[g]]$group.w) <-
                    c("lavaan.vector", "numeric")
            }
        }
    }

    OUT <- resList
    if(ngroups == 1L && drop.list.single.group) {
        OUT <- OUT[[1]]
    } else {
        if(length(object@Data@group.label) > 0L) {
            names(OUT) <- unlist(object@Data@group.label)
        }
    }

    OUT
}


lav_object_inspect_cov_lv <- function(object, correlation.metric = FALSE,
    add.labels = FALSE, add.class = FALSE, drop.list.single.group = FALSE) {

    G <- object@Data@ngroups

    # compute lv covar
    OUT <- computeVETA(lavmodel       = object@Model, 
                       lavsamplestats = object@SampleStats,
                       remove.dummy.lv = TRUE)

    # cor + labels + class
    for(g in 1:G) {

        if(correlation.metric && nrow(OUT[[g]]) > 1L) {
            # note: cov2cor fails if matrix is empty!
            OUT[[g]] <- cov2cor(OUT[[g]])
        }

        if(add.labels) {
            colnames(OUT[[g]]) <- rownames(OUT[[g]]) <- 
                object@pta$vnames$lv[[g]]
        }

        if(add.class) {
            class(OUT[[g]]) <- c("lavaan.matrix.symmetric", "matrix")
        }
    }

    if(G == 1L && drop.list.single.group) {
        OUT <- OUT[[1]]
    } else {
        if(length(object@Data@group.label) > 0L) {
            names(OUT) <- unlist(object@Data@group.label)
        }
    }

    OUT
}

lav_object_inspect_mean_lv <- function(object,
    add.labels = FALSE, add.class = FALSE, drop.list.single.group = FALSE) {

    G <- object@Data@ngroups

    # compute lv means
    OUT <- computeEETA(lavmodel       = object@Model, 
                       lavsamplestats = object@SampleStats,
                       remove.dummy.lv = TRUE)
    OUT <- lapply(OUT, as.numeric)

    # labels + class
    for(g in 1:G) {
        if(add.labels && length(OUT[[g]]) > 0L) {
            names(OUT[[g]]) <- object@pta$vnames$lv.regular[[g]]
        }
        if(add.class) {
            class(OUT[[g]]) <- c("lavaan.vector", "numeric")
        }
    }

    if(G == 1L && drop.list.single.group) {
        OUT <- OUT[[1]]
    } else {
        if(length(object@Data@group.label) > 0L) {
            names(OUT) <- unlist(object@Data@group.label)
        }
    }

    OUT
}

lav_object_inspect_cov_all <- function(object, correlation.metric = FALSE,
    add.labels = FALSE, add.class = FALSE, drop.list.single.group = FALSE) {

    G <- object@Data@ngroups

    # compute extended model implied covariance matrix (both ov and lv)
    OUT <- computeCOV(lavmodel = object@Model, 
                      lavsamplestats = object@SampleStats,
                      remove.dummy.lv = TRUE)

    # cor + labels + class
    for(g in 1:G) {

        if(correlation.metric && nrow(OUT[[g]]) > 1L) {
            # note: cov2cor fails if matrix is empty!
            OUT[[g]] <- cov2cor(OUT[[g]])
        }

        if(add.labels) {
            NAMES <- c(object@pta$vnames$ov.model[[g]],
                       object@pta$vnames$lv.regular[[g]])
            colnames(OUT[[g]]) <- rownames(OUT[[g]]) <- NAMES
        }
        if(add.class) {
            class(OUT[[g]]) <- c("lavaan.matrix.symmetric", "matrix")
        }
    }

    if(G == 1L && drop.list.single.group) {
        OUT <- OUT[[1]]
    } else {
        if(length(object@Data@group.label) > 0L) {
            names(OUT) <- unlist(object@Data@group.label)
        }
    }

    OUT
}


lav_object_inspect_cov_ov <- function(object, correlation.metric = FALSE,
    add.labels = FALSE, add.class = FALSE, drop.list.single.group = FALSE) {

    G <- object@Data@ngroups

    # get model-implied covariance matrix observed

    if(object@Model@conditional.x) {
        OUT <- object@implied$res.cov
    } else {
        OUT <- object@implied$cov
    }

    # cor + labels + class
    for(g in 1:G) {
 
        if(correlation.metric && nrow(OUT[[g]]) > 1L) {
            # note: cov2cor fails if matrix is empty!
            OUT[[g]] <- cov2cor(OUT[[g]])
        }

        if(add.labels) {
            colnames(OUT[[g]]) <- rownames(OUT[[g]]) <- 
                object@pta$vnames$ov.model[[g]]
        }
        if(add.class) {
            class(OUT[[g]]) <- c("lavaan.matrix.symmetric", "matrix")
        }
    }

    if(G == 1L && drop.list.single.group) {
        OUT <- OUT[[1]]
    } else {
        if(length(object@Data@group.label) > 0L) {
            names(OUT) <- unlist(object@Data@group.label)
        }
    }

    OUT
}

lav_object_inspect_mean_ov <- function(object,
    add.labels = FALSE, add.class = FALSE, drop.list.single.group = FALSE) {

    G <- object@Data@ngroups

    # compute lv means
    if(object@Model@conditional.x) {
        OUT <- object@implied$res.int
    } else {
        OUT <- object@implied$mean
    }
   
    # make numeric
    OUT <- lapply(OUT, as.numeric)

    # labels + class
    for(g in 1:G) {
        if(add.labels && length(OUT[[g]]) > 0L) {
            names(OUT[[g]]) <- object@pta$vnames$ov.model[[g]]
        }
        if(add.class) {
            class(OUT[[g]]) <- c("lavaan.vector", "numeric")
        }
    }

    if(G == 1L && drop.list.single.group) {
        OUT <- OUT[[1]]
    } else {
        if(length(object@Data@group.label) > 0L) {
            names(OUT) <- unlist(object@Data@group.label)
        }
    }

    OUT
}

lav_object_inspect_th <- function(object,
    add.labels = FALSE, add.class = FALSE, drop.list.single.group = FALSE) {

    G <- object@Data@ngroups

    # thresholds
    if(object@Model@conditional.x) {
        OUT <- object@implied$res.th
    } else {
        OUT <- object@implied$th
    }

    # make numeric
    OUT <- lapply(OUT, as.numeric)

    # labels + class
    for(g in 1:G) {
        if(length(object@Model@num.idx[[g]]) > 0L) {
            NUM.idx <- which(object@Model@th.idx[[g]] == 0)
            OUT[[g]] <- OUT[[g]][ -NUM.idx ]
        }
        if(add.labels && length(OUT[[g]]) > 0L) {
            names(OUT[[g]]) <- object@pta$vnames$th[[g]]
        }
        if(add.class) {
            class(OUT[[g]]) <- c("lavaan.vector", "numeric")
        }
    }

    if(G == 1L && drop.list.single.group) {
        OUT <- OUT[[1]]
    } else {
        if(length(object@Data@group.label) > 0L) {
            names(OUT) <- unlist(object@Data@group.label)
        }
    }

    OUT
}

lav_object_inspect_vy <- function(object,
    add.labels = FALSE, add.class = FALSE, drop.list.single.group = FALSE) {

    G <- object@Data@ngroups

    # 'unconditional' model-implied variances
    #  - same as diag(Sigma.hat) if all Y are continuous)
    #  - 1.0 (or delta^2) if categorical
    #  - if also Gamma, cov.x is used (only if categorical)

    OUT <- computeVY(lavmodel = object@Model, GLIST = NULL, 
                     lavsamplestats = object@SampleStats,
                     diagonal.only = TRUE)
                     

    # labels + class
    for(g in 1:G) {
        if(add.labels && length(OUT[[g]]) > 0L) {
            if(object@Model@categorical) {
                 names(OUT[[g]]) <- object@pta$vnames$ov.nox[[g]]
            } else {
                 names(OUT[[g]]) <- object@pta$vnames$ov[[g]]
            }
        }
        if(add.class) {
            class(OUT[[g]]) <- c("lavaan.vector", "numeric")
        }
    }

    if(G == 1L && drop.list.single.group) {
        OUT <- OUT[[1]]
    } else {
        if(length(object@Data@group.label) > 0L) {
            names(OUT) <- unlist(object@Data@group.label)
        }
    }

    OUT
}


lav_object_inspect_theta <- function(object, correlation.metric = FALSE,
    add.labels = FALSE, add.class = FALSE, drop.list.single.group = FALSE) {

    G <- object@Data@ngroups

    # get residual covariances
    OUT <- computeTHETA(lavmodel = object@Model)

    # labels + class
    for(g in 1:G) {
        
        if(correlation.metric && nrow(OUT[[g]]) > 0L) {
            OUT[[g]] <- cov2cor(OUT[[g]])
        }

        if(add.labels && length(OUT[[g]]) > 0L) {
            colnames(OUT[[g]]) <- rownames(OUT[[g]]) <- 
                object@pta$vnames$ov.model[[g]]
        }

        if(add.class) {
            class(OUT[[g]]) <- c("lavaan.matrix.symmetric", "matrix")
        }
    }

    if(G == 1L && drop.list.single.group) {
        OUT <- OUT[[1]]
    } else {
        if(length(object@Data@group.label) > 0L) {
            names(OUT) <- unlist(object@Data@group.label)
        }
    }

    OUT
}


lav_object_inspect_missing_coverage <- function(object,
    add.labels = FALSE, add.class = FALSE, drop.list.single.group = FALSE) {

    G <- object@Data@ngroups

    # get missing covarage
    OUT <- vector("list", G)
   
    for(g in 1:G) {
        if(!is.null(object@Data@Mp[[g]])) {
            OUT[[g]] <- object@Data@Mp[[g]]$coverage
        } else {
            nvar <- length(object@Data@ov.names[[g]])
            OUT[[g]] <- matrix(1.0, nvar, nvar)
        }

        if(add.labels && length(OUT[[g]]) > 0L) {
            colnames(OUT[[g]]) <- rownames(OUT[[g]]) <-
                object@pta$vnames$ov.model[[g]]
        }

        if(add.class) {
            class(OUT[[g]]) <- c("lavaan.matrix.symmetric", "matrix")
        }
    }

    if(G == 1L && drop.list.single.group) {
        OUT <- OUT[[1]]
    } else {
        if(length(object@Data@group.label) > 0L) {
            names(OUT) <- unlist(object@Data@group.label)
        }
    }

    OUT
}

lav_object_inspect_missing_patterns <- function(object,
    add.labels = FALSE, add.class = FALSE, drop.list.single.group = FALSE) {

    G <- object@Data@ngroups

    # get missing covarage
    OUT <- vector("list", G)
   
    for(g in 1:G) {
        if(!is.null(object@Data@Mp[[g]])) {
            OUT[[g]] <- object@Data@Mp[[g]]$pat
        } else {
            nvar <- length(object@Data@ov.names[[g]])
            OUT[[g]] <- matrix(TRUE, 1L, nvar)
            rownames(OUT[[g]]) <- object@Data@nobs[[g]]
        }

        if(add.labels && length(OUT[[g]]) > 0L) {
            colnames(OUT[[g]]) <- object@pta$vnames$ov.model[[g]]
        }

        if(add.class) {
            class(OUT[[g]]) <- c("lavaan.matrix", "matrix")
        }
    }

    if(G == 1L && drop.list.single.group) {
        OUT <- OUT[[1]]
    } else {
        if(length(object@Data@group.label) > 0L) {
            names(OUT) <- unlist(object@Data@group.label)
        }
    }

    OUT
}

lav_object_inspect_empty_idx <- function(object,
                                         drop.list.single.group = FALSE) {
   
    G <- object@Data@ngroups

    # get empty idx
    OUT <- vector("list", G)

    for(g in 1:G) {
        if(!is.null(object@Data@Mp[[g]])) {
            OUT[[g]] <- object@Data@Mp[[g]]$empty.idx
        } else {
            OUT[[g]] <- integer(0L)
        }
    }
    
    if(G == 1L && drop.list.single.group) {
        OUT <- OUT[[1]]
    } else {
        if(length(object@Data@group.label) > 0L) {
            names(OUT) <- unlist(object@Data@group.label)
        }
    }

    OUT
}


lav_object_inspect_wls_est <- function(object,
    add.labels = FALSE, add.class = FALSE, drop.list.single.group = FALSE) {

    G <- object@Data@ngroups
    OUT <- lav_model_wls_est(object@Model) #,
                             #cov.x = object@SampleStats@cov.x)

    for(g in 1:G) {
        if(add.labels && length(OUT[[g]]) > 0L) {
            #FIXME!!!!
            #names(OUT[[g]]) <- ??
        }

        if(add.class) {
            class(OUT[[g]]) <- c("lavaan.vector", "numeric")
        }
    }

    if(G == 1L && drop.list.single.group) {
        OUT <- OUT[[1]]
    } else {
        if(length(object@Data@group.label) > 0L) {
            names(OUT) <- unlist(object@Data@group.label)
        }
    }

    OUT
}

lav_object_inspect_wls_obs <- function(object,
    add.labels = FALSE, add.class = FALSE, drop.list.single.group = FALSE) {

    G <- object@Data@ngroups
    OUT <- object@SampleStats@WLS.obs

    for(g in 1:G) {
        if(add.labels && length(OUT[[g]]) > 0L) {
            #FIXME!!!!
            #names(OUT[[g]]) <- ??
        }

        if(add.class) {
            class(OUT[[g]]) <- c("lavaan.vector", "numeric")
        }
    }

    if(G == 1L && drop.list.single.group) {
        OUT <- OUT[[1]]
    } else {
        if(length(object@Data@group.label) > 0L) {
            names(OUT) <- unlist(object@Data@group.label)
        }
    }

    OUT
}

lav_object_inspect_wls_v <- function(object,
    add.labels = FALSE, add.class = FALSE, drop.list.single.group = FALSE) {

    # shortcuts
    G <- object@Data@ngroups

    OUT <- lav_model_wls_v(lavmodel       = object@Model,
                           lavsamplestats = object@SampleStats,
                           structured     = TRUE,
                           lavdata        = object@Data)

    # if estimator == "DWLS" or "ULS", we only stored the diagonal
    # hence, we create a full matrix here
    if(object@Options$estimator %in% c("DWLS", "ULS")) {
        OUT <- lapply(OUT, 
            function(x) { nr = NROW(x); diag(x, nrow=nr, ncol=nr) })
    }

    # label + class
    for(g in 1:G) {
        if(add.labels && nrow(OUT[[g]]) > 0L) {
            #FIXME!!!!
            #names(OUT[[g]]) <- ??
        }

        if(add.class) {
            class(OUT[[g]]) <- c("lavaan.matrix", "matrix")
        }
    }

    if(G == 1L && drop.list.single.group) {
        OUT <- OUT[[1]]
    } else {
        if(length(object@Data@group.label) > 0L) {
            names(OUT) <- unlist(object@Data@group.label)
        }
    }

    OUT
}


lav_object_inspect_sampstat_gamma <- function(object,
    add.labels = FALSE, add.class = FALSE, drop.list.single.group = FALSE) {

    # shortcuts
    G <- object@Data@ngroups

    if(!is.null(object@SampleStats@NACOV[[1]])) {
        OUT <- object@SampleStats@NACOV
    } else {
        OUT <- lavGamma(object)
    }

    if(G == 1L && drop.list.single.group) {
        OUT <- OUT[[1]]
    } else {
        if(length(object@Data@group.label) > 0L) {
            names(OUT) <- unlist(object@Data@group.label)
        }
    }

    OUT
}


lav_object_inspect_gradient <- function(object,
    add.labels = FALSE, add.class = FALSE) {

    if(object@SampleStats@missing.flag ||
       object@Options$estimator == "PML") {
        group.weight <- FALSE
    } else {
        group.weight <- TRUE
    }

    OUT <- lav_model_gradient(lavmodel       = object@Model,
                              GLIST          = NULL,
                              lavsamplestats = object@SampleStats,
                              lavdata        = object@Data,
                              lavcache       = object@Cache,
                              type           = "free",
                              verbose        = FALSE,
                              group.weight   = group.weight)

    # labels
    if(add.labels) {
        names(OUT) <- lav_partable_labels(object@ParTable, type="free")
    }

    # class
    if(add.class) {
        class(OUT) <- c("lavaan.vector", "numeric")
    }

    OUT
}

lav_object_inspect_hessian <- function(object,
    add.labels = FALSE, add.class = FALSE) {

    OUT <- lav_model_hessian(lavmodel       = object@Model, 
                             lavsamplestats = object@SampleStats,
                             lavdata        = object@Data,
                             lavcache       = object@Cache,
                             group.weight   = TRUE)

    # labels
    if(add.labels) {
        colnames(OUT) <- rownames(OUT) <-
            lav_partable_labels(object@ParTable, type="free")
    }

    # class
    if(add.class) {
        class(OUT) <- c("lavaan.matrix.symmetric", "matrix")
    }

    OUT
}

lav_object_inspect_information <- function(object, 
    information = "default", augmented = FALSE, inverted = FALSE,
    add.labels = FALSE, add.class = FALSE) {

    if(information == "default") {
        information <- object@Options$information
    } 

    if(information == "expected" || information == "observed") {
        OUT <- lav_model_information(lavmodel =  object@Model,
                  lavsamplestats = object@SampleStats,
                  lavdata        = object@Data,
                  lavcache       = object@Cache,
                  information    = information,
                  augmented      = augmented,
                  inverted       = inverted)
    } else if(information == "first.order") {
        B0 <- lav_model_information_firstorder(lavmodel =  object@Model,
              lavsamplestats = object@SampleStats,
              lavdata        = object@Data,
              lavcache       = object@Cache,
              check.pd       = FALSE,
              augmented      = augmented,
              inverted       = inverted)
        attr(B0, "B0.group") <- NULL
        OUT <- B0
    }

    # labels
    if(add.labels) {
        NAMES <- lav_partable_labels(object@ParTable, type="free")
        if(augmented) {
            nExtra <- nrow(OUT) - length(NAMES)
            if(nExtra > 0L) {
                NAMES <- c(NAMES, paste("aug", 1:nExtra, sep=""))
            }
        }
        colnames(OUT) <- rownames(OUT) <- NAMES
    }

    # class
    if(add.class) {
        class(OUT) <- c("lavaan.matrix.symmetric", "matrix")
    }

    OUT
}

# only to provide a direct function to the old 'getVariability()' function
lav_object_inspect_firstorder <- function(object, 
    add.labels = FALSE, add.class = FALSE) {

     B0 <- lav_model_information_firstorder(lavmodel =  object@Model,
              lavsamplestats = object@SampleStats,
              lavdata        = object@Data,
              lavcache       = object@Cache,
              check.pd       = FALSE,
              augmented      = FALSE,
              inverted       = FALSE)
    attr(B0, "B0.group") <- NULL
    OUT <- B0

    # labels
    if(add.labels) {
        colnames(OUT) <- rownames(OUT) <-
            lav_partable_labels(object@ParTable, type="free")
    }

    # class
    if(add.class) {
        class(OUT) <- c("lavaan.matrix.symmetric", "matrix")
    }

    OUT
}

lav_object_inspect_vcov <- function(object, standardized = FALSE,
    type = "std.all", free.only = TRUE,
    add.labels = FALSE, add.class = FALSE, remove.duplicated = FALSE) {

    npar <- max(object@ParTable$free)
    if(object@optim$npar == 0) {
        OUT <- matrix(0,0,0)
    } else {
        # check if we already have it
        tmp <- try(slot(object, "vcov"), silent = TRUE)
        if(!inherits(tmp, "try-error") && !is.null(object@vcov$vcov)) {
            OUT <- object@vcov$vcov
        } else {
        # compute it again
            OUT <- lav_model_vcov(lavmodel       = object@Model,
                                  lavsamplestats = object@SampleStats,
                                  lavoptions     = object@Options,
                                  lavdata        = object@Data,
                                  lavcache       = object@Cache
                                 )
        }
    }
   
    # strip attributes
    attr(OUT, "E.inv") <- NULL
    attr(OUT, "B0") <- NULL
    attr(OUT, "B0.group") <- NULL
    attr(OUT, "Delta") <- NULL
    attr(OUT, "WLS.V") <- NULL
    attr(OUT, "BOOT.COEF") <- NULL
    attr(OUT, "BOOT.TEST") <- NULL

    # standardized?
    if(standardized) {
        if(type == "std.lv") {
            JAC <- try(lav_func_jacobian_complex(func = standardize.est.lv.x,
                x = object@optim$x, lavobject = object), silent = TRUE)
            if(inherits(JAC, "try-error")) { # eg. pnorm()
                JAC <- lav_func_jacobian_simple(func = standardize.est.lv.x,
                    x = object@optim$x, lavobject = object)
            }
        } else if(type == "std.all") {
            JAC <- try(lav_func_jacobian_complex(func = standardize.est.all.x,
                x = object@optim$x, object = object), silent = TRUE)
            if(inherits(JAC, "try-error")) { # eg. pnorm()
                JAC <- lav_func_jacobian_simple(func = standardize.est.all.x,
                    x = object@optim$x, lavobject = object)
            }
        } else if(type == "std.nox") {
            JAC <- 
                try(lav_func_jacobian_complex(func = standardize.est.all.nox.x,
                    x = object@optim$x, lavobject = object), silent = TRUE)
            if(inherits(JAC, "try-error")) { # eg. pnorm()
                JAC <- 
                    lav_func_jacobian_simple(func = standardize.est.all.nox.x,
                        x = object@optim$x, lavobject = object)
            }
        }

        # JAC contains *all* parameters in the parameter table
        if(free.only) { 
            free.idx <- which(object@ParTable$free > 0L)
            JAC <- JAC[free.idx,, drop = FALSE]
        }
        OUT <- JAC %*% OUT %*% t(JAC)
    }

    # labels
    if(add.labels) {
        colnames(OUT) <- rownames(OUT) <-
            lav_partable_labels(object@ParTable, type="free")
    }

    # alias?
    if(remove.duplicated && object@Model@eq.constraints) {
        simple.flag <- lav_constraints_check_simple(object@Model)
        if(simple.flag) {
            LAB <- lav_partable_labels(object@ParTable, type="free")
            dup.flag <- duplicated(LAB)
            OUT <- OUT[!dup.flag, !dup.flag, drop = FALSE]
        } else {
            warning("lavaan WARNING: alias is TRUE, but equality constraints do not appear to be simple; returning full vcov")
        }
    }

    # class
    if(add.class) {
        class(OUT) <- c("lavaan.matrix.symmetric", "matrix")
    }

    OUT
}

lav_object_inspect_vcov_def <- function(object, standardized = FALSE,
    type = "std.all", add.labels = FALSE, add.class = FALSE) {

    lavmodel    <- object@Model
    lavpartable <- object@ParTable
    def.idx <- which(lavpartable$op == ":=")

    if(length(def.idx) == 0L) {
        return( matrix(0,0,0) )
    }

    if(standardized) {
        # compute VCOV for "free" parameters only
        VCOV <- lav_object_inspect_vcov(object, 
                                        standardized = TRUE,
                                        type = type, free.only = FALSE,
                                        add.labels = FALSE, add.class = FALSE)
        OUT <- VCOV[def.idx, def.idx, drop = FALSE]
    } else {

        # get free parameters
        x <- lav_model_get_parameters(lavmodel, type = "free")

        # bootstrap or not?
        if(!is.null(object@boot$coef)) {
            BOOT <- object@boot$coef
            BOOT.def <- apply(BOOT, 1L, lavmodel@def.function)
            if(length(def.idx) == 1L) {
                BOOT.def <- as.matrix(BOOT.def)
            } else {
                BOOT.def <- t(BOOT.def)
            }
            OUT <- cov(BOOT.def)
        } else {
            # VCOV
            VCOV <- lav_object_inspect_vcov(object,
                                            standardized = FALSE,
                                            type = type, free.only = TRUE,
                                            add.labels = FALSE, 
                                            add.class = FALSE)
    
            # regular delta method
            JAC <- try(lav_func_jacobian_complex(func = lavmodel@def.function,
                       x = x), silent=TRUE)
            if(inherits(JAC, "try-error")) { # eg. pnorm()
                JAC <- lav_func_jacobian_simple(func = lavmodel@def.function,
                                                x = x)
            }
            OUT <- JAC %*% VCOV %*% t(JAC)
        }
    }

    # labels
    if(add.labels) {
        LHS.names <- lavpartable$lhs[def.idx]
        colnames(OUT) <- rownames(OUT) <- LHS.names
    }

    # class
    if(add.class) {
        class(OUT) <- c("lavaan.matrix.symmetric", "matrix")
    }

    OUT
}

lav_object_inspect_UGamma <- function(object,
    add.labels = FALSE, add.class = FALSE) {

    out <- lav_test_satorra_bentler(lavobject     = object,
                                    return.ugamma = TRUE)
    OUT <- out$UGamma

    # labels
    if(add.labels) {
       # colnames(OUT) <- rownames(OUT) <-
    }

    # class
    if(add.class) {
        class(OUT) <- c("lavaan.matrix.symmetric", "matrix")
    }

    OUT
}

# Delta (jacobian: d samplestats / d free_parameters)
lav_object_inspect_delta <- function(object,
    add.labels = FALSE, add.class = FALSE, drop.list.single.group = FALSE) {

    OUT <- computeDelta(object@Model)

    # labels
    lavmodel <- object@Model
    categorical    <- lavmodel@categorical
    conditional.x  <- lavmodel@conditional.x
    group.w.free   <- lavmodel@group.w.free
    nvar           <- lavmodel@nvar
    num.idx        <- lavmodel@num.idx
    th.idx         <- lavmodel@th.idx
    nexo           <- lavmodel@nexo
    nblocks        <- lavmodel@nblocks

    if(add.labels) {
        PNAMES <- lav_partable_labels(object@ParTable, type="free")

        for(g in 1:nblocks) {
            colnames(OUT[[g]]) <- PNAMES

            if(conditional.x) {
                ov.names <- object@pta$vnames$ov.nox[[g]]
            } else {
                ov.names <- object@pta$vnames$ov[[g]]
            }
            ov.names.x <- object@pta$vnames$ov.x[[g]]
            nvar <- length(ov.names)


            names.cov <- names.cor <- names.var <- character(0L)
            names.mu <- names.pi <- names.th <- character(0L)
            names.gw <- character(0L)
 
            # Sigma
            # - if continuous: vech(Sigma)
            # - if categorical: first numeric variances, then
            tmp <- apply(expand.grid(ov.names, ov.names), 1L, 
                         paste, collapse = "~~")
            if(categorical) {
                names.cor <- tmp[lav_matrix_vech_idx(nvar, diagonal = FALSE)]
                names.var <- tmp[lav_matrix_diag_idx(nvar)[num.idx[[g]]]]
            } else {
                names.cov <- tmp[lav_matrix_vech_idx(nvar, diagonal = TRUE)]
            }

            # Mu
            if(!categorical && lavmodel@meanstructure) {
                names.mu <- paste(ov.names, "~1", sep = "")
            }

            # Pi
            if(conditional.x && lavmodel@nexo[g] > 0L) {
               names.pi <- apply(expand.grid(ov.names, ov.names.x), 1L, 
                                 paste, collapse = "~")
            }

            # th
            if(categorical) {
                names.th <- object@pta$vnames$th[[g]]
                # interweave numeric intercepts, if any
                if(length(num.idx[[g]]) > 0L) {
                    tmp <- character( length(th.idx[[g]]) )
                    tmp[ th.idx[[g]] > 0 ] <- names.th
                    tmp[ th.idx[[g]] == 0 ] <- paste(ov.names[ num.idx[[g]] ],
                                                     "~1", sep = "")
                    names.th <- tmp
                }
            }

            # gw
            if(group.w.free) {
                names.gw <- "w"
            }

            rownames(OUT[[g]]) <- c(names.gw,
                                    names.th, names.mu, 
                                    names.pi, 
                                    names.cov, names.var, names.cor)

            # class
            if(add.class) {
                class(OUT[[g]]) <- c("lavaan.matrix", "matrix")
            }

        } # g
    } # labels

    if(nblocks == 1L && drop.list.single.group) {
        OUT <- OUT[[1]]
    } else {
        if(length(object@Data@group.label) > 0L) {
            names(OUT) <- unlist(object@Data@group.label)
        }
    }

    OUT
}

lav_object_inspect_zero_cell_tables <- function(object,
            add.labels = FALSE, add.class = FALSE,
            drop.list.single.group = FALSE) {

    # categorical?
    if(!object@Model@categorical) {
        warning("lavaan WARNING: no categorical variables in fitted model")
        return(invisible(list()))
    }

    lavdata <- object@Data

    # create 2-way tables
    TABLE <- lavTables(object, dimension = 2L, output = "data.frame",
                       statistic = NULL)

    # select tables with empty cells
    empty.id <- TABLE$id[which(TABLE$obs.freq == 0)]
   
    
    if(length(empty.id) == 0L) {
        # only when lavInspect() is used, give message
        if(add.class) {
            cat("(There are no tables with empty cells for this fitted model)\n")
        }
        return(invisible(list()))
    } else {
        OUT <- lav_tables_cells_format(TABLE[TABLE$id %in% empty.id,],
                   lavdata = lavdata,
                   drop.list.single.group = drop.list.single.group)
    }

    OUT
}
