/*
 * Copyright 2013 Canonical Ltd.
 *
 * This file is part of powerd.
 *
 * powerd is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * powerd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <assert.h>
#include <signal.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <limits.h>
#include <unistd.h>
#include <inttypes.h>
#include <sys/types.h>
#include <glib-object.h>
#include <gio/gio.h>
#include <glib.h>
#include <uuid/uuid.h>
#include <powerd.h>

#include "powerd-cli.h"

#define TEST_NUM_SYS_REQUESTS 5
#define TEST_NUM_DISP_REQUESTS (POWERD_NUM_DISPLAY_STATES * 2)

/* Set to TRUE during tests to silence expected errors */
gboolean silent_errors = FALSE;

GDBusProxy *powerd_proxy = NULL;
char test_dbusname[128] = "";

/* For tests */
GDBusConnection *powerd_cli_bus;
const char *powerd_cli_bus_name;

struct PublicSysRequest {
    const char *name;
    const char *owner;
    int state;
};

struct PublicDispRequest {
    const char *name;
    const char *owner;
    enum powerd_display_state state;
    guint32 flags;
};

struct SysRequestStats {
    const char *owner;
    const char *name;
    unsigned active_count;
    guint64 active_time;
    guint64 max_active_time;
    guint64 active_since;
};

struct DispRequestStats {
    const char *owner;
    const char *name;
    unsigned active_count;
    guint64 active_time;
    guint64 max_active_time;
    guint64 active_since;
    guint64 disp_on_time;
    guint64 disp_on_since;
    guint64 flag_on_time[POWERD_NUM_DISPLAY_FLAGS];
    guint64 flag_on_since[POWERD_NUM_DISPLAY_FLAGS];
};

static GMainLoop *main_loop = NULL;
static powerd_cookie_t main_cookie;
static gboolean checkForDbusName(const char *dbusname, int count,
        GArray *requests, gboolean isSys);
static void sig_handler(int signum);

static void
silence_errors(gboolean silent)
{
    silent_errors = silent;
}

static void
on_powerd_signal (GDBusProxy *proxy,
           gchar      *sender_name,
           gchar      *signal_name,
           GVariant   *parameters,
           gpointer    user_data)
{
    int system_state;
    int display_state;
    guint32 display_flags;

    if (!strcmp(signal_name,"SysPowerStateChange")) {
        g_variant_get(parameters, "(i)", &system_state);
        printf("Received %s: state=%d\n", signal_name, system_state);
    }
    else if (!strcmp(signal_name,"DisplayPowerStateChange")) {
        g_variant_get(parameters, "(iu)", &display_state, &display_flags);
        printf("Received %s: state=%d flags=%#08x\n",
            signal_name, display_state, display_flags);
    }
    else {
        cli_debug("Unknown signal from %s: %s", sender_name, signal_name);
    }
}

gboolean
requestSysState(const char *name, int state,
        powerd_cookie_t *cookie)
{
    GVariant *ret = NULL;
    GError *error = NULL;
    const char *cookie_ptr;
    gboolean success = TRUE;

    if (cookie == NULL) {
        cli_warn("NULL cookie passed to %s", __func__);
        return FALSE;
    }

    ret = g_dbus_proxy_call_sync(powerd_proxy,
            "requestSysState",
            g_variant_new("(si)", name, state),
            G_DBUS_CALL_FLAGS_NONE,
            -1,
            NULL,
            &error);
    if (ret == NULL) {
        cli_warn("requestSysState failed: %s", error->message);
        g_error_free(error);
        return FALSE;
    }

    g_variant_get(ret, "(&s)", &cookie_ptr);
    if (strlen(cookie_ptr) != sizeof(powerd_cookie_t) - 1) {
        cli_warn("Returned cookie has incorrect size");
        success = FALSE;
    } else {
        strncpy(*cookie, cookie_ptr, sizeof(powerd_cookie_t));
        cli_debug("Got cookie: %s", *cookie);
    }

    g_variant_unref(ret);
    return success;
}

static gboolean
requestDisplayState(struct PublicDispRequest pdr, const char *name,
        powerd_cookie_t *cookie)
{
    GVariant *ret = NULL;
    GError *error = NULL;
    const char *cookie_ptr;
    gboolean success = TRUE;

    if (cookie == NULL) {
        cli_warn("NULL cookie passed to %s", __func__);
        return FALSE;
    }

    ret = g_dbus_proxy_call_sync(powerd_proxy,
            "requestDisplayState",
            g_variant_new("(siu)", name, pdr.state, pdr.flags),
            G_DBUS_CALL_FLAGS_NONE,
            -1,
            NULL,
            &error);
    if (ret == NULL) {
        cli_warn("requestDisplayState failed: %s", error->message);
        g_error_free(error);
        return FALSE;
    }

    g_variant_get(ret, "(&s)", &cookie_ptr);
    if (strlen(cookie_ptr) != sizeof(powerd_cookie_t) - 1) {
        cli_warn("Returned cookie has incorrect size");
        success = FALSE;
    } else {
        strncpy(*cookie, cookie_ptr, sizeof(powerd_cookie_t));
        cli_debug("Got cookie: %s", *cookie);
    }

    g_variant_unref(ret);
    return success;
}

static gboolean
updateDisplayState(struct PublicDispRequest pdr,
        powerd_cookie_t cookie)
{
    GVariant *ret = NULL;
    GError *error = NULL;
    gboolean success = TRUE;

    if (cookie == NULL) {
        cli_warn("NULL cookie passed to %s", __func__);
        return FALSE;
    }

    ret = g_dbus_proxy_call_sync(powerd_proxy,
            "updateDisplayState",
            g_variant_new("(siu)", cookie, pdr.state, pdr.flags),
            G_DBUS_CALL_FLAGS_NONE,
            -1,
            NULL,
            &error);
    if (!ret) {
        cli_warn("updateDisplayState failed: %s", error->message);
        g_error_free(error);
        return FALSE;
    }

    g_variant_unref(ret);
    return success;
}

static GArray*
listSysRequests()
{
    GVariant *ret, *item = NULL;
    GVariantIter *iter = NULL;
    GArray *retarray = g_array_new(FALSE, FALSE, sizeof(struct PublicSysRequest));
    GError *error = NULL;
    struct PublicSysRequest psr;

    ret = g_dbus_proxy_call_sync(powerd_proxy,
            "listSysRequests",
            NULL,
            G_DBUS_CALL_FLAGS_NONE,
            -1,
            NULL,
            &error);
    if (ret == NULL) {
        cli_warn("listSysRequests failed: %s", error->message);
        g_error_free(error);
    }
    else {
        g_variant_get(ret, "(a(ssi))", &iter);
        while ((item = g_variant_iter_next_value (iter))) {
            g_variant_get_child (item, 0, "s", &psr.name);
            g_variant_get_child (item, 1, "s", &psr.owner);
            g_variant_get_child (item, 2, "i", &psr.state);
            g_array_append_val(retarray, psr);
            g_variant_unref(item);
        }
        g_variant_unref(ret);
    }
    return retarray;
}

static void
printSysRequests(GArray *requests)
{
    int i;
    struct PublicSysRequest *psr;
    printf("System State Requests:\n");
    if (requests->len == 0) {
        printf("  None\n");
    } else {
        for (i = 0; i < requests->len; i++) {
            psr = &g_array_index(requests, struct PublicSysRequest, i);
            printf("  Name: %s, Owner: %s, State: %d\n", psr->name,
                   psr->owner, psr->state);
        }
    }
}

static GArray*
listDisplayRequests()
{
    GVariant *ret, *item = NULL;
    GVariantIter *iter = NULL;
    GArray *retarray = g_array_new(FALSE, FALSE, sizeof(struct PublicDispRequest));
    GError *error = NULL;
    struct PublicDispRequest pdr;

    ret = g_dbus_proxy_call_sync(powerd_proxy,
            "listDisplayRequests",
            NULL,
            G_DBUS_CALL_FLAGS_NONE,
            -1,
            NULL,
            &error);
    if (ret == NULL) {
        cli_warn("listDisplayRequests failed: %s", error->message);
        g_error_free(error);
    }
    else {
        g_variant_get(ret, "(a(ssiu))", &iter);
        while ((item = g_variant_iter_next_value (iter))) {
            g_variant_get_child(item, 0, "s", &pdr.name);
            g_variant_get_child(item, 1, "s", &pdr.owner);
            g_variant_get_child(item, 2, "i", &pdr.state);
            g_variant_get_child(item, 3, "u", &pdr.flags);
            g_array_append_val(retarray, pdr);
            g_variant_unref(item);
        }
        g_variant_unref(ret);
    }
    return retarray;
}

static void
printDisplayRequests(GArray *requests)
{
    int i;
    struct PublicDispRequest *pdr;
    printf("Display State Requests:\n");
    if (requests->len == 0) {
        printf("  None\n");
    } else {
        for (i = 0; i < requests->len; i++) {
            pdr = &g_array_index(requests, struct PublicDispRequest, i);
            printf("  Name: %s, Owner: %s, State: %d, Flags: %#08x\n",
                   pdr->name, pdr->owner, pdr->state, pdr->flags);
        }
    }
}

static GArray *
getSysRequestStats(void)
{
    GVariant *ret, *item;
    GVariantIter *iter;
    GArray *retarray;
    GError *error;

    retarray =  g_array_new(FALSE, FALSE, sizeof(struct SysRequestStats));

    error = NULL;
    ret = g_dbus_proxy_call_sync(powerd_proxy,
            "getSysRequestStats",
            NULL,
            G_DBUS_CALL_FLAGS_NONE,
            -1,
            NULL,
            &error);
    if (ret == NULL) {
        cli_warn("getSysRequestStats failed: %s", error->message);
        g_error_free(error);
    } else {
        g_variant_get(ret, "(a(ssuttt))", &iter);
        while ((item = g_variant_iter_next_value (iter))) {
            struct SysRequestStats stats;
            g_variant_get_child(item, 0, "s", &stats.owner);
            g_variant_get_child(item, 1, "s", &stats.name);
            g_variant_get_child(item, 2, "u", &stats.active_count);
            g_variant_get_child(item, 3, "t", &stats.active_time);
            g_variant_get_child(item, 4, "t", &stats.max_active_time);
            g_variant_get_child(item, 5, "t", &stats.active_since);
            g_array_append_val(retarray, stats);
            g_variant_unref(item);
        }
        g_variant_unref(ret);
    }
    return retarray;
}

static void
printSysRequestStats(GArray *stats)
{
    int i;
    struct SysRequestStats *stat;
    printf("System Request Statistics:\n");
    if (stats->len == 0) {
        printf("  None\n");
    } else {
        printf("  %-16.16s %-20.20s %-8.8s %-16.16s %-16.16s %-16.16s\n",
               "", "", "Active", "", "Max Active", "");
        printf("  %-16.16s %-20.20s %-8.8s %-16.16s %-16.16s %-16.16s\n",
               "Owner", "Name", "Count", "Active Time", "Time", "Active Since");
        for (i = 0; i < stats->len; i++) {
            stat = &g_array_index(stats, struct SysRequestStats, i);
            printf("  %-16.16s %-20.20s %-8u %-16.6f %-16.6f %-16.6f\n",
                   stat->owner, stat->name, stat->active_count,
                   (double)stat->active_time / 1000000.0f,
                   (double)stat->max_active_time / 1000000.0f,
                   (double)stat->active_since / 1000000.0f);
        }
    }
}

static GArray *
getDispRequestStats(void)
{
    GVariant *ret, *item;
    GVariantIter *iter;
    GArray *retarray;
    GError *error;

    retarray =  g_array_new(FALSE, FALSE, sizeof(struct DispRequestStats));

    error = NULL;
    ret = g_dbus_proxy_call_sync(powerd_proxy,
            "getDispRequestStats",
            NULL,
            G_DBUS_CALL_FLAGS_NONE,
            -1,
            NULL,
            &error);
    if (ret == NULL) {
        cli_warn("getDispRequestStats failed: %s", error->message);
        g_error_free(error);
    } else {
        g_variant_get(ret, "(a(ssutttttatat))", &iter);
        while ((item = g_variant_iter_next_value (iter))) {
            struct DispRequestStats stats;
            GVariantIter *array_iter;
            guint64 val;
            int i;

            g_variant_get_child(item, 0, "s", &stats.owner);
            g_variant_get_child(item, 1, "s", &stats.name);
            g_variant_get_child(item, 2, "u", &stats.active_count);
            g_variant_get_child(item, 3, "t", &stats.active_time);
            g_variant_get_child(item, 4, "t", &stats.max_active_time);
            g_variant_get_child(item, 5, "t", &stats.active_since);
            g_variant_get_child(item, 6, "t", &stats.disp_on_time);
            g_variant_get_child(item, 7, "t", &stats.disp_on_since);

            g_variant_get_child(item, 8, "at", &array_iter);
            i = 0;
            while (g_variant_iter_loop(array_iter, "t", &val)) {
                if (i >= POWERD_NUM_DISPLAY_FLAGS)
                    break;
                stats.flag_on_time[i++] = val;
            }
            g_variant_iter_free(array_iter);

            g_variant_get_child(item, 9, "at", &array_iter);
            i = 0;
            while (g_variant_iter_loop(array_iter, "t", &val)) {
                if (i >= POWERD_NUM_DISPLAY_FLAGS)
                    break;
                stats.flag_on_since[i++] = val;
            }
            g_variant_iter_free(array_iter);

            g_array_append_val(retarray, stats);
            g_variant_unref(item);
        }
        g_variant_unref(ret);
    }
    return retarray;
}

static void
printDispRequestStats(GArray *stats)
{
    int i, j;
    struct DispRequestStats *stat;
    printf("Display Request Statistics:\n");
    if (stats->len == 0) {
        printf("  None\n");
    } else {
        printf("  %-16.16s %-20.20s %-8.8s %-16.16s %-16.16s %-16.16s %-16.16s %-16.16s",
               "", "", "Active", "", "Max Active", "", "Display On",
               "Display On");
        for (i = 0; i < POWERD_NUM_DISPLAY_FLAGS; i++)
            printf(" Flag %-2d          Flag %-2d         ", i, i);
        printf("\n  %-16.16s %-20.20s %-8.8s %-16.16s %-16.16s %-16.16s %-16.16s %-16.16s",
               "Owner", "Name", "Count", "Active Time", "Time", "Active Since",
               "Time", "Since");
        for (i = 0; i < POWERD_NUM_DISPLAY_FLAGS; i++)
            printf(" On Time          On Since        ");
        printf("\n");
        for (i = 0; i < stats->len; i++) {
            stat = &g_array_index(stats, struct DispRequestStats, i);
            printf("  %-16.16s %-20.20s %-8u %-16.6f %-16.6f %-16.6f %-16.6f %-16.6f",
                   stat->owner, stat->name, stat->active_count,
                   (double)stat->active_time / 1000000.0f,
                   (double)stat->max_active_time / 1000000.0f,
                   (double)stat->active_since / 1000000.0f,
                   (double)stat->disp_on_time / 1000000.0f,
                   (double)stat->disp_on_since / 1000000.0f);
            for (j = 0; j < POWERD_NUM_DISPLAY_FLAGS; j++) {
                printf(" %-16.6f", (double)stat->flag_on_time[j] / 1000000.0f);
                printf(" %-16.6f", (double)stat->flag_on_since[j] / 1000000.0f);
            }
            printf("\n");
        }
    }
}

gboolean
clearSysState(powerd_cookie_t cookie)
{
    GVariant *ret = NULL;
    GError *error = NULL;

    ret = g_dbus_proxy_call_sync(powerd_proxy,
            "clearSysState",
            g_variant_new("(s)", cookie),
            G_DBUS_CALL_FLAGS_NONE,
            -1,
            NULL,
            &error);
    if (ret == NULL) {
        cli_warn("clearSysState failed: %s", error->message);
        g_error_free(error);
        return FALSE;
    }
    g_variant_unref(ret);
    return TRUE;
}

static gboolean
clearDisplayState(powerd_cookie_t cookie)
{
    GVariant *ret = NULL;
    GError *error = NULL;

    ret = g_dbus_proxy_call_sync(powerd_proxy,
            "clearDisplayState",
            g_variant_new("(s)", cookie),
            G_DBUS_CALL_FLAGS_NONE,
            -1,
            NULL,
            &error);
    if (ret == NULL) {
        cli_warn("clearDisplayState failed: %s", error->message);
        g_error_free(error);
        return FALSE;
    }
    g_variant_unref(ret);
    return TRUE;
}

static void
listenForSignals(GDBusProxy *proxy)
{
    main_loop = g_main_loop_new (NULL, FALSE);

    signal(SIGINT, sig_handler);

    g_signal_connect(proxy, "g-signal", G_CALLBACK (on_powerd_signal), NULL);

    printf("Waiting for events, press ctrl-c to quit\n");
    g_main_loop_run(main_loop);
    g_main_loop_unref(main_loop);
}

static void
cb_child_watch(GPid pid,
                gint status,
                gpointer *data)
{
    GArray *requests = NULL;

    // Make sure that clean-up worked
    if (test_dbusname[0] != 0) {
        requests = listSysRequests();
        do_test(checkForDbusName(test_dbusname, 0, requests, TRUE));
        g_array_free(requests, TRUE);
        requests = listDisplayRequests();
        do_test(checkForDbusName(test_dbusname, 0, requests, TRUE));
        g_array_free(requests, TRUE);
    }
    else {
        printf("Did not find child's dbus name\n");
        printf("    result: FAILED\n");
    }

    // Make sure clean-up didn't remove our request, note we don't pass
    // the child PID here.
    requests = listSysRequests();
    do_test(checkForDbusName(powerd_cli_bus_name, 1, requests, TRUE));
    g_array_free(requests, TRUE);

    /* Close pid */
    g_spawn_close_pid(pid);

    g_main_loop_quit(main_loop);

    //cleanup
    do_test(clearSysState(main_cookie) == TRUE);
    requests = listSysRequests();
    do_test(checkForDbusName(powerd_cli_bus_name, 0, requests, TRUE));
    g_array_free(requests, TRUE);
}

static gboolean
cb_out_watch(GIOChannel *channel,
            GIOCondition cond,
            gpointer *data)
{
    gchar *string = NULL;
    gsize size;
    GIOStatus status;

    if (cond == G_IO_HUP) {
        g_io_channel_unref(channel);
        return FALSE;
    }

    status = g_io_channel_read_line(channel, &string, &size, NULL, NULL);
    if (status == G_IO_STATUS_NORMAL) {
        if (sscanf(string, "DBUSNAME: %s\n", test_dbusname) == 1) {
            cli_debug("Parent found child's dbusname as: %s", test_dbusname);
        }
        g_free(string);
    }
    return TRUE;
}

static void
runDbusNameVanishTests(char *progname)
{
    gint out = -1;
    gboolean ret;
    GPid pid;
    GIOChannel *out_ch = NULL;

    gchar *argv[] = {progname, "dbusnametest", NULL};

    main_loop = g_main_loop_new (NULL, FALSE);

    // Hold active state request as long we're running tests
    requestSysState("main-req", POWERD_SYS_STATE_ACTIVE, &main_cookie);

    ret = g_spawn_async_with_pipes(NULL, argv, NULL,
        G_SPAWN_SEARCH_PATH|G_SPAWN_DO_NOT_REAP_CHILD,
        NULL, NULL, &pid, NULL, &out, NULL, NULL);

    if (!ret) {
        cli_warn("Failed to spawn test app");
        return;
    }

    g_child_watch_add(pid, (GChildWatchFunc)cb_child_watch, NULL);

    out_ch = g_io_channel_unix_new(out);
    g_io_add_watch(out_ch, G_IO_IN|G_IO_HUP, (GIOFunc)cb_out_watch, NULL);

    g_main_loop_run(main_loop);
    g_main_loop_unref(main_loop);
}

static void
runSysTests()
{
    int i;
    powerd_cookie_t cookie, cookies[TEST_NUM_SYS_REQUESTS];
    GArray *requests = NULL;

    silence_errors(TRUE);

    // Hold active state request as long we're running tests
    requestSysState("main-req", POWERD_SYS_STATE_ACTIVE, &main_cookie);

    for (i = 0; i < TEST_NUM_SYS_REQUESTS; i++) {
        char name[16];
        snprintf(name, sizeof(name), "test-cookie-%d", i);
        do_test(requestSysState(name, POWERD_SYS_STATE_ACTIVE, &cookies[i]) == TRUE);
    }

    // Make sure we have at least NUM_REQUESTS + 1
    requests = listSysRequests();
    do_test(requests->len >= TEST_NUM_SYS_REQUESTS+1);
    do_test(checkForDbusName(powerd_cli_bus_name, TEST_NUM_SYS_REQUESTS+1, requests, TRUE));
    g_array_free(requests, TRUE);

    for (i = 0; i < TEST_NUM_SYS_REQUESTS; i++)
        do_test(clearSysState(cookies[i]) == TRUE);

    // We should still have at least 1 request here
    requests = listSysRequests();
    do_test(requests->len >= 1);
    do_test(checkForDbusName(powerd_cli_bus_name, 1, requests, TRUE));
    g_array_free(requests, TRUE);

    // Now try and freeing the same cookies again. They should all be
    // invalid and thus the request fails.
    for (i = 0; i < TEST_NUM_SYS_REQUESTS; i++)
        do_test(clearSysState(cookies[i]) == FALSE);

    // Test releasing an invalid cookie format
    do_test(clearSysState("bad cookie") == FALSE);

    // We should still have at least 1 request here
    requests = listSysRequests();
    do_test(requests->len >= 1);
    do_test(checkForDbusName(powerd_cli_bus_name, 1, requests, TRUE));
    g_array_free(requests, TRUE);

    //cannot request the suspend state, this will fail
    do_test(requestSysState("test-cookie", POWERD_SYS_STATE_SUSPEND, &cookie) == FALSE);
    //invalid values
    do_test(requestSysState("test-cookie", -1, &cookie) == FALSE);
    do_test(requestSysState("test-cookie", POWERD_NUM_POWER_STATES, &cookie) == FALSE);
    do_test(requestSysState("test-cookie", POWERD_NUM_POWER_STATES+1, &cookie) == FALSE);

    //cleanup
    do_test(clearSysState(main_cookie) == TRUE);
    do_test(checkForDbusName(powerd_cli_bus_name, 0, requests, TRUE));

    silence_errors(FALSE);
}

static void
runDisplayTests()
{
    int i;
    int state = 0;
    gboolean bright = FALSE;
    powerd_cookie_t cookie, cookies[TEST_NUM_DISP_REQUESTS];
    GArray *requests = NULL;
    struct PublicDispRequest pdr = {0,};

    silence_errors(TRUE);

    // Hold active state request as long we're running tests
    requestSysState("main-req", POWERD_SYS_STATE_ACTIVE, &main_cookie);

    for (i = 0; i < TEST_NUM_DISP_REQUESTS; i++) {
        char name[16];
        snprintf(name, sizeof(name), "disp-test-%d", i);
        pdr.state = state++;
        pdr.flags = 0;
        if (bright) {
            pdr.flags |= POWERD_DISPLAY_FLAG_BRIGHT;
        }
        do_test(requestDisplayState(pdr, name, &cookies[i]) == TRUE);
        if (state >= POWERD_NUM_DISPLAY_STATES) {
            state = 0;
        }
        bright = !bright;
    }

    // Make sure we have at least NUM_REQUESTS
    requests = listDisplayRequests();
    do_test(requests->len >= TEST_NUM_DISP_REQUESTS);
    // We should see our PID NUM_REQUESTS times
    do_test(checkForDbusName(powerd_cli_bus_name, TEST_NUM_DISP_REQUESTS, requests, FALSE));
    g_array_free(requests, TRUE);

    for (i = 0; i < TEST_NUM_DISP_REQUESTS; i++)
        do_test(clearDisplayState(cookies[i]) == TRUE);

    requests = listDisplayRequests();
    // We aren't holding anymore with this PID
    do_test(checkForDbusName(powerd_cli_bus_name, 0, requests, FALSE));
    g_array_free(requests, TRUE);

    // Now try and freeing the same cookies again. They should all be
    // invalid and thus the request fails.
    for (i = 0; i < TEST_NUM_DISP_REQUESTS; i++)
        do_test(clearDisplayState(cookies[i]) == FALSE);

    // Test releasing an invalid cookie format
    do_test(clearDisplayState("bad cookie") == FALSE);

    //invalid values
    pdr.state = -1;
    do_test(requestDisplayState(pdr, "disp-test", &cookie) == FALSE);

    pdr.state = POWERD_NUM_DISPLAY_STATES;
    do_test(requestDisplayState(pdr, "disp-test", &cookie) == FALSE);

    pdr.state = POWERD_DISPLAY_STATE_ON;
    pdr.flags = 0xdeadbeef;
    do_test(requestDisplayState(pdr, "disp-test", &cookie) == FALSE);

    // updateDisplayState tests
    pdr.state = POWERD_DISPLAY_STATE_ON;
    pdr.flags = 0;
    do_test(requestDisplayState(pdr, "disp-test", &cookie) == TRUE);

    // update with same state should be okay
    do_test(updateDisplayState(pdr, cookie) == TRUE);

    pdr.state = POWERD_DISPLAY_STATE_DONT_CARE;
    do_test(updateDisplayState(pdr, cookie) == TRUE);

    pdr.flags = POWERD_DISPLAY_FLAG_DISABLE_AUTOBRIGHTNESS;
    do_test(updateDisplayState(pdr, cookie) == TRUE);

    pdr.state = -1;
    do_test(updateDisplayState(pdr, cookie) == FALSE);

    do_test(clearDisplayState(cookie) == TRUE);

    // update with same cookie after clear should fail
    do_test(updateDisplayState(pdr, cookie) == FALSE);

    // update with garbage cookie should faile
    do_test(updateDisplayState(pdr, "bad cookie") == FALSE);

    //cleanup
    do_test(clearSysState(main_cookie) == TRUE);
    do_test(checkForDbusName(powerd_cli_bus_name, 0, requests, TRUE));

    silence_errors(FALSE);
}

// Check that dbus owner appears count times in a list of sys/disp requests
static gboolean
checkForDbusName(const char *dbusname,
        int count,
        GArray *requests,
        gboolean isSys)
{
    int i;
    int found = 0;
    struct PublicSysRequest *psr;
    struct PublicDispRequest *pdr;

    if (requests == NULL) {
        cli_warn("NULL requests passed to %s", __func__);
        return FALSE;
    }

    for (i=0; i<requests->len; i++)
    {
        if (isSys) {
            psr = &g_array_index(requests, struct PublicSysRequest, i);
            if (!(strcmp(psr->owner, dbusname))) {
                found++;
            }
        }
        else {
            pdr = &g_array_index(requests, struct PublicDispRequest, i);
            if (!(strcmp(pdr->owner, dbusname))) {
                found++;
            }
        }
    }

    if (found != count) {
        cli_debug("Expected %d requests from DbusName (%s), found %d",
            count, dbusname, found);
        return FALSE;
    }
    return TRUE;
}

static void
sig_handler(int signum)
{
    g_main_loop_quit(main_loop);
}

/* Caller ensures that we have at least 4 arguments in argv */
static gboolean
buildDisplayRequest(char **argv,
        int argc,
        struct PublicDispRequest *pdr)
{
    int i;

    if (!strcasecmp(argv[2],"on")) {
        pdr->state = POWERD_DISPLAY_STATE_ON;
        cli_debug("Requesting Display On");
    }
    else if (!strcasecmp(argv[2],"dc")) {
        pdr->state = POWERD_DISPLAY_STATE_DONT_CARE;
        cli_debug("Requesting Display Don't Care");
    }
    else {
        fprintf(stderr,"invalid state %s, must be either on or dc\n",
            argv[2]);
        return FALSE;
    }

    pdr->flags = 0;
    for (i=3; i<argc; i++) {
        if (!strcmp(argv[i],"proximity")) {
            pdr->flags |= POWERD_DISPLAY_FLAG_USE_PROXIMITY;
            cli_debug("Requesting Proximity Sensor Enabled");
        }
        else if (!strcmp(argv[i],"disableab")) {
            pdr->flags |= POWERD_DISPLAY_FLAG_DISABLE_AUTOBRIGHTNESS;
            cli_debug("Requesting Proximity Disable AutoBrightness");
        }
        else if (!strcasecmp(argv[i],"bright")) {
            pdr->flags |= POWERD_DISPLAY_FLAG_BRIGHT;
            cli_debug("Requesting Bright");
        }
    }

    return TRUE;
}

static void
setUserAutobrightness(gboolean enable)
{
    GVariant *ret = NULL;
    GError *error = NULL;
    
    ret = g_dbus_proxy_call_sync(powerd_proxy,
            "userAutobrightnessEnable",
            g_variant_new("(b)", enable),
            G_DBUS_CALL_FLAGS_NONE,
            -1, NULL, &error);
    if (!ret) {
        cli_warn("userAutobrightnessEnable failed: %s", error->message);
        g_error_free(error);
    } else {
        g_variant_unref(ret);
    }
}

static gboolean
getBrightnessParams(int *min, int *max, int *dflt, gboolean *ab_supported)
{
    GVariant *ret = NULL;
    GError *error = NULL;

    ret = g_dbus_proxy_call_sync(powerd_proxy,
            "getBrightnessParams",
            NULL,
            G_DBUS_CALL_FLAGS_NONE,
            -1, NULL, &error);
    if (!ret) {
        cli_warn("getBrightnessParams failed: %s", error->message);
        g_error_free(error);
        return FALSE;
    }

    g_variant_get(ret, "((iiib))", min, max, dflt, ab_supported);
    g_variant_unref(ret);
    return TRUE;
}

static void setUserBrightness(int brightness)
{
    GVariant *ret = NULL;
    GError *error = NULL;
    
    ret = g_dbus_proxy_call_sync(powerd_proxy,
            "setUserBrightness",
            g_variant_new("(i)", brightness),
            G_DBUS_CALL_FLAGS_NONE,
            -1, NULL, &error);
    if (!ret) {
        cli_warn("setUserBrightness failed: %s", error->message);
        g_error_free(error);
    } else {
        g_variant_unref(ret);
    }
}

static void
usage(const char *progname)
{
    printf("\n\n%s Options:\n",progname);
    printf("active - request the active state, prevent the system from\n"\
           "\tsuspending. This does not modify the screen state.\n");
    printf("active-nc - same as active, but relies on powerd to cleanup\n"\
            "\tthe request. This is only for testing and will immediately\n"\
            "exit, causing the request to be dropped.\n");
    printf("autobrightness <enable|disable> - enable or disable autobrightness.\n");
    printf("brightness <brightness> - set user screen brightness.\n");
    printf("brightness-params - get brightness parameters (min, max, etc.)\n");
    printf("clear-sys <cookie> - clear a System state request given a cookie.\n");
    printf("clear-disp <cookie> - clear a Display state request given a cookie.\n");
    printf("client-test - test powerd registration / ack API.\n");
    printf("display <on|dc> [bright] [proximity] [disableab]\n"\
           "\tMake a display state request with the input parameters.\n"\
           "\tThe first argument represents the state of the display:\n"\
           "\tOn (on) or Don't Care (dc),\n"\
           "\tThe final optional arguments respectively:\n"\
           "\t * make the screen bright [bright]\n"\
           "\t * enable the proximity sensor [proximity]\n"\
           "\t * disable autobrightness [disableab]\n");
    printf("help\t- display this usage information.\n");
    printf("list\t- list outstanding requests.\n");
    printf("listen\t- listen for signals from powerd. This runs a\n"\
           "\t  gmainloop and must be manually killed.\n");
    printf("stats\t- print request statistics.\n");
    printf("test\t- runs tests.\n");
}

static void
sigint_quit(int signal)
{
    /* do nothing */
}

int main (int argc, char **argv)
{
    GError *error = NULL;
    uid_t myeuid;
    GArray *requests = NULL;
    struct PublicDispRequest pdr;
    GDBusConnection *bus = NULL;
    const char *bus_name = NULL;
    powerd_cookie_t cookie;

    if ((argc < 2) || (argc > 6)) {
        fprintf(stderr,"Incorrect number of options\n");
        usage(argv[0]);
        return -1;
    }

    if ((strcmp(argv[1],"list")) &&
       (strcmp(argv[1],"stats")) &&
       (strcmp(argv[1],"active")) &&
       (strcmp(argv[1],"active-nc")) &&
       (strcmp(argv[1],"test")) &&
       (strcmp(argv[1],"dbusnametest")) &&
       (strcmp(argv[1],"listen")) &&
       (strcmp(argv[1],"display")) &&
       (strcmp(argv[1],"help")) &&
       (strcmp(argv[1],"clear-disp")) &&
       (strcmp(argv[1],"clear-sys")) &&
       (strcmp(argv[1],"client-test")) &&
       (strcmp(argv[1], "autobrightness")) &&
       (strcmp(argv[1], "brightness-params")) &&
       (strcmp(argv[1], "brightness"))) {
        fprintf(stderr,"Invalid option %s\n",argv[1]);
        usage(argv[0]);
        return -1;
    }

    if (!strcmp(argv[1],"help")) {
        usage(argv[0]);
        return 0;
    }

    myeuid = geteuid();
    if (myeuid != 0) {
        fprintf(stderr,"%s: Running as user is not fully supported.\n", argv[0]);
    }

    signal(SIGINT, sigint_quit);

    powerd_proxy = g_dbus_proxy_new_for_bus_sync(G_BUS_TYPE_SYSTEM,
                G_DBUS_PROXY_FLAGS_NONE,
                NULL,
                "com.canonical.powerd",
                "/com/canonical/powerd",
                "com.canonical.powerd",
                NULL,
                &error);

    if (error != NULL) {
        cli_warn("could not connect to powerd: %s", error->message);
        g_error_free(error);
        return -1;
    }

    if (!strcmp(argv[1],"list")) {
        requests = listSysRequests();
        printSysRequests(requests);
        g_array_free(requests, TRUE);
        requests = listDisplayRequests();
        printDisplayRequests(requests);
        g_array_free(requests, TRUE);
    }
    else if (!strcmp(argv[1], "stats")) {
        requests = getSysRequestStats();
        printSysRequestStats(requests);
        g_array_free(requests, TRUE);

        requests = getDispRequestStats();
        printDispRequestStats(requests);
        g_array_free(requests, TRUE);
    }
    else if (!strcmp(argv[1],"active-nc")) {
        requestSysState("active-nc", POWERD_SYS_STATE_ACTIVE,&cookie);
        printf("Power State requested, cookie is %s\n", cookie);
        return 0;
    }
    else if (!strcmp(argv[1],"active")) {
        requestSysState("active", POWERD_SYS_STATE_ACTIVE,&cookie);
        printf("Power State requested, cookie is %s.\nPress ctrl-c to exit.\n",
            cookie);
        pause();  /* wait for SIGINT */
        clearSysState(cookie);
        return 0;
    }
    else if (!strcmp(argv[1],"display")) {
        if (argc < 3) {
            fprintf(stderr,"display requires other arguments\n");
            usage(argv[0]);
            return -1;
        }
        if (buildDisplayRequest(argv,argc,&pdr) == FALSE) {
            usage(argv[0]);
            return -1;
        }
        requestDisplayState(pdr, "disp-test", &cookie);
        printf("Display State requested, cookie is %s.\nPress ctrl-c to exit.\n",
            cookie);
        pause();  /* wait for SIGINT */
        clearDisplayState(cookie);
        return 0;
    }
    else if (!strncmp(argv[1],"clear",strlen("clear"))) {
        if (argc != 3) {
            fprintf(stderr,"clear requires a cookie\n");
            usage(argv[0]);
            return -1;
        }
        else {
            uuid_t uuid;
            if (uuid_parse(argv[2], uuid)==0) {
                if (!strcmp(argv[1],"clear-sys")) {
                    clearSysState(argv[2]);
                }
                else if (!strcmp(argv[1],"clear-disp")) {
                    clearDisplayState(argv[2]);
                }
                else {
                    // Note: We should never get here due to earlier checks
                    fprintf(stderr,"Invalid option %s\n",argv[1]);
                    usage(argv[0]);
                    return -1;
                }
            }
            else {
                fprintf(stderr,"invalid cookie, cookie should be a UUID\n");
                usage(argv[0]);
            }
        }
    }
    else if (!strcmp(argv[1],"test")) {
        powerd_cli_bus = g_bus_get_sync(G_BUS_TYPE_SYSTEM, NULL, NULL);
        if (!powerd_cli_bus) {
            fprintf(stderr, "Could not connect to message bus\n");
            exit(-1);
        }
        powerd_cli_bus_name = g_dbus_connection_get_unique_name(powerd_cli_bus);
        if (!powerd_cli_bus_name) {
            fprintf(stderr, "Could not get unique bus name\n");
            exit(-1);
        }

        runDbusNameVanishTests(argv[0]);
        runSysTests();
        runDisplayTests();

        g_object_unref(powerd_cli_bus);
        powerd_cli_bus = NULL;
        powerd_cli_bus_name = NULL;
    }
    else if (!strcmp(argv[1],"listen")) {
        listenForSignals(powerd_proxy);
    }
    else if (!strcmp(argv[1],"dbusnametest")) {
        bus = g_bus_get_sync(G_BUS_TYPE_SYSTEM, NULL, NULL);
        bus_name = g_dbus_connection_get_unique_name(bus);
        cli_debug("Child dbus name is %s", bus_name);
        // This printf is read by the parent
        printf("DBUSNAME: %s\n", bus_name);
        // Grab some requests so we can see that they get cleared later
        requestSysState("dbusnametest1", POWERD_SYS_STATE_ACTIVE,&cookie);
        requestSysState("dbusnametest2", POWERD_SYS_STATE_ACTIVE,&cookie);
        requestSysState("dbusnametest3", POWERD_SYS_STATE_ACTIVE,&cookie);
        pdr.state = POWERD_DISPLAY_STATE_DONT_CARE;
        pdr.flags = 0;
        requestDisplayState(pdr, "dbusnametest1", &cookie);
        requestDisplayState(pdr, "dbusnametest2", &cookie);
        requestDisplayState(pdr, "dbusnametest3", &cookie);
        g_object_unref(bus);
        // Exit here without cleanup
        return 0;
    }
    else if (!strcmp(argv[1], "client-test")) {
        powerd_cli_client_test(powerd_proxy);
    } else if (!strcmp(argv[1], "autobrightness")) {
        gboolean enable;
        if (argc != 3) {
            usage(argv[0]);
            exit(-1);
        }
        if (!strcmp(argv[2], "enable")) {
            enable = TRUE;
        } else if (!strcmp(argv[2], "disable")) {
            enable = FALSE;
        } else {
            fprintf(stderr, "Invalid autobrightness state\n");
            usage(argv[0]);
            exit(-1);
        }
        setUserAutobrightness(enable);
    } else if (!strcmp(argv[1], "brightness-params")) {
        int min, max, dflt, ab;
        if (getBrightnessParams(&min, &max, &dflt, &ab)) {
            printf("Minimum Brightness: %d\n"
                   "Maximum Brightness: %d\n"
                   "Default Brightness: %d\n"
                   "Autobrightness:     %ssupported\n",
                   min, max, dflt, ab ? "" : "not ");
        }
    } else if (!strcmp(argv[1], "brightness")) {
        long brightness;
        char *endp;
        if (argc != 3) {
            usage(argv[0]);
            exit(-1);
        }
        errno = 0;
        brightness = strtol(argv[2], &endp, 10);
        if (errno == ERANGE || *endp != '\0' ||
            brightness < 0 || brightness > INT_MAX) {
            fprintf(stderr, "Invalid brightness %s\n", argv[2]);
            usage(argv[0]);
            exit(-1);
        }
        setUserBrightness((int)brightness);
    }
    return 0;
}
