/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment OpenGL ES-CM 1.1 plugin
 *
 * Copyright © 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Loïc Molinari <loic@fluendo.com>
 */

/*
 * Object used to handle the OpenGL ES texturing.
 *
 * FIXME: The behaviour is currently undefined if the user sends us an image
 *        with an unsupported color space (ie a pixel format which is not
 *        returned by pgm_viewport_get_pixel_formats).
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <string.h>     /* memset */
#include "pgmglestexture.h"

GST_DEBUG_CATEGORY_EXTERN (pgm_gles_debug);
#define GST_CAT_DEFAULT pgm_gles_debug

/* Keep track of the context and OpenGL procedures */
static const PgmGlesContextProcAddress *gles = NULL;
static const PgmGlesContext *glescontext = NULL;

/* Private functions  */

/* Returns number rounded up to the next power of 2. This is the branch-free
 * implementation detailed in "Hacker's Delight" by Henry S. Warren, Jr. */
static guint
get_upper_power_of_two (guint number)
{
  number = number - 1;
  number = number | (number >> 1);
  number = number | (number >> 2);
  number = number | (number >> 4);
  number = number | (number >> 8);
  number = number | (number >> 16);
  return number + 1;
}

/* Specifies texture parameters of the currently bound texture */
static void
set_texture_parameters (PgmGlesTexture *glestexture)
{
  /* Texture wrapping */
  gles->tex_parameter_i (PGM_GLES_TEXTURE_2D, PGM_GLES_TEXTURE_WRAP_S,
                         glestexture->wrap_s);
  gles->tex_parameter_i (PGM_GLES_TEXTURE_2D, PGM_GLES_TEXTURE_WRAP_T,
                         glestexture->wrap_t);

  /* Texture filtering */
  gles->tex_parameter_i (PGM_GLES_TEXTURE_2D, PGM_GLES_TEXTURE_MIN_FILTER,
                         glestexture->filter);
  gles->tex_parameter_i (PGM_GLES_TEXTURE_2D, PGM_GLES_TEXTURE_MAG_FILTER,
                         glestexture->filter);
}

/* */
static void
create_texture (PgmGlesTexture *glestexture)
{
  guint8 *pixels =
    g_malloc0 (glestexture->width_pot * glestexture->height_pot * 4);

  gles->bind_texture (PGM_GLES_TEXTURE_2D, glestexture->id[0]);
  gles->tex_image_2d (PGM_GLES_TEXTURE_2D, 0, glestexture->format,
                      glestexture->width_pot, glestexture->height_pot, 0,
                      glestexture->format, PGM_GLES_UNSIGNED_BYTE, pixels);
  set_texture_parameters (glestexture);

  g_free (pixels);
}

/*  */
static void
upload_texture (PgmGlesTexture *glestexture,
                void *buffer)
{
  gles->bind_texture (PGM_GLES_TEXTURE_2D, glestexture->id[0]);
  gles->tex_sub_image_2d (PGM_GLES_TEXTURE_2D, 0, 0, 0, glestexture->width,
                          glestexture->height, glestexture->format,
                          PGM_GLES_UNSIGNED_BYTE, buffer);
}

/* */
static void
update_format (PgmGlesTexture *glestexture,
               PgmImagePixelFormat csp)
{
  if (csp == PGM_IMAGE_RGB)
    glestexture->format = PGM_GLES_RGB;
  else if (csp == PGM_IMAGE_RGBA)
    glestexture->format = PGM_GLES_RGBA;
  else if (glescontext->feature_mask & PGM_GLES_FEAT_TEXTURE_FORMAT_BGRA
           && csp == PGM_IMAGE_BGRA)
    glestexture->format = PGM_GLES_BGRA;
  else
    glestexture->format = -1;
}

/* Update the normalized size fields */
static void
update_normalized_size (PgmGlesTexture *texture)
{
  texture->norm_width = (gfloat) texture->width / texture->width_pot;
  texture->norm_height = (gfloat) texture->height / texture->height_pot;
  texture->inv_norm_width = 1.0f / texture->norm_width;
  texture->inv_norm_height = 1.0f / texture->norm_height;
}

/* Frees the current stored buffer if any */
static void
free_buffer (PgmGlesTexture *glestexture)
{
  switch (glestexture->storage)
    {
    case PGM_GLES_TEXTURE_GST_BUFFER:
      if (glestexture->data.gstbuffer)
        {
          gst_buffer_unref (glestexture->data.gstbuffer);
          glestexture->data.gstbuffer = NULL;
        }
      break;

    case PGM_GLES_TEXTURE_PIXBUF:
      if (glestexture->data.pixbuf)
        {
          gdk_pixbuf_unref (glestexture->data.pixbuf);
          glestexture->data.pixbuf = NULL;
        }
      break;

    case PGM_GLES_TEXTURE_BUFFER:
      if (glestexture->data.buffer)
        {
          if (!glestexture->shared)
            g_free (glestexture->data.buffer);
          glestexture->data.buffer = NULL;
        }
      break;

    default:
      break;
    }

  glestexture->storage = PGM_GLES_TEXTURE_CLEAN;
}

/* Constructor */
static void
init_texture (PgmGlesTexture *glestexture)
{
  /* Content */
  glestexture->storage = PGM_GLES_TEXTURE_CLEAN;
  glestexture->data.gstbuffer = NULL;
  glestexture->width = -1;
  glestexture->height = -1;
  glestexture->width_pot = -1;
  glestexture->height_pot = -1;
  glestexture->stride = -1;
  glestexture->size = -1;
  glestexture->norm_width = -1.0f;
  glestexture->norm_height = -1.0f;
  glestexture->inv_norm_width = -1.0f;
  glestexture->inv_norm_height = -1.0f;
  glestexture->format = -1;

  /* Parameters */
  glestexture->matrix = pgm_mat4x4_new_identity ();
  glestexture->filter = PGM_GLES_LINEAR;
  glestexture->wrap_s = PGM_GLES_CLAMP_TO_EDGE;
  glestexture->wrap_t = PGM_GLES_CLAMP_TO_EDGE;

  /* State flags */
  glestexture->flags = PGM_GLES_TEXTURE_IDENTITY_MATRIX;

  /* Indentifiants */
  glestexture->id = NULL;

  glestexture->shared = 0;
}

/* Destructor */
static void
dispose_texture (PgmGlesTexture *glestexture)
{
  free_buffer (glestexture);

  if (glestexture->id)
    pgm_gles_texture_clean (glestexture);

  pgm_mat4x4_free (glestexture->matrix);
  glestexture->matrix = NULL;
}

/* Global pointers initialization */
static void
class_init (PgmGlesContext *_glescontext)
{
  glescontext = _glescontext;
  gles = glescontext->gles;
}

/* Public functions */

PgmGlesTexture *
pgm_gles_texture_new (PgmGlesContext *_glescontext)
{
  PgmGlesTexture *glestexture;

  /* Initialize the global pointers at first instantiation */
  if (G_UNLIKELY (!glescontext))
    class_init (_glescontext);

  /* And instantiate */
  glestexture = g_slice_new0 (PgmGlesTexture);
  init_texture (glestexture);

  return glestexture;
}

void
pgm_gles_texture_free (PgmGlesTexture *glestexture)
{
  g_return_if_fail (glestexture != NULL);

  dispose_texture (glestexture);
  g_slice_free (PgmGlesTexture, glestexture);
  glestexture = NULL;
}

void
pgm_gles_texture_set_buffer (PgmGlesTexture *glestexture,
                             guchar *buffer,
                             PgmImagePixelFormat csp,
                             guint width,
                             guint height,
                             guint size,
                             guint stride,
                             gboolean share)
{
  free_buffer (glestexture);

  glestexture->storage = PGM_GLES_TEXTURE_BUFFER;
  if (share)
    {
      glestexture->data.buffer = buffer;
      glestexture->shared = 1;
    }
  else
    {
      glestexture->data.buffer = g_memdup (buffer, size);
      glestexture->shared = 0;
    }
  glestexture->width = width;
  glestexture->height = height;
  glestexture->stride = stride;
  glestexture->size = size;
  glestexture->width_pot = get_upper_power_of_two (width);
  glestexture->height_pot = get_upper_power_of_two (height);
  glestexture->csp = csp;

  update_format (glestexture, csp);
  update_normalized_size (glestexture);
}

void
pgm_gles_texture_set_pixbuf (PgmGlesTexture *glestexture,
                             GdkPixbuf *pixbuf)
{
  free_buffer (glestexture);

  glestexture->storage = PGM_GLES_TEXTURE_PIXBUF;
  glestexture->data.pixbuf = gdk_pixbuf_ref (pixbuf);
  glestexture->width = gdk_pixbuf_get_width (pixbuf);
  glestexture->height = gdk_pixbuf_get_height (pixbuf);
  glestexture->stride = gdk_pixbuf_get_rowstride (pixbuf);
  glestexture->size = glestexture->stride * glestexture->height;
  glestexture->width_pot = get_upper_power_of_two (glestexture->width);
  glestexture->height_pot = get_upper_power_of_two (glestexture->height);
  if (gdk_pixbuf_get_has_alpha (pixbuf))
    glestexture->csp = PGM_IMAGE_RGBA;
  else
    glestexture->csp = PGM_IMAGE_RGB;

  update_format (glestexture, glestexture->csp);
  update_normalized_size (glestexture);
}

void
pgm_gles_texture_set_gst_buffer (PgmGlesTexture *glestexture,
                                 GstBuffer *gstbuffer,
                                 PgmImagePixelFormat csp,
                                 guint width,
                                 guint height,
                                 guint stride)
{
  free_buffer (glestexture);

  glestexture->storage = PGM_GLES_TEXTURE_GST_BUFFER;
  glestexture->data.gstbuffer = gst_buffer_ref (gstbuffer);
  glestexture->size = GST_BUFFER_SIZE (gstbuffer);
  glestexture->width = width;
  glestexture->width_pot = get_upper_power_of_two (width);
  glestexture->height = height;
  glestexture->height_pot = get_upper_power_of_two (height);
  glestexture->stride = stride;
  glestexture->csp = csp;

  update_format (glestexture, csp);
  update_normalized_size (glestexture);
}

void
pgm_gles_texture_update_gst_buffer (PgmGlesTexture *glestexture,
                                    GstBuffer *gstbuffer)
{
  if (glestexture->data.gstbuffer)
    gst_buffer_unref (glestexture->data.gstbuffer);
  glestexture->data.gstbuffer = gst_buffer_ref (gstbuffer);
  glestexture->storage = PGM_GLES_TEXTURE_GST_BUFFER;
}

void
pgm_gles_texture_bind (PgmGlesTexture *glestexture)
{
  g_return_if_fail (glestexture != NULL);

  if (!glestexture->id)
    return;

  gles->bind_texture (PGM_GLES_TEXTURE_2D, glestexture->id[0]);

  /* Push the current texture matrix, and load our matrix */
  if (!(glestexture->flags & PGM_GLES_TEXTURE_IDENTITY_MATRIX))
    {
      gles->matrix_mode (PGM_GLES_TEXTURE);
      gles->push_matrix ();
      gles->load_matrix_f (glestexture->matrix->m);
      gles->matrix_mode (PGM_GLES_MODELVIEW);
    }
}

void
pgm_gles_texture_unbind (PgmGlesTexture *glestexture)
{
  g_return_if_fail (glestexture != NULL);

  if (!glestexture->id)
    return;

  gles->bind_texture (PGM_GLES_TEXTURE_2D, 0);

  /* Pop our texture matrix from the stack */
  if (!(glestexture->flags & PGM_GLES_TEXTURE_IDENTITY_MATRIX))
    {
      gles->matrix_mode (PGM_GLES_TEXTURE);
      gles->pop_matrix ();
      gles->matrix_mode (PGM_GLES_MODELVIEW);
    }
}

void
pgm_gles_texture_generate (PgmGlesTexture *glestexture)
{
  pgm_gles_texture_clean (glestexture);

  glestexture->id = g_slice_alloc0 (sizeof (guint));
  gles->gen_textures (1, glestexture->id);

  create_texture (glestexture);
}

void
pgm_gles_texture_clean (PgmGlesTexture *glestexture)
{
  if (glestexture->id)
    {
      gles->delete_textures (1, glestexture->id);
      g_slice_free1 (sizeof (guint), glestexture->id);
      glestexture->id = NULL;
    }
}

void
pgm_gles_texture_upload (PgmGlesTexture *glestexture)
{
  void *buffer;

  if (G_UNLIKELY (!glestexture->id))
    return;

  switch (glestexture->storage)
    {
    case PGM_GLES_TEXTURE_GST_BUFFER:
      buffer = GST_BUFFER_DATA (glestexture->data.gstbuffer);
      break;

    case PGM_GLES_TEXTURE_PIXBUF:
      buffer = gdk_pixbuf_get_pixels (glestexture->data.pixbuf);
      break;

    case PGM_GLES_TEXTURE_BUFFER:
      buffer = glestexture->data.buffer;
      break;

    default:
      return;
    }

  /* FIXME: This test must not be there! At that point the content must be valid
   * in every case. The issue is that it rarely appears that a buffer coming
   * from a GdkPixbuf is NULL, causing a crash in the OpenGL texture update
   * function. That is a workaround to fix this issue, the real source of the
   * problem still need to be found though. */
  if (G_LIKELY (buffer != NULL))
    upload_texture (glestexture, buffer);

  free_buffer (glestexture);
}

void
pgm_gles_texture_update (PgmGlesTexture *glestexture)
{
  if (!glestexture->id)
    return;

  gles->bind_texture (PGM_GLES_TEXTURE_2D, glestexture->id[0]);
  set_texture_parameters (glestexture);
}

void
pgm_gles_texture_set_matrix (PgmGlesTexture *glestexture,
                             PgmMat4x4 *matrix)
{
  pgm_mat4x4_set_from_mat4x4 (glestexture->matrix, matrix);

  /* Check identity so that we can avoid pushing our texture matrix */
  if (!pgm_mat4x4_is_identity (matrix))
    {
      PgmMat4x4 *scale, *transform, *transpose;

      /* We need to scale, apply the transformation and unscale to hide
       * the use of power-of-two textures. */
      scale = pgm_mat4x4_new_scale_from_scalars (glestexture->norm_width,
                                                 glestexture->norm_height, 1.0f);
      transform = pgm_mat4x4_multiply_mat4x4 (scale, glestexture->matrix);
      pgm_mat4x4_scale_from_scalars (transform, glestexture->inv_norm_width,
                                     glestexture->inv_norm_height, 1.0f);
      transpose = pgm_mat4x4_transpose (transform);
      pgm_mat4x4_set_from_mat4x4 (glestexture->matrix, transpose);
      pgm_mat4x4_free (scale);
      pgm_mat4x4_free (transform);
      pgm_mat4x4_free (transpose);

      glestexture->flags &= ~PGM_GLES_TEXTURE_IDENTITY_MATRIX;
    }
  else
    glestexture->flags |= PGM_GLES_TEXTURE_IDENTITY_MATRIX;
}
