/* NVTV Recalc -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This is the recalc code from Brooktree, slightly modified. The
 * legal status of this code is unclear, but since it was freely
 * available on the Brooktree public ftp server before Brooktree
 * became Conexant, and most of the equations are easily derived from
 * the public documentation, I think it is save to use it.
 * 
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "local.h"
#include "tv_chip.h"
#include "calc_bt.h"

#define DebugOut(x) printf(x)

/* ucPixelInputMode */

#define BT868_PIXEL_INPUT_MODE_24BIT_RGB_MUX	(1 << 0)
#define BT868_PIXEL_INPUT_MODE_16BIT_YCRCB_MUX	(1 << 1)

/*  dwFilterFlags */

#define BT868_FILTER_FLAGS_DISFFILT	(1 << 0)

/* dwRegisterFlags */

#define BT868_REGISTER_FLAGS_ENOUT	((unsigned long) 1L << 0)
#define BT868_REGISTER_FLAGS_ENBLANKO	((unsigned long) 1L << 1) 
#define BT868_REGISTER_FLAGS_VSYNCDUR	((unsigned long) 1L << 2) 
#define BT868_REGISTER_FLAGS_SETUP	((unsigned long) 1L << 3)
#define BT868_REGISTER_FLAGS_VBLANKDLY	((unsigned long) 1L << 4)
#define BT868_REGISTER_FLAGS_ENXCLK	((unsigned long) 1L << 5)
#define BT868_REGISTER_FLAGS_BYPLL	((unsigned long) 1L << 6)
#define BT868_REGISTER_FLAGS_ECLIP	((unsigned long) 1L << 7)
#define BT868_REGISTER_FLAGS_SCRESET	((unsigned long) 1L << 8)
#define BT868_REGISTER_FLAGS_NIOUT	((unsigned long) 1L << 9)
#define BT868_REGISTER_FLAGS_PAL	((unsigned long) 1L << 10)
#define BT868_REGISTER_FLAGS_625LINE	((unsigned long) 1L << 11)
#define BT868_REGISTER_FLAGS_ENDOT	((unsigned long) 1L << 12)
#define BT868_REGISTER_FLAGS_SLAVE	((unsigned long) 1L << 13)
#define BT868_REGISTER_FLAGS_MODE2X	((unsigned long) 1L << 14)
#define BT868_REGISTER_FLAGS_ECBAR	((unsigned long) 1L << 15)

/*  dwImplementationFlags; */

#define BT868_IMPLEMENTATION_FLAGS_DISCONFREE	(1 << 0)
#define BT868_IMPLEMENTATION_FLAGS_CCIR601	(1 << 1)
#define BT868_IMPLEMENTATION_FLAGS_DOUBLEPLL	(1 << 2)
#define BT_IMPLEMENTATION_FLAGS_CONEXANT	(1 << 3)

/*  dwInOutScanFlags; */

#define BT868_INOUTSCAN_FLAGS_OUT_INTERLACED	(1 << 0)
#define BT868_INOUTSCAN_FLAGS_IN_INTERLACED	(1 << 1)

/*  ucTVOutMode

0 = NTSC,  1 = NTSC-50, 2 = PAL-BDGHI, 3 = PAL-N, 4 = PAL-Nc,
5 = PAL-M, 6 = PAL-M60, 7 = PAL-60 */

/* vsr = vertical scale ratio? vsrq = vsr quantized ? */

struct settings {
  double dCHARCLK;
  double dFXTAL; // 13500000;
  double dMinFrontPorchBC; // 13; //  { 0 - 20 }
  double dMinBackPorchBC;  // 3;   //  { 0 - 20 }
  double dHSYNOFFSET;      // 0;   //  { 0 - 511 }
  double dHSYNWIDTH;       // 2;    //  { 0 - 63 }
  double dVSYNOFFSET;      // 0;   //  { -1024 - 1023 } 
  double dVSYNWIDTH;	 // 1;    //  { 0 - 7 } 
  double dALO; // 0;   //  { 288, 243, 288, 243 }
  double dTLO; 	// 0;   //  { 312.5, 262.5, 312, 262 }
  double dATO; 	// 0;   //  { .000052, .00005265556, .000052, .00005265556 }
  double dTTO; 	// 0;   //  { .000064, .0000635556, .000064, .0000635556 }
  double dFSC; 	// 0;   FSC = Freq Sub Carrier
  double dHSYNCWIDTH; 	// .0000047;
  double dFRONTPORCH; 	// .0000015;
  double dHBURSTBEGIN;
  double dHBURSTEND; 	
  double dHBLANKO; 	
  double dHBLANKI; 	
  double dMSC; 	
  double dHXTAL; 	
  double dPHZINC; 	
  double dVBLANKO; 	
  double dVSRQ;
  double dMaxHsyncDrift; 
  double dMinFrontPorchIn;
  double dFrontPorchIn; 	
  double dFrontPorchOut; 	
  double dBackPorchIn; 	
  double dBackPorchOut; 
  double dFCLK; 	
  double dHACTIVEI; 	// 640;   //  { 640, 800 } 
  double dVACTIVEI; 	// 480;   //  { 480, 600 } 
  double dHOC; 		// .1379;   //  { 0 - .25 } 
  double dVOC; 		// .1358;   //  { 0 - .25 } 
  double dVSR; 		// 0;
  double dTLI; 		// 0;
  double dVLINESI; 	// 0;
  double dHCLKO; 	// 0;
  double dHCLKI; 	// 0;
  double dTPI; 		// 0;
  double dHFRACT; 	// 0;
  double dMFP; 		// 0;
  double dMBP; 		// 0;
  double dVACTIVEO; 	// 0;
  double dVBLANKI; 	// 0;
  double dPLLINT; 	// 0;
  double dPLLFRACT; 	// 0;
  double dPLLRATIO; 	// 0;
  double dSINX; 	// 0;
  double dV100; 	// 0;
  double dSYNCAMP; 	// 0;
  double dBSTAMP; 	// 0;
  double dMY; 	// 0;
  double dMCR; 	// 0;
  double dMCB; 	// 0;
  double dVSCALE; 	// 0;
  double dCLKRATIO; /* NEW */
  long ucPHASEOFF; 	// 0;
  long ucLUMADLY; 	// 0;
  long ucOUTMODE; 	// 0;
  long ucAutoConfigMode; 	// 0;
  long ucTVOutMode; 		// 0;
  long ucPixelInputMode; 	
  long ucCLPF; 		// 0;      
  long ucYLPF; 	// 3;      
  long ucFSELY; 	// 3;     
  long ucFSELC; 	// 3;     
  long ucYCORING; 	// 0;   
  long ucCCORING; 	// 0;   
  long ucYATTENUATE; 	// 0;
  long ucCATTENUATE; 	// 0;
  long ucDIS_YFLPF; 	// 0;
  long ucDIS_GMSHY; 	// 1;
  long ucDIS_GMUSHY; 	// 1;
  long ucDIS_GMSHC; 	// 1;
  long ucDIS_GMUSHC; 	// 1;
  long ucOUTMUXA; 	// 0;
  long ucOUTMUXB; 	// 2;
  long ucOUTMUXC; 	// 1;
  long ucOUTMUXD; 	// 0;
  unsigned long dwRegisterFlags;
  unsigned long dwImplementationFlags; 	
  unsigned long dwInOutScanFlags; 	
  unsigned long dwFilterFlags; 	// 0;
};

typedef struct settings * PENCODER_SETTINGS;

inline double min (double a, double b)
{
  if (a < b) return a; else return b;
}

inline double max (double a, double b)
{
  if (a > b) return a; else return b;
}

void  Bt868EncoderInitializeDefault( PENCODER_SETTINGS pS )
{
  pS->dCHARCLK = 1; /* FIXME TEST 8; */ /*  { 1, 8, 9 } */
  pS->dFXTAL = 13500000;   	/*  { any }*/
  pS->dMinFrontPorchBC = 13;   	/*  { 0 - 20 }*/
  pS->dMinBackPorchBC = 3;   	/*  { 0 - 20 }*/
  pS->dHSYNOFFSET = 0;   	/*  { 0 - 511 }*/
  pS->dHSYNWIDTH = 2;   	/*  { 0 - 63 }*/
  pS->dVSYNOFFSET = 0;   	/*  { -1024 - 1023 } */
  pS->dVSYNWIDTH = 1;   	/*  { 0 - 7 } */
  pS->dALO = 0;   	/*  { 288, 243, 288, 243 }*/
  pS->dTLO = 0;   	/*  { 312.5, 262.5, 312, 262 }*/
  pS->dATO = 0;   	/*  { .000052, .00005265556, .000052, .00005265556 }*/
  pS->dTTO = 0;   	/*  { .000064, .0000635556, .000064, .0000635556 }*/
  pS->dFSC = 0;
  pS->dHSYNCWIDTH = .0000047;
  pS->dFRONTPORCH = .0000015;
  pS->dHBURSTBEGIN = 0;
  pS->dHBURSTEND = 0;
  pS->dHBLANKO = 0;
  pS->dHBLANKI = 0;
  pS->dMSC = 0;
  pS->dHXTAL = 0;
  pS->dPHZINC = 0;
  pS->dVBLANKO = 0;
  pS->dVSRQ = 0;
  pS->dCLKRATIO = 1.0;
  pS->dMaxHsyncDrift = 0;
  pS->dMinFrontPorchIn = 0;
  pS->dFrontPorchIn = 0;
  pS->dFrontPorchOut = 0;
  pS->dBackPorchIn = 0;
  pS->dBackPorchOut = 0;
  pS->dFCLK = 0;
  pS->dHACTIVEI = 640; 	/*  { 640, 800 } */
  pS->dVACTIVEI = 480; 	/*  { 480, 600 } */
  pS->dHOC = .1379;   	/*  { 0 - .25 } */
  pS->dVOC = .1358;   	/*  { 0 - .25 } */
  pS->dVSR = 0;
  pS->dTLI = 0;
  pS->dVLINESI = 0;
  pS->dHCLKO = 0;
  pS->dHCLKI = 0;
  pS->dTPI = 0;
  pS->dHFRACT = 0;
  pS->dMFP = 0;
  pS->dMBP = 0;
  pS->dVACTIVEO = 0;
  pS->dVBLANKI = 0;
  pS->dPLLINT = 0;
  pS->dPLLFRACT = 0;
  pS->dPLLRATIO = 0;
  pS->dSINX = 0;
  pS->dV100 = 0;
  pS->dSYNCAMP = 0;
  pS->dBSTAMP = 0;
  pS->dMY = 0;
  pS->dMCR = 0;
  pS->dMCB = 0;
  pS->dVSCALE = 0;
  pS->ucPHASEOFF = 0;
  pS->ucLUMADLY = 0;
  pS->ucOUTMODE = 0;
  pS->ucAutoConfigMode = 0;
  pS->ucTVOutMode = 0;
  pS->ucPixelInputMode = BT868_PIXEL_INPUT_MODE_24BIT_RGB_MUX;
  pS->ucCLPF = 0;      
  pS->ucYLPF = 3;      
  pS->ucFSELY = 3;     
  pS->ucFSELC = 3;     
  pS->ucYCORING = 0;   
  pS->ucCCORING = 0;   
  pS->ucYATTENUATE = 0;
  pS->ucCATTENUATE = 0;
  pS->ucDIS_YFLPF = 0;
  pS->ucDIS_GMSHY = 1;
  pS->ucDIS_GMUSHY = 1;
  pS->ucDIS_GMSHC = 1;
  pS->ucDIS_GMUSHC = 1;
  pS->ucOUTMUXA = 0;
  pS->ucOUTMUXB = 2;
  pS->ucOUTMUXC = 1;
  pS->ucOUTMUXD = 0;
  pS->dwRegisterFlags = BT868_REGISTER_FLAGS_ENOUT 
              | BT868_REGISTER_FLAGS_ENBLANKO 
              | BT868_REGISTER_FLAGS_VSYNCDUR 
              | BT868_REGISTER_FLAGS_SETUP;
  pS->dwImplementationFlags = BT868_IMPLEMENTATION_FLAGS_DISCONFREE;
  pS->dwInOutScanFlags = BT868_INOUTSCAN_FLAGS_OUT_INTERLACED;
  pS->dwFilterFlags = 0;
}


void Bt868SetAutoConfigMode (PENCODER_SETTINGS pS, int ucMode)
{
  pS->ucAutoConfigMode = ucMode;
  pS->dwInOutScanFlags |= BT868_INOUTSCAN_FLAGS_OUT_INTERLACED;
  pS->dwRegisterFlags &= ~BT868_REGISTER_FLAGS_VBLANKDLY;
  pS->dwRegisterFlags &= ~BT868_REGISTER_FLAGS_ENXCLK;
  pS->dwRegisterFlags &= ~BT868_REGISTER_FLAGS_BYPLL;
  pS->dwRegisterFlags &= ~BT868_REGISTER_FLAGS_ECLIP;
  pS->dwRegisterFlags &= ~BT868_REGISTER_FLAGS_SCRESET;
  pS->dwRegisterFlags &= ~BT868_REGISTER_FLAGS_NIOUT;
  pS->ucCLPF = 3;
  pS->ucCLPF = 0;

  if( ( ucMode == 0 ) || ( ucMode == 2 ) || ( ucMode == 4 ) || ( ucMode == 6 ) )
  {
    //  NTSC
    pS->ucTVOutMode = 0;
    pS->dwRegisterFlags &= ~BT868_REGISTER_FLAGS_PAL;
    pS->dwRegisterFlags |= BT868_REGISTER_FLAGS_VSYNCDUR;
    pS->dwRegisterFlags &= ~BT868_REGISTER_FLAGS_625LINE;
    pS->dwRegisterFlags |= BT868_REGISTER_FLAGS_SETUP;
    if( ( ucMode == 0 ) || ( ucMode == 4 ) )
    {
      //  640x480
      pS->dHACTIVEI = 640;
      pS->dVACTIVEI = 480;
      pS->dHOC = .1379;
      pS->dVOC = .1358;
    }
    else
    {
      //  800x600
      pS->dHACTIVEI = 800;
      pS->dVACTIVEI = 600;
      pS->dHOC = .2162;
      pS->dVOC = .1152;
    }
  }
  else
  {
    //  PAL
    pS->ucTVOutMode = 2;
    pS->dwRegisterFlags |= BT868_REGISTER_FLAGS_PAL;
    pS->dwRegisterFlags &= ~BT868_REGISTER_FLAGS_VSYNCDUR;
    pS->dwRegisterFlags |= BT868_REGISTER_FLAGS_625LINE;
    pS->dwRegisterFlags &= ~BT868_REGISTER_FLAGS_SETUP;
    if( ( ucMode == 1 ) || ( ucMode == 5 ) )
    {
      //  640x480
      pS->dHACTIVEI = 640;
      pS->dVACTIVEI = 480;
      pS->dHOC = .1656;
      pS->dVOC = .1667;
    }
    else
    {
      //  800x600
      pS->dHACTIVEI = 800;
      pS->dVACTIVEI = 600;
      pS->dHOC = .1453;
      pS->dVOC = .1319;
    }
  }
}

static void recalc_outfield (PENCODER_SETTINGS pS)
{
  if( ( pS->ucTVOutMode == 0 ) 
        || ( pS->ucTVOutMode == 1 ) 
        || ( pS->ucTVOutMode == 5 ) /* PAL_M */
        || ( pS->ucTVOutMode == 6 ) 
        || ( pS->ucTVOutMode == 7 ) )
  {
    pS->dALO = 243;
    pS->dATO = .00005265556;
    pS->dTTO = .00006355556;
    if( pS->dwInOutScanFlags & BT868_INOUTSCAN_FLAGS_OUT_INTERLACED )
    {
      pS->dTLO = 262.5;
    } else {
      pS->dTLO = 262;
    }
  } else {
    pS->dALO = 288;
    pS->dATO = .000052;
    pS->dTTO = .000064;
    if( pS->dwInOutScanFlags & BT868_INOUTSCAN_FLAGS_OUT_INTERLACED )
    {
      pS->dTLO = 312.5;
    } else {
      pS->dTLO = 312;
    }
  }
}

static void recalc_tvformat (PENCODER_SETTINGS pS)
{
  double  dACTIVEBEGIN = 0;
  double  dIMAGECENTER = 0;
  double  dHSYNCFREQ = 0;
  double  dBURSTSTART = 0;
  double  dBURSTEND = 0;

  switch( pS->ucTVOutMode )
  {
    case 0: /* NTSC */
      pS->dFSC = 3579545;
      dBURSTSTART = .0000053;
      dBURSTEND = .00000782;
      dHSYNCFREQ = .000063555;
      dACTIVEBEGIN = .0000094;
      dIMAGECENTER = .000035667;
      break;
    case 1: /* NTSC-50 Hz */
      pS->dFSC = 3579545;
      dBURSTSTART = .0000053;
      dBURSTEND = .00000782;
      dHSYNCFREQ = .000064;
      dACTIVEBEGIN = .0000094;
      dIMAGECENTER = .000035667;
      break;
    case 2: /* PAL-BDGHI */
      pS->dFSC = 4433618.75;
      dBURSTSTART = .0000056;
      dBURSTEND = .00000785;
      dHSYNCFREQ = .000064;
      dACTIVEBEGIN = .0000105;
      dIMAGECENTER = .000036407;
      break;
    case 3: /* PAL-N */
      pS->dFSC = 4433618.75;
      dBURSTSTART = .0000056;
      dBURSTEND = .00000785;
      dHSYNCFREQ = .000064;
      dACTIVEBEGIN = .0000094;
      dIMAGECENTER = .000035667;
      break;
    case 4: /* PAL-Nc */
      pS->dFSC = 3582056.25;
      dBURSTSTART = .0000056;
      dBURSTEND = .00000811;
      dHSYNCFREQ = .000064;
      dACTIVEBEGIN = .0000105;
      dIMAGECENTER = .000036407;
      break;
    case 5: /* PAL-M */
      pS->dFSC = 3575611.88;
      dBURSTSTART = .0000058;
      dBURSTEND = .00000832;
      dHSYNCFREQ = .000063555;
      dACTIVEBEGIN = .0000094;
      dIMAGECENTER = .000035667;
      break;
    case 6: /* PAL-M60 */
      pS->dFSC = 3575611.88;
      dBURSTSTART = .0000058;
      dBURSTEND = .00000832;
      dHSYNCFREQ = .000063555;
      dACTIVEBEGIN = .0000094;
      dIMAGECENTER = .000035667;
      break;
    case 7: /* PAL-60 */ 
      pS->dFSC = 4433619.49;
      dBURSTSTART = .0000056;
      dBURSTEND = .00000785;
      dHSYNCFREQ = .000063555;
      dACTIVEBEGIN = .0000094;
      dIMAGECENTER = .000035667;
      break;
  }
}

static void recalc_prepare (PENCODER_SETTINGS pS)
{
  pS->dVSR = pS->dVACTIVEI / ( pS->dALO * ( 1 - pS->dVOC ) );
  if( pS->dwImplementationFlags & BT868_IMPLEMENTATION_FLAGS_DISCONFREE )
  {
    pS->dVSR = (long)( ( pS->dVSR * pS->dTLO ) + 0.5 ) / pS->dTLO;
  }

  pS->dVSCALE = (long)( ( pS->dVSR - 1 ) * 4096 + 0.5 );
  pS->dTLI = pS->dVSR * pS->dTLO;
  pS->dVLINESI = (long)pS->dTLI;
  pS->dHCLKO = ( 2 * ( ( pS->dHACTIVEI / ( 1.0 - pS->dHOC ) ) 
                   * ( pS->dTTO / pS->dATO ) ) ) + 0.5;
  pS->dTPI = ( ( pS->dHCLKO * pS->dCLKRATIO / pS->dVSR ) * 256 + 0.5 ) / 256.0;
  pS->dHFRACT = (long)( ( pS->dTPI - (long)pS->dTPI ) * 256 );
  pS->dHCLKI = (long)pS->dTPI;
  pS->dMFP = max( 14.0, pS->dMinFrontPorchBC );
  pS->dMBP = max( 4.0, pS->dMinBackPorchBC );

#if 0
  if( pS->dTPI < ( pS->dHACTIVEI + pS->dMFP + pS->dMBP ) )
  {
    //  error
    DebugOut( "error: dTPI < ( dHACTIVEI + dMFP + dMBP )\n" );
        return;
  }
#endif
  pS->dVACTIVEO = ( ( pS->dVACTIVEI + 3 ) / pS->dVSR ) + 1;
}

static void recalc_find_oc (PENCODER_SETTINGS pS, 
  double dMinHOC, double dMaxHOC, double dMinVOC, double dMaxVOC,
  RecalcFindRes callback)
{
  double  dBestVOC = 0;
  double  dTempVOC = 0;
  double  dBestHOC = 0;
  double  dTempHOC = 0;
  double  dMinTLI = 0;
  double  dMaxTLI = 0;
  double  dTempTLI = 0;
  double  dBestTLI = 0;
  double  dMinHCLKO = 0;
  double  dMaxHCLKO = 0;
  double  dBestHCLKO = 0;
  double  dBestMetric = 1000;
  double  dTempVSR = 0;
  double  dBestVSR = 0;
  double  dMinTPI = 0;
  double  dTempTPI = 0;
  double  dBestTPI = 0;
  double  dTempCLKRATIO = 1; /* NEW */
  double  dBestCLKRATIO = 1; /* NEW */
  int     actCLKRATIO;
  int     maxCLKRATIO;
  double  dTempHCLKO = 0;
  double  dTempVACTIVEO = 0;
  double  dTempTCCI = 0;
  double  dDelta = 0;
  double  dMetric = 0;
  double  dAspect = 0;
  double  dMinHBT = 0;

  if (dMinHOC < 0) dMinHOC = 0;
  if (dMinVOC < 0) dMinVOC = 0;

  dMinTLI= (long)(pS->dVACTIVEI / ((1 - dMinVOC) * pS->dALO) * pS->dTLO);
  dMaxTLI = (long)(pS->dVACTIVEI / ((1 - dMaxVOC) * pS->dALO) * pS->dTLO);
  dMinHCLKO = (long) ((pS->dHACTIVEI * 2) / 
		      ((1 - dMinHOC) * (pS->dATO / pS->dTTO)));
  dMaxHCLKO = (long) ((pS->dHACTIVEI * 2) / 
		      ((1 - dMaxHOC) * (pS->dATO / pS->dTTO)));

  maxCLKRATIO = 0;
  if (pS->dwImplementationFlags & BT_IMPLEMENTATION_FLAGS_CONEXANT) {
    maxCLKRATIO = 1;
  }
  for (actCLKRATIO = 0; actCLKRATIO <= maxCLKRATIO; actCLKRATIO++)
  {
    dTempCLKRATIO = 1.0; 
    if (actCLKRATIO) dTempCLKRATIO = 3.0/2.0; 
    for(dTempTLI = dMinTLI; dTempTLI <= dMaxTLI; dTempTLI++)
    {                                   
      dTempVSR = dTempTLI / pS->dTLO;
      dTempVACTIVEO = (long)(((pS->dVACTIVEI * pS->dTLO) + 
			      (dTempTLI - 1)) / dTempTLI);
      dTempVOC = 1 - dTempVACTIVEO / pS->dALO;
      dMinTPI = pS->dHACTIVEI / (1 - (dMinHBT / (pS->dTTO * dTempVSR)));

      for(dTempHCLKO = dMinHCLKO; dTempHCLKO <= dMaxHCLKO; 
	   dTempHCLKO++)
      {                                 
	dTempTPI = (dTempHCLKO * dTempCLKRATIO) * (pS->dTLO / dTempTLI);
	dTempTCCI = dTempTPI / pS->dCHARCLK;

	if((dTempTCCI == (long)dTempTCCI) && (dTempTPI >= dMinTPI))
	{
	  int flag;

	  dTempHOC = 1 - ((pS->dHACTIVEI / (dTempHCLKO / 2)) / 
			  (pS->dATO / pS->dTTO));
	  dDelta = fabs(dTempHOC - pS->dHOC) + fabs(dTempVOC - pS->dVOC);
	  dMetric = ((dTempHOC - pS->dHOC) * (dTempHOC - pS->dHOC)) + 
		    ((dTempVOC - pS->dVOC) * (dTempVOC - pS->dVOC)) + 
		    (2 * dDelta * dDelta);
	  dAspect =  (1.0 - dTempHOC) / (1.0 - dTempVOC);
	  flag = 0;
	  if (((int) dTempTPI) % 8 == 0) flag |= BT_CALC_CHARCLK8;
	  if (((int) dTempTPI) % 9 == 0) flag |= BT_CALC_CHARCLK9;
	  if (actCLKRATIO) flag |= BT_CALC_RATIO32;
	  if (callback) callback (dTempHOC, dTempVOC, dMetric, dAspect, 
				  flag);
	  if(dMetric < dBestMetric)
	  {
	    dBestMetric = dMetric;
	    dBestHOC = dTempHOC;
	    dBestVOC = dTempVOC;
	    dBestVSR = dTempVSR;
	    dBestTLI = dTempTLI;
	    dBestTPI = dTempTPI;
	    dBestHCLKO = dTempHCLKO;
	    dBestCLKRATIO = dTempCLKRATIO;
	  }
	} /* valid solution */                             
      } /* dTempHCLKO loop */
    } /* dTempTLI loop */
  } /* CLKRATIO loop */

  if(dBestMetric == 1000)
  {
    //  error
#if 0
    DebugOut("error: dBestMetric == 1000");
#endif
    return;
  }

  pS->dVSR = dBestVSR;
  pS->dVSCALE = (long)((dBestVSR - 1) * 4096 + 0.5);
  pS->dVLINESI = (long)dBestTLI;
  pS->dHCLKO = (long)dBestHCLKO;
  pS->dHFRACT = 0;
  pS->dHCLKI = (long)dBestTPI;
  pS->dVACTIVEO = (long)((pS->dVACTIVEI + 3) / dBestVSR) + 1;
  pS->dHOC = dBestHOC;
  pS->dVOC = dBestVOC;
  pS->dCLKRATIO = dBestCLKRATIO;
}

void Bt868ReCalc(PENCODER_SETTINGS pS)
{
  double  dPI = 3.141593;
  double  dTotalHBlankO = 0;
  double  dTotalHBlankI = 0;
  double  dHeadRoom = 0;
  double  dMaxHR = 0;
  double  dFifoMargin = 0;
  double  dFifoSize;

  dFifoSize = 800;
  if (pS->dwImplementationFlags & BT_IMPLEMENTATION_FLAGS_CONEXANT) {
    dFifoSize = 1024;
  }
  recalc_outfield (pS);
  recalc_tvformat (pS); /* on pS->ucTVOutMode */
  recalc_prepare (pS); /* FIXME: shouldn't have to do full prepare ... */
  recalc_find_oc (pS, pS->dHOC - .05, pS->dHOC + .05, 
		      pS->dVOC - .05, pS->dVOC + .05, NULL);
  if( pS->dwImplementationFlags & BT868_IMPLEMENTATION_FLAGS_CCIR601)
  {
    pS->dHCLKO = 1716;
    pS->dHCLKI = pS->dHCLKO / 2;
    pS->dVACTIVEO = 240;
    pS->dHBLANKI = 10;
    pS->dHACTIVEI = 720;
    pS->dVLINESI = 262;
    pS->dVBLANKI = 19;
    pS->dVACTIVEI = 240;
    pS->dVSCALE = 0;
    pS->ucPixelInputMode = BT868_PIXEL_INPUT_MODE_16BIT_YCRCB_MUX;
    pS->dwRegisterFlags |= BT868_REGISTER_FLAGS_VBLANKDLY;
    pS->dwRegisterFlags |= BT868_REGISTER_FLAGS_ENXCLK;
    pS->dwRegisterFlags |= BT868_REGISTER_FLAGS_ENDOT;
    pS->dwRegisterFlags |= BT868_REGISTER_FLAGS_SLAVE;
    pS->dwRegisterFlags &= ~BT868_REGISTER_FLAGS_ENOUT;
    pS->dwRegisterFlags &= ~BT868_REGISTER_FLAGS_ENBLANKO;
    pS->dwFilterFlags |= BT868_FILTER_FLAGS_DISFFILT;
    pS->dHOC = 0;
    pS->dVOC = 0;
    pS->dwInOutScanFlags &= ~BT868_INOUTSCAN_FLAGS_IN_INTERLACED;
  }

  /* Calculate parameters relative to HCLKO, i.e. numbers represent fractions
     of a whole line. The HXTAL constant is F_sc / F_line, i.e.
     HXTAL = F_xtal / F_line. */
  switch( pS->ucTVOutMode )
  {
    case 0: /* NTSC  */
    case 1: /* NTSC-50 ?? */
      pS->dVBLANKO = (long)( 140 - ( pS->dVACTIVEO / 2.0 ) + 0.5 );
      pS->dHBURSTBEGIN = 2 * (long)( ( 0.0416958041958 * pS->dHCLKO ) + 0.5 );
      pS->dHBURSTEND = 2 * (long)( ( 0.06270104895105 * pS->dHCLKO ) + 0.5) - 128;
      pS->dHSYNCWIDTH = 2 * (long)( ( 0.03697552447552 * pS->dHCLKO ) + 0.5 );
      pS->dHBLANKO = 2 * (long)( ( 0.280597027972 * pS->dHCLKO ) + 0.5 ) - (long)( pS->dHACTIVEI ) + 15;
      pS->dMSC = (long)( 0x80000000 * ( 455.0 / pS->dHCLKO ) + 0.5 ) + pS->dPHZINC;
      pS->dHXTAL = ( pS->dFXTAL * 227.5 ) / pS->dFSC;
      break;
    case 3: /* PAL-N */
      pS->dVBLANKO = (long)( 167 - ( pS->dVACTIVEO / 2.0 ) + 0.5 );
      pS->dHBURSTBEGIN = (long)( ( 0.0875 * pS->dHCLKO ) + 0.5 );
      pS->dHBURSTEND = (long)( ( 0.12265625 * pS->dHCLKO ) + 0.5 ) - 128;
      pS->dHSYNCWIDTH = (long)( ( 0.0734375 * pS->dHCLKO ) + 0.5 );
      pS->dHBLANKO = 2 * (long)( ( 0.2786484375 * pS->dHCLKO ) + 0.5 ) - (long)( pS->dHACTIVEI ) + 15;
      pS->dMSC = (long)( ( ( 567.5032 / pS->dHCLKO ) * 0x80000000 ) + 0.5 ) + pS->dPHZINC;
      pS->dHXTAL = ( pS->dFXTAL * 283.7516 ) / pS->dFSC;
      break;
    case 4: /* PAL-Nc */
      pS->dVBLANKO = (long)( 167 - ( pS->dVACTIVEO / 2.0 ) + 0.5 );
      pS->dHBURSTBEGIN = 2 * (long)( ( 0.04375 * pS->dHCLKO ) + 0.5 );
      pS->dHBURSTEND = 2 * (long)( ( 0.063359375 * pS->dHCLKO ) + 0.5 ) - 128;
      pS->dHSYNCWIDTH = 2 * (long)( ( 0.03671875 * pS->dHCLKO ) + 0.5 );
      pS->dHBLANKO = 2 * (long)( ( 0.2844296875 * pS->dHCLKO ) + 0.5 ) - (long)( pS->dHACTIVEI ) + 15;
      pS->dMSC = (long)( ( ( 458.5032 / pS->dHCLKO ) * 0x80000000 ) + 0.5 ) + pS->dPHZINC;
      pS->dHXTAL = ( pS->dFXTAL * 229.2516 ) / pS->dFSC;
      break;
    case 5: /* PAL-M */
      pS->dVBLANKO = (long)( 140 - ( pS->dVACTIVEO / 2.0 ) + 0.5);
      pS->dHBURSTBEGIN = (long)( ( 0.09125874125874 * pS->dHCLKO ) + 0.5 );
      pS->dHBURSTEND = 2 * (long)( ( 0.06545454545455 * pS->dHCLKO ) + 0.5 ) - 128;
      pS->dHSYNCWIDTH = 2 * (long)( ( 0.03697552447552 * pS->dHCLKO ) + 0.5 );
      pS->dHBLANKO = (long)( ( 0.5611940559441 * pS->dHCLKO ) + 0.5 ) - (long)( pS->dHACTIVEI ) + 15;
      pS->dMSC = (long)( 0x80000000 * ( 454.5 / pS->dHCLKO ) + 0.5 ) + pS->dPHZINC;
      pS->dHXTAL = ( pS->dFXTAL * 227.25 ) / pS->dFSC;
      break;
    case 7: /* PAL-60 ? */
      pS->dVBLANKO = (long)( 140 - ( pS->dVACTIVEO / 2.0 ) + 0.5 );
      pS->dHBURSTBEGIN = 2 * (long)( 151.2 * ( pS->dHCLKO / 3432 ) + 0.5 );
      pS->dHBURSTEND = 2 * (long)( ( 211.95 * ( pS->dHCLKO / 3432 ) ) + 0.5 ) - 128;
      pS->dHSYNCWIDTH = 2 * (long)( 126.9 * ( pS->dHCLKO / 3432 ) + 0.5 );
      pS->dHBLANKO = 2 * (long)( ( .280597027972 * pS->dHCLKO ) + 0.5 ) - (long)( pS->dHACTIVEI ) + 15;
      pS->dMSC = (long)( 4294967296.0 * ( ( 2216809375.0 / 7867132.0 ) / 
                         pS->dHCLKO ) + 0.5) + pS->dPHZINC;
      //pS->dMSC = (long)( 0x80000000 * ( 563.5622676726 / pS->dHCLKO ) + 0.5) + pS->dPHZINC;
      //pS->dHXTAL = ( pS->dFXTAL * 281.7811338363 ) / pS->dFSC;
      pS->dHXTAL = ( pS->dFXTAL * ( 2216809375.0 / 7867132.0 ) ) / ( pS->dFSC );
      break;
    default: /* PAL-BDGHI, PAL-M60 ? */
      //  625-line Systems
      pS->dVBLANKO = (long)( 167 - ( pS->dVACTIVEO / 2.0 ) + 0.5 );
      pS->dHBURSTBEGIN = 2 * (long)( ( 0.04375 * pS->dHCLKO ) + 0.5 );
      pS->dHBURSTEND = 2 * (long)( ( 0.061328125 * pS->dHCLKO ) + 0.5 ) - 128;
      pS->dHSYNCWIDTH = 2 * (long)( ( 0.03671875 * pS->dHCLKO ) + 0.5 );
      pS->dHBLANKO = 2 * (long)( ( 0.2844296875 * pS->dHCLKO ) + 0.5 ) - (long)( pS->dHACTIVEI ) + 15;
      pS->dMSC = (long)( ( 567.5032 / pS->dHCLKO ) * 0x80000000 + 0.5 ) + pS->dPHZINC;
      pS->dHXTAL = ( pS->dFXTAL * 283.7516 ) / pS->dFSC;
  }

  // Vertical input blanking
  pS->dVSRQ = ( (long)( pS->dVSR * 4096.0 + .5 ) ) / 4096.0;

  if( pS->dVSRQ < pS->dVSR )
  {
    // These calculations are in units of dHCLKO 
    pS->dMaxHsyncDrift = ( pS->dVSRQ - pS->dVSR ) * pS->dTLO / pS->dVSR * pS->dHCLKO;
    pS->dMinFrontPorchIn = pS->dMFP / ( pS->dHCLKI * pS->dVSR ) * pS->dHCLKO;
    pS->dFrontPorchOut = pS->dHCLKO - pS->dHBLANKO - pS->dHACTIVEI * 2;
    dFifoMargin = ( dFifoSize - pS->dHACTIVEI ) * 2;

    // Check for fifo overflow 
    if( pS->dFrontPorchOut + dFifoMargin < -pS->dMaxHsyncDrift + pS->dMinFrontPorchIn )
    {
      dTotalHBlankO = pS->dHCLKO - pS->dHACTIVEI * 2;
      dTotalHBlankI = ( pS->dHCLKI - pS->dHACTIVEI ) / pS->dHCLKI / pS->dVSR * pS->dHCLKO;

      // Try forcing the Hsync drift the opposite direction 
      pS->dMaxHsyncDrift = ( pS->dVSRQ + 1.0 / 4096 - pS->dVSR ) * pS->dTLO / pS->dVSR * pS->dHCLKO;

      // Check that fifo overflow and underflow can be avoided 
      if( dTotalHBlankO + dFifoMargin >= dTotalHBlankI + pS->dMaxHsyncDrift )
      {
        pS->dVSRQ = pS->dVSRQ + 1.0 / 4096;
        pS->dVSCALE = (long)( ( pS->dVSRQ - 1 ) * 4096 );
      }
      // NOTE: If fifo overflow and underflow can't be avoided,  
      //       alternative overscan compensation ratios should   
      //       be selected and all calculations repeated.  If    
      //       that is not feasible, the calculations for        
      //       H_BLANKI below will delay the overflow or under-  
      //       flow as much as possible, to minimize the visible 
      //       artifacts.                                        
    }
  }

  pS->dVBLANKI = (long)( ( pS->dVBLANKO - 1 ) * pS->dVSRQ );

  if( pS->dwImplementationFlags & BT868_IMPLEMENTATION_FLAGS_CCIR601 )
  {
    pS->dVSCALE = 0;
    pS->dVBLANKI = 19;
  }

  // Horizontal input blanking

  // These calculations are in units of dHCLKI 
  dTotalHBlankI = pS->dHCLKI - pS->dHACTIVEI;
  pS->dFrontPorchIn = max( pS->dMFP, min( dTotalHBlankI / 8.0, dTotalHBlankI - pS->dMBP ) );
  pS->dBackPorchIn = dTotalHBlankI - pS->dFrontPorchIn;
  pS->dMaxHsyncDrift = ( pS->dVSRQ - pS->dVSR ) * pS->dTLO * pS->dHCLKI;
  dTotalHBlankO = ( pS->dHCLKO - pS->dHACTIVEI * 2.0 ) / pS->dHCLKO * pS->dVSR * pS->dHCLKI;
  pS->dBackPorchOut = pS->dHBLANKO / pS->dHCLKO * pS->dVSR * pS->dHCLKI;
  pS->dFrontPorchOut = dTotalHBlankO - pS->dBackPorchOut;
  dFifoMargin = ( dFifoSize - pS->dHACTIVEI ) * 2.0 / pS->dHCLKO * pS->dVSR * pS->dHCLKI;
  // This may be excessive, but is adjusted by the code. 
  dHeadRoom = 32.0; 
  
  // Check that fifo overflow and underflow can be avoided 
  if( ( dTotalHBlankO + dFifoMargin ) >= ( dTotalHBlankI + abs( pS->dMaxHsyncDrift ) ) )
  {
    dMaxHR = ( dTotalHBlankO + dFifoMargin ) - ( dTotalHBlankI - abs( pS->dMaxHsyncDrift ) );
    if( dMaxHR < ( dHeadRoom * 2.0 ) )
    {
      dHeadRoom = (long)( dMaxHR / 2.0);
    }

    // Check for overflow 
    if( ( ( pS->dFrontPorchOut + dFifoMargin ) - dHeadRoom ) < ( pS->dFrontPorchIn - min( pS->dMaxHsyncDrift, 0 ) ) )
    {
      pS->dFrontPorchIn = max( pS->dMFP, ( pS->dFrontPorchOut + dFifoMargin + min( pS->dMaxHsyncDrift, 0 ) - dHeadRoom ) );
      pS->dBackPorchIn = dTotalHBlankI - pS->dFrontPorchIn;
    }

    // Check for underflow 
    if( pS->dBackPorchOut - dHeadRoom < pS->dBackPorchIn + max( pS->dMaxHsyncDrift, 0 ) )
    {
      pS->dBackPorchIn = max( pS->dMBP, ( pS->dBackPorchOut - max( pS->dMaxHsyncDrift, 0 ) - dHeadRoom ) );
      pS->dFrontPorchIn = dTotalHBlankI - pS->dBackPorchIn;
    }
  }
  else if( pS->dMaxHsyncDrift < 0 )
  {
    // Delay the overflow as long as possible 
    pS->dBackPorchIn = min( ( pS->dBackPorchOut - 1 ), ( dTotalHBlankI - pS->dMFP ) );
    pS->dFrontPorchIn = dTotalHBlankI - pS->dBackPorchIn;
  }
  else
  {
    // Delay the underflow as long as possible 
    pS->dFrontPorchIn = min( ( pS->dFrontPorchOut + dFifoMargin - 1 ), ( dTotalHBlankI - pS->dMBP ) );
    pS->dBackPorchIn = dTotalHBlankI - pS->dFrontPorchIn;
  }

  pS->dHBLANKI = (long)( pS->dBackPorchIn );
  if( pS->dwImplementationFlags & BT868_IMPLEMENTATION_FLAGS_CCIR601 )
  {
    pS->dHBLANKI = 10;
  }


  pS->dPLLRATIO = (long)(6.0 * (pS->dHCLKO / pS->dHXTAL) * pS->dCLKRATIO *
			 0x10000 + 0.5);
  
  if( pS->dwImplementationFlags & BT868_IMPLEMENTATION_FLAGS_DOUBLEPLL )
  {
    // Double PLL frequency
    pS->dPLLRATIO *= 2;
  }

  pS->dFCLK = ( pS->dFXTAL * pS->dPLLRATIO / 6.0 ) / 0x10000;
  pS->dPLLINT = (long)( pS->dPLLRATIO / 0x10000 );
  pS->dPLLFRACT = (long)( pS->dPLLRATIO - pS->dPLLINT * 0x10000 );
  pS->dSINX = sin( dPI * pS->dFSC / pS->dFCLK ) / ( dPI * pS->dFSC / pS->dFCLK );

  if( pS->dwRegisterFlags & BT868_REGISTER_FLAGS_MODE2X )
  {
    // Internal clock divided by 2
    pS->dSINX = sin( dPI * pS->dFSC / ( pS->dFCLK / 2 ) ) / ( dPI * pS->dFSC / ( pS->dFCLK / 2 ) );
  }

  if( ( pS->ucTVOutMode == 0 ) || ( pS->ucTVOutMode == 1 ) )
  {
    pS->dV100 = 676.203;
    pS->dSYNCAMP = 229;
    pS->dBSTAMP = (long)( ( 114.5 / pS->dSINX ) + 0.5 );
  }
  else
  {
    pS->dV100 = 716.1;
    pS->dSYNCAMP = 240;
    pS->dBSTAMP = (long)( ( 84.84 / pS->dSINX ) + 0.5 );
  }

  if (pS->ucPixelInputMode <= 3) 
  {
    // RBG modes 
    pS->dMY = (long)( ( pS->dV100 / 5.099501953125 ) + 0.5 );
    pS->dMCR = (long)( ( pS->dV100 / ( 5.746659386525 * pS->dSINX ) ) + 0.5 );
    pS->dMCB = (long)( ( pS->dV100 / ( 10.22275919266 * pS->dSINX ) ) + 0.5 );
  }
  else                  
  {
    // YCRCB modes 
    pS->dMY = (long)( ( pS->dV100 / 4.379572265625 )  + 0.5 );
    pS->dMCR = (long)( ( pS->dV100 * ( 0.2767455917354 / pS->dSINX ) ) + 0.5 );
    pS->dMCB = (long)( ( pS->dV100 * ( 0.1966701565097 / pS->dSINX ) ) + 0.5 );

    if( pS->dwRegisterFlags & BT868_REGISTER_FLAGS_ECBAR )
    {
      // makes correction for ROM values
      pS->dMCR /= 2;
      pS->dMCB /= 2;
    }
  }
}

void recalc_system (TVSystem system, PENCODER_SETTINGS pS, TVBtRegs *b)
{
  pS->dwInOutScanFlags |= BT868_INOUTSCAN_FLAGS_OUT_INTERLACED;
  pS->ucCLPF = 3;
  pS->ucCLPF = 0;
  switch (system) 
    {
    case TV_SYSTEM_NTSC:
      pS->ucTVOutMode = 0; /* NTSC */
      pS->dwRegisterFlags |= BT868_REGISTER_FLAGS_VSYNCDUR;
      pS->dwRegisterFlags &= ~BT868_REGISTER_FLAGS_625LINE;
      pS->dwRegisterFlags |= BT868_REGISTER_FLAGS_SETUP;
      pS->dwRegisterFlags &= ~BT868_REGISTER_FLAGS_PAL;
      if (b) b->flags1 = BT_FLAG1_NTSC;
      break;
    case TV_SYSTEM_NTSC_J:
      pS->ucTVOutMode = 0; /* NTSC */
      pS->dwRegisterFlags |= BT868_REGISTER_FLAGS_VSYNCDUR;
      pS->dwRegisterFlags &= ~BT868_REGISTER_FLAGS_625LINE;
      pS->dwRegisterFlags &= ~BT868_REGISTER_FLAGS_SETUP;
      pS->dwRegisterFlags &= ~BT868_REGISTER_FLAGS_PAL;
      if (b) b->flags1 = BT_FLAG1_NTSC_J;
      break;
    case TV_SYSTEM_PAL: 
      pS->ucTVOutMode = 2; /* PAL-BDGHI */
      pS->dwRegisterFlags &= ~BT868_REGISTER_FLAGS_VSYNCDUR;
      pS->dwRegisterFlags |= BT868_REGISTER_FLAGS_625LINE;
      pS->dwRegisterFlags &= ~BT868_REGISTER_FLAGS_SETUP;
      pS->dwRegisterFlags |= BT868_REGISTER_FLAGS_PAL;
      if (b) b->flags1 = BT_FLAG1_PAL_BDGHI;
      break;
    case TV_SYSTEM_PAL_N:
      pS->ucTVOutMode = 3; /* PAL-N */
      pS->dwRegisterFlags |= BT868_REGISTER_FLAGS_VSYNCDUR;
      pS->dwRegisterFlags |= BT868_REGISTER_FLAGS_625LINE;
      pS->dwRegisterFlags |= BT868_REGISTER_FLAGS_SETUP;
      pS->dwRegisterFlags |= BT868_REGISTER_FLAGS_PAL;
      if (b) b->flags1 = BT_FLAG1_PAL_N;
      break;
    case TV_SYSTEM_PAL_NC:
      pS->ucTVOutMode = 4; /* PAL-Nc */ 
      pS->dwRegisterFlags &= ~BT868_REGISTER_FLAGS_VSYNCDUR;
      pS->dwRegisterFlags |= BT868_REGISTER_FLAGS_625LINE;
      pS->dwRegisterFlags &= ~BT868_REGISTER_FLAGS_SETUP;
      pS->dwRegisterFlags |= BT868_REGISTER_FLAGS_PAL;
      if (b) b->flags1 = BT_FLAG1_PAL_BDGHI;
      break;
    case TV_SYSTEM_PAL_M:
      pS->ucTVOutMode = 5; /* PAL-M */
      pS->dwRegisterFlags |= BT868_REGISTER_FLAGS_VSYNCDUR;
      pS->dwRegisterFlags &= ~BT868_REGISTER_FLAGS_625LINE;
      pS->dwRegisterFlags |= BT868_REGISTER_FLAGS_SETUP;
      pS->dwRegisterFlags |= BT868_REGISTER_FLAGS_PAL;
      if (b) b->flags1 = BT_FLAG1_PAL_M;
      break;
    case TV_SYSTEM_PAL_60: 
      pS->ucTVOutMode = 7; /* PAL-60 */
      pS->dwRegisterFlags |= BT868_REGISTER_FLAGS_VSYNCDUR;
      pS->dwRegisterFlags &= ~BT868_REGISTER_FLAGS_625LINE;
      pS->dwRegisterFlags &= ~BT868_REGISTER_FLAGS_SETUP;
      pS->dwRegisterFlags |= BT868_REGISTER_FLAGS_PAL;
      if (b) b->flags1 = BT_FLAG1_PAL_60;
      break;
    case TV_SYSTEM_PAL_M60:
      pS->ucTVOutMode = 6; /* PAL-60 */
      /* Not documented ... */
      break; 
    default:
      break;
  }
}

void recalc_find (TVSystem system, int hres, int vres, 
  double hoc_min, double hoc_max, double voc_min, double voc_max,
  RecalcFindRes callback, int flags)
{
  struct settings s;

  Bt868EncoderInitializeDefault (&s);
  s.dwImplementationFlags |= flags;
  recalc_system (system, &s, NULL);
  s.dHOC = (hoc_min + hoc_max) / 2.0; /* FIXME */
  s.dVOC = (voc_min + voc_max) / 2.0;
  s.dHACTIVEI = hres & ~0x7;
  s.dVACTIVEI = vres;
  recalc_outfield (&s);
  recalc_tvformat (&s); /* on s.ucTVOutMode */
  recalc_prepare (&s); /* FIXME: shouldn't have to do full prepare ... */
  recalc_find_oc (&s, hoc_min, hoc_max, voc_min, voc_max, callback);
}

void recalc_bt_find (TVSystem system, int hres, int vres, 
  double hoc_min, double hoc_max, double voc_min, double voc_max,
  RecalcFindRes callback)
{
  recalc_find (system, hres, vres, hoc_min, hoc_max, voc_min, voc_max,
	       callback, 0);
}

void recalc_cx_find (TVSystem system, int hres, int vres, 
  double hoc_min, double hoc_max, double voc_min, double voc_max,
  RecalcFindRes callback)
{
  recalc_find (system, hres, vres, hoc_min, hoc_max, voc_min, voc_max,
	       callback, BT_IMPLEMENTATION_FLAGS_CONEXANT);
}

void calc_nv_btcx (int hres, int vres, TVRegs *r)
{
  /* This is just in approximation, and by no means accurate. */

  r->crtc.nv.HDisplay    = hres;
  r->crtc.nv.HTotal      = r->enc.bt.h_clki & ~0x7;
  /* Some attempts to get good sync values: */
  /* r->crtc.nv.HSyncStart  = 
    (r->crtc.nv.HTotal + r->crtc.nv.HDisplay) / 2 + 8; */
  r->crtc.nv.HSyncStart = 
    ((r->crtc.nv.HTotal - r->enc.bt.h_blanki) & ~0x7) + 48;
  /* 48 seems to work in many PAL cases, but it may be completely wrong. */

  /* Now try to sanitize at least the worst cases */
  if (r->crtc.nv.HSyncStart <= r->crtc.nv.HDisplay) 
    r->crtc.nv.HSyncStart = r->crtc.nv.HDisplay + 8;
  if (r->crtc.nv.HSyncStart >= r->crtc.nv.HTotal) 
    r->crtc.nv.HSyncStart = r->crtc.nv.HDisplay + 8;
  r->crtc.nv.HSyncEnd    = r->crtc.nv.HSyncStart + 24; /* wild guess */
  if (r->crtc.nv.HSyncEnd >= r->crtc.nv.HTotal) 
    r->crtc.nv.HSyncStart = r->crtc.nv.HTotal;

  r->crtc.nv.VDisplay    = vres;
  r->crtc.nv.VTotal      = r->enc.bt.v_linesi; 
  /* Some attempts to get good sync values: */
  /* r->crtc.nv.VSyncStart = 
     (r->crtc.nv.VTotal + r->crtc.nv.VDisplay) / 2 + 2; */
  r->crtc.nv.VSyncStart = r->crtc.nv.VTotal - r->enc.bt.v_blanki;
  if (r->crtc.nv.VSyncStart <= r->crtc.nv.VDisplay) 
    r->crtc.nv.VSyncStart = r->crtc.nv.VDisplay + 8;
  r->crtc.nv.VSyncEnd    = r->crtc.nv.VSyncStart + 2;

  r->crtc.nv.flags       = 0;
  r->devFlags = DEV_TELEVISION | DEV_MONITOR;
}

void recalc_bt_copy (PENCODER_SETTINGS s, TVBtRegs *r)
{
  r->hsynoffset   = (int) s->dHSYNOFFSET;
  r->vsynoffset   = (int) s->dVSYNOFFSET;
  r->hsynwidth    = (int) s->dHSYNWIDTH;
  r->vsynwidth    = (int) s->dVSYNWIDTH;
  r->h_clko       = (int) s->dHCLKO;
  r->h_active     = (int) s->dHACTIVEI;
  r->hsync_width  = (int) s->dHSYNCWIDTH;
  r->hburst_begin = (int) s->dHBURSTBEGIN;
  r->hburst_end   = (int) s->dHBURSTEND;
  r->h_blanko     = (int) s->dHBLANKO;
  r->v_blanko     = (int) s->dVBLANKO;
  r->v_activeo    = (int) s->dVACTIVEO;
  r->h_fract      = (int) s->dHFRACT;
  r->h_clki       = (int) s->dHCLKI;
  r->h_blanki     = (int) s->dHBLANKI;
  r->v_linesi     = (int) s->dVLINESI;
  r->v_blanki     = (int) s->dVBLANKI;
  r->v_activei    = (int) s->dVACTIVEI;
  r->v_scale      = (int) s->dVSCALE;
  r->pll_fract    = (int) s->dPLLFRACT;
  r->pll_int      = (int) s->dPLLINT;
  r->sync_amp     = (int) s->dSYNCAMP;
  r->bst_amp      = (int) s->dBSTAMP;
  r->mcr          = (int) s->dMCR;
  r->mcb          = (int) s->dMCB;
  r->my           = (int) s->dMY;
  r->msc          = (long) s->dMSC;
}

void recalc_bt_custom (TVSystem system, int hres, int vres, 
  double hoc, double voc, TVRegs *r)
{
  struct settings s;

  Bt868EncoderInitializeDefault (&s);
  recalc_system (system, &s, &r->enc.bt);
  s.dHOC = hoc;
  s.dVOC = voc;
  s.dHACTIVEI = hres & ~0x7;
  s.dVACTIVEI = vres;
  Bt868ReCalc (&s);
  recalc_bt_copy (&s, &r->enc.bt);
}

void recalc_cx_custom (TVSystem system, int hres, int vres, 
  double hoc, double voc, TVRegs *r)
{
  struct settings s;

  Bt868EncoderInitializeDefault (&s);
  s.dwImplementationFlags |= BT_IMPLEMENTATION_FLAGS_CONEXANT;
  recalc_system (system, &s, &r->enc.cx.bt);
  s.dHOC = hoc;
  s.dVOC = voc;
  s.dHACTIVEI = hres & ~0x7;
  s.dVACTIVEI = vres;
  Bt868ReCalc (&s);
  recalc_bt_copy (&s, &r->enc.cx.bt);
  r->enc.bt.flags1 |= CX_FLAG1_EXT;
  r->enc.cx.wsdat = 0;
  if (s.dCLKRATIO != 1.0) r->enc.cx.flags5 |= CX_FLAG5_PLL_32CLK;
  /* Remaining init will be done in data_init_cx */
}

/*

Mode     Total
         
         872 +185 1000  34.5  MHz  815  +43
Large    840 +209 1000  33.25 MHz  791  +49
Large    832 +217   "     "

Small800 816 +153  960  36    MHz  807   +9
Small800 784 +185   "

Small768 816 +241 1000  34    MHz  759  +57

         816 +241 1000  32.75 MHz

*/

