/*
 * libsysactivity
 * http://sourceforge.net/projects/libsysactivity/
 * Copyright (c) 2009-2011 Carlos Olmedo Escobar <carlos.olmedo.e@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <errno.h>
#include <pthread.h>
#include <sys/param.h>
#include <sys/sysctl.h>
#include <string.h>

#include "utils.h"

pthread_key_t buffer_key;
pthread_key_t buffer_size_key;
pthread_key_t number_disks_key;

static void create_keys();
void assign(struct sa_disk* __restrict__ dst, const struct diskstats* __restrict__ src) SA_NONNULL;

int sa_open_disk() {
	pthread_once_t key_once = PTHREAD_ONCE_INIT;
	if (pthread_once(&key_once, create_keys))
		return ENOSYS;

	return 0;
}

int sa_count_disks(uint16_t* number) {
	if (number == NULL)
		return EINVAL;

	int mib[] = { CTL_HW, HW_DISKCOUNT };
	int n_disks;
	size_t size = sizeof number;
	if (sysctl(mib, 2, &n_disks, &size, NULL, 0) < 0)
		return ENOSYS;

	*number = n_disks;
	return 0;
}

int sa_reset_disks() {
	int ret;
	uint16_t number;
	size_t size;
	struct diskstats* buffer = pthread_getspecific(buffer_key);
	size_t buffer_size = (size_t) pthread_getspecific(buffer_size_key);
	int mib[] = { CTL_HW, HW_DISKSTATS };

	do {
		ret = sa_count_disks(&number);
		if (ret != 0)
			return ret;

		size = number * sizeof(struct diskstats);
		if (buffer_size < size) {
			safe_realloc((void*) &buffer, &size);
			if (pthread_setspecific(buffer_key, buffer))
				return ENOSYS;
			buffer_size = size;
			if (pthread_setspecific(buffer_size_key, (void *) buffer_size))
				return ENOSYS;
			if (buffer == NULL)
				return ENOMEM;
		}

		errno = 0;
		if (sysctl(mib, 2, buffer, &buffer_size, NULL, 0) < 0 && errno != ENOMEM)
			return ENOSYS;
	} while (errno == ENOMEM);

	if (pthread_setspecific(number_disks_key, (void *) (int) number))
		return ENOSYS;

	return 0;
}

int sa_get_disks_ids(char* dst, uint16_t dst_size, uint16_t* written) {
	if (dst == NULL || dst_size == 0 || written == NULL)
		return EINVAL;

	struct diskstats* buffer = pthread_getspecific(buffer_key);
	size_t number_disks = (size_t) pthread_getspecific(number_disks_key);

	*written = 0;
	size_t i;
	for (i = 0; i < number_disks; ++i) {
		if (i == dst_size)
			return ENOMEM;

		strlcpy(&dst[i * SA_DISK_NAME], buffer[i].ds_name, DS_DISKNAMELEN);
		++(*written);
	}
	return 0;
}

int sa_get_disk(char* name, struct sa_disk* dst) {
	if (name == NULL || dst == NULL)
		return EINVAL;

	struct diskstats* buffer = pthread_getspecific(buffer_key);
	size_t number_disks = (size_t) pthread_getspecific(number_disks_key);

	size_t i;
	for (i = 0; i < number_disks; ++i) {
		if (strncmp(buffer[i].ds_name, name, DS_DISKNAMELEN) != 0)
			continue;

		assign(dst, &buffer[i]);
		return 0;
	}
	return ENODEV;
}

int sa_get_disks(struct sa_disk* dst, uint16_t dst_size, uint16_t* written) {
	if (dst == NULL || dst_size == 0 || written == NULL)
		return EINVAL;

	struct diskstats* buffer = pthread_getspecific(buffer_key);
	size_t number_disks = (size_t) pthread_getspecific(number_disks_key);

	*written = 0;
	size_t i;
	for (i = 0; i < number_disks; ++i) {
		if (i == dst_size)
			return ENOMEM;

		assign(&dst[i], &buffer[i]);
		++(*written);
	}
	return 0;
}

static void create_keys() {
	static short keys_created = 0;
	if (keys_created) {
		return;
	}

	pthread_key_create(&buffer_key, free_keys);
	pthread_key_create(&buffer_size_key, NULL);
	pthread_key_create(&number_disks_key, NULL);

	keys_created = 1;
}

void assign(struct sa_disk* __restrict__ dst, const struct diskstats* __restrict__ src) {
	strlcpy(dst->name, src->ds_name, DS_DISKNAMELEN);
	dst->reads = src->ds_rxfer;
	dst->writes = src->ds_wxfer;
	dst->bytes_read = src->ds_rbytes;
	dst->bytes_written = src->ds_wbytes;
}
