/*--------------------------------------------------------------------------*/
/* ALBERTA_UTIL:  tools for messages, memory allocation, parameters, etc.   */
/*                                                                          */
/* file: alberta_util_inline.h                                              */
/*                                                                          */
/* description: header for inline functions of the ALBERTA_UTIL package     */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Universitaet Bremen                                          */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERTA                       */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2003)                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/

#ifndef _ALBERTA_UTIL_INLINE_H_
#define _ALBERTA_UTIL_INLINE_H_

#define DNRM2_F77  dnrm2_
#define DAXPY_F77  daxpy_
#define DEXPY_F77  dexpy_
#define DMXPY_F77  dmxpy_
#define DCOPY_F77  dcopy_
#define DDOT_F77   ddot_
#define DSCAL_F77  dscal_
#define DSWAP_F77  dswap_
#define DXPAY_F77  dxpay_
#define DRANDN_F77 drandn_

extern double DNRM2_F77(int *n, const double *x, int *ix);
extern void DAXPY_F77(int *n, double *alpha, const double *x, int *ix,
                      double *y, int *iy);
extern void DEXPY_F77(int *n, const double *x, int *ix, double *y, int *iy);
extern void DMXPY_F77(int *n, const double *x, int *ix, double *y, int *iy);
extern void DCOPY_F77(int *n, const double *x, int *ix, double *y, int *iy);
extern double DDOT_F77(int *n, const double *x, int *ix, const double *y, 
		       int *iy);
extern void DSCAL_F77(int *n, double *alpha, double *x, int *ix);
extern void DSWAP_F77(int *n, double *x, int *ix, double *y, int *iy);
extern void DXPAY_F77(int *n, const double *x, int *ix, double *alpha,
                      double *y, int *iy);
extern void DRANDN_F77(int *n, double *x, int *ix); 

static inline double dnrm2(int n, const double *x, int ix)
{
  return DNRM2_F77(&n, x, &ix);
}

static inline void daxpy(int n, double alpha, const double *x, 
			 int ix, double *y, int iy)
{
  DAXPY_F77(&n, &alpha, x, &ix, y, &iy);
  return;
}

static inline void dexpy(int n, const double *x, int ix, double *y, int iy)
{
  DEXPY_F77(&n, x, &ix, y, &iy);
  return;
}

static inline void dmxpy(int n, const double *x, int ix, double *y, int iy)
{
  DMXPY_F77(&n, x, &ix, y, &iy);
  return;
}

static inline void dcopy(int n, const double *x, int ix, double *y, int iy)
{
  DCOPY_F77(&n, x, &ix, y, &iy);
  return;
}

static inline double ddot(int n, const double *x, int ix,
			  const double *y, int iy)
{
  return DDOT_F77(&n, x, &ix, y, &iy);
}

static inline void dscal(int n, double alpha, double *x, int ix)
{
  DSCAL_F77(&n, &alpha, x, &ix);
  return;
}

static inline void dswap(int n, double *x, int ix, double *y, int iy)
{
  DSWAP_F77(&n, x, &ix, y, &iy);
  return;
}

static inline void dxpay(int n, const double *x, int ix, double alpha,
			 double *y, int iy)
{
  DXPAY_F77(&n, x, &ix, &alpha, y, &iy);
  return;
}

static inline void drandn(int n, double *x, int ix)
{
  DRANDN_F77(&n, x, &ix);
  return;
}

static inline void dset(int n, double alpha, double *x, int ix)
{
  int i, nix = n*ix;
  for (i = 0; i < nix; i += ix)
    x[i] = alpha;
  return;
}

#endif  /* _ALBERTA_UTIL_INLINE_H_  */
