/***************************************************************************
                          chathistorymanager.h  -  chat logs API
                             -------------------
    begin                : Mon Jun 28 2010
    copyright            : (C) 2010 by Valerio Pilo
    email                : valerio@kmess.org
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef CHATHISTORYMANAGER_H
#define CHATHISTORYMANAGER_H

#include <QDomNode>
#include <QMap>
#include <QMutex>
#include <QStringList>


// Forward declarations
class ChatHistoryWriter;


// Custom types

/**
 * Conversation meta data
 *
 * This structure contains some metadata about a certain conversation,
 * more precisely about how the conversation is stored within the XML
 * logs.
 */
struct Conversation
{
  // The time when the conversation started
  quint64 timestamp;
  // Path where this conversation is stored
  QString filePath;
  // Initial position to fetch from the file
  quint64 startPosition;
  // Final position to fetch from the file
  quint64 endPosition;
};

// List of Conversation (indexed by timestamp)
typedef QMap<quint64,Conversation> ConversationList;



/**
 * Chat history management API
 *
 * @author Valerio Pilo
 */
class ChatHistoryManager
{


  public: // Public enumerations

    // Operations results
    enum Result
    {
      RESULT_OK
    , RESULT_INVALID_ACCOUNT
    , RESULT_INVALID_CONTACT
    , RESULT_NON_EXISTING_DIRECTORY
    , RESULT_UNREADABLE_LOGS
    , RESULT_INVALID_TIMESTAMP
    };


  public: // Public static methods

    // Get the account where to search for chat logs
    static QString account();
    // Retrieve the XML of a chat log header
    static QString chatHeader( const quint64 timestamp );
    // Retrieve a full chat log in form of XML
    static QString chatLog( const QList< quint64 > timestamps, bool insertHeaders = true, bool filtered = true );
    // Get the list of contacts for which logs are available
    static QStringList contactsList();
    // Retrieve a chat log writer ojcect
    static ChatHistoryWriter* getWriter( const QString& handle );
    // Get the contact handle we will search chat logs of
    static QString handle();
    // Retrieve the last chat had with a contact, in form of XML
    static QString lastChatLog( const QString& handle );
    // Add a new chat to the logs, or update an ongoing one.
    static void updateChatLog( const QString& handle, const Conversation& chatLogData );
    // Change the account where to search for chat logs
    static void setAccount( const QString newAccount );
    // Change the contact handle we will search chat logs of
    static void setHandle( const QString newHandle );
    // Get the status of the chat history retrieval service
    static Result result();
    // Retrieve all the dates in which conversations were recorded for a contact, and the name of the file they are in
    static ConversationList timestamps();


  private: // Private static methods

    // Parse a chat log file and retrieve from it a list of conversations
    static ConversationList parseLogFile( const QString& fileName );


  private: // Private static members

    // Mutex to ensure data integrity
    static QMutex mutex_;
    // Current account
    static QString account_;
    // Current contact handle
    static QString handle_;
    // Contacts for which logs are available
    static QStringList contactsList_;
    // Result of the last operation
    static Result result_;
    // List of conversation timestamps
    static ConversationList timestamps_;



};



#endif // CHATHISTORYMANAGER_H
