/***************************************************************************
 *   Copyright (C) 2004-2006 by Jim Campbell                               *
 *   ifpgui@gmail.com                                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "ifp_gui.h"
#include "ifpguiview.h"
#include "radiofreqdlg.h"
#include "playlist/playlistloader.h"

#include <QtGui>

#ifdef BSD4
#include "libgen.h"
#endif

#include "backend/iointerface.h"
#include "gui_user.h"

ifpGUIView::ifpGUIView( QMainWindow *parent )
        : QWidget((QWidget*) parent )
{
    int w_width, w_height, ifp_width, dir_width;
    QRect rect;
    QString settings_path;

    parent_window = parent;

    settings_path = QDir::homePath() + "/.ifpgui_rc";
    guiSettings = new Settings(settings_path);
    guiSettings->getWindowSize(&w_width, &w_height);

    //check widths for sane values
    rect = QApplication::desktop()->geometry();
    if ((w_width > 50) && (w_width <= rect.width()))
        parent->resize(w_width, w_height);

    // setup our layout manager to automatically add our widgets
    top_layout = new QGridLayout(this);

    split = new QSplitter(this);
    top_layout->addWidget(split);
    dirview = new fileManagerView(split, tr("Local File System"), TRUE, IO_INTF_TYPE_LOCAL);
    dirview->setMinimumSize(100, 100);
    dirview->setMaximumSize(2000, 2000);
    dirview->setDir(guiSettings->getDefaultFolder());
    dirview->show();

    ifpview = new fileManagerView(split, tr("iRiver File System"), FALSE, IO_INTF_TYPE_IFP);
    ifpview->setMinimumSize(100, 100);
    ifpview->setMaximumSize(2000, 2000);
    ifpview->show();

    createToolBars();

    // add the transfer toolbar at the bottom
    top_layout->addWidget(ifpTransferToolBar);

    //check widths for sane values
    QList<int> list;
    dir_width = guiSettings->getDirViewWidth();
    if ((dir_width > 0) && (dir_width < parent->width() - 50))
        list.append(dir_width);
    else
        list.append(parent->width() / 2);

    ifp_width = guiSettings->getIfpViewWidth();
    if ((ifp_width > 0) && (ifp_width < parent->width() - 50))
        list.append(ifp_width);
    else
        list.append(parent->width() / 2);

    split->setSizes(list);

    hideTransferProgress();
}

ifpGUIView::~ifpGUIView()
{
    int w_width, w_height, cnt;
    QString settings_path, destDir;

    w_width  = parent_window->width();
    w_height = parent_window->height();
    guiSettings->setWindowSize(w_width, w_height);
    destDir = dirview->currentDirectory();
    guiSettings->setDefaultFolder(destDir);

    cnt = 0;
    QList<int> list = split->sizes();
    QList<int>::Iterator it = list.begin();

    while( it != list.end() )
    {
        switch(cnt)
        {
        case 0:
            guiSettings->setDirViewWidth(*it);
            break;
        case 1:
            guiSettings->setIfpViewWidth(*it);
            break;
        }
        ++cnt;
        ++it;
    }

    settings_path = QDir::homePath() + "/.ifpgui_rc";
    guiSettings->saveSettings(settings_path);

    release_ifp_device();
}

void ifpGUIView::enableWindow(bool enable)
{
    dirview->setEnabled(enable);
    ifpview->setEnabled(enable);
    ((ifp_gui*)parent())->enableWindow(enable);
}

void ifpGUIView::createToolBars()
{
    //---------------------------------------------------------------------------
    // Progress of file transfer
    //---------------------------------------------------------------------------
    QFrame *frame = new QFrame();
    frame->setFrameStyle(QFrame::StyledPanel | QFrame::Raised);

    ifpTransferToolBar = new QToolBar(tr("Transfer Progress"), this);
    ifpTransferToolBar->setAllowedAreas (Qt::BottomToolBarArea);

    transferLayout = new QVBoxLayout();
    transferLayout->setSpacing(1);
    transferLayout->setMargin(1);
    ifpTransferToolBar->addWidget(frame);

    frame->setLayout(transferLayout);

    fileprogress = new QProgressBar();
    fileprogress->setMaximum(25);

    totalprogress = new QProgressBar();
    totalprogress->setMaximum(25);

    lblFilename = new QLabel(tr("FileName"));
    lblFilename->setAlignment(Qt::AlignVCenter | Qt::AlignLeft);
    lblFilename->setMinimumSize(400, 30);
    lblFilename->setMaximumSize(400, 30);

    btnCancel = new QPushButton(tr("Cancel"));
    btnCancel->setMaximumSize(200, 30);

    connect(btnCancel, SIGNAL(clicked(bool)),
            this, SLOT(btnCancel_clicked(bool)));

    transferLayout->setAlignment(Qt::AlignCenter);
    transferLayout->addWidget(lblFilename);
    transferLayout->setStretch(0, 50);
    transferLayout->addWidget(fileprogress);
    transferLayout->setStretch(1, 50);
    transferLayout->addWidget(totalprogress);
    transferLayout->setStretch(2, 50);
    transferLayout->addWidget(btnCancel, 0, Qt::AlignHCenter);

    cancel_progress = false;

    setFileProgressTarget(0);
    setFileProgress(0);
    setTotalProgressTarget(-1);
    setTotalProgress(0);

}

int ifpGUIView::updateFileAndStatus(void)
{
    int ret, tret;
    QString Message;
    TiFPreadFileTree t_ifpTree;

    ret = updateStatus();
    tret = t_ifpTree.iFPreadFileTree();
    if (tret != IO_INTF_OK)
    {
        Message = QObject::tr("Could not get the file list.  Check your battery.  Try replugging the USB cable.");
        showInfoMsgBox(QObject::tr("iRiver Upload File Status"), Message);
    }
    ifpview->fmRefreshFileItem();
    qApp->processEvents();

    return(ret);
}

int ifpGUIView::updateStatus(void)
{
    int ret;

    if (isProcessing())
        return(0);

    ret = -1;
    if (ifp_dh != NULL)
    {
        setCapacity(getCapacity(), getFreeSpace());
        ret = getBatteryStatus();
        if (ret < 0)
        {
            release_ifp_device();
            ifpview->setDir("/");
            ifpview->fmRefreshFileItem();
        }
    }
    return(ret);
}

void ifpGUIView::setBatteryStatus(int bStatus)
{
    emit signalSetBatteryStatus( bStatus );
}

void ifpGUIView::setCapacity(int capSize, int capFree)
{
    emit signalSetCapacity(capSize, capFree);
}

void ifpGUIView::uploadPlaylist()
{
    PlaylistLoader loader;
    QFile playlistfile;
    QFileInfo finfo;
    FileInfos fi;
    PlayList::iterator it;
    QString playlistname, destDir, Message, err, playlistDir;
    int ret, playlistcount, playlistsize, freespace;
    //ifp_progress_info pginfo;
    QList<FileInfos> file_info_list;

    if (ifp_dh == NULL)
    {
        showInfoMsgBox(tr("iFP GUI Upload Playlist - Error"), iFPNotConnectedMsg);
        return;
    }

    playlistDir = guiSettings->getPlaylistFolder();
    playlistname = "";
    playlistname = QFileDialog::getOpenFileName(this,
                                                tr("Choose a playlist"),
                                                playlistDir,
                                                tr("PlayList (*.m3u)"));
    if (playlistname == "")
        return;

    finfo.setFile(playlistname);
    playlistDir = finfo.path();
    guiSettings->setPlaylistFolder(playlistDir);
    playlistfile.setFileName(playlistname);
    loader.load(playlistfile, PlaylistLoader::M3U);
    destDir = ifpview->currentDirectory();

    playlistcount = loader.numFilesInPlaylist();
    playlistsize  = loader.sizeOfPlaylist();
    freespace     = getFreeSpace();

    if ((playlistcount <= 0) || (playlistsize <= 0))
    {
        Message = tr("There are no files in the playlist.\n");
        Message += tr("It is possible your playlist is not supported.\n");
        Message += tr("Make sure your playlist is in the #EXTM3U playlist format.\n");
        showInfoMsgBox(tr("iFP GUI Upload Playlist - Error"), Message);
        return;
    }
    else if (freespace < playlistsize)
    {
        Message = tr("There is not enough free space to upload the entire playlist.");
        showInfoMsgBox(tr("iFP GUI Upload Playlist - Error"), Message);
        return;
    }

    //----------------------------------------------------------------
    // make a list of all selected files then call exec_TransferUpload
    //----------------------------------------------------------------
    for(it = loader.list.begin(); it != loader.list.end(); ++it)
    {
        finfo.setFile((*it).filename());
        //pginfo.total_transfered += finfo.size();
        if (finfo.isFile())
        {
            fi = FileInfos((*it).filename(), IO_INTF_FILE, finfo.size(), NULL);
            file_info_list.append(fi);
        }
        else if (finfo.isDir())
        {
            fi = FileInfos(QString((*it).filename()), IO_INTF_DIR, 0, NULL);
            file_info_list.append(fi);
        }
    }
    ret = dirview->fmUploadFileList(destDir, &file_info_list);

    ifpview->setDir(destDir);
    ifpview->fmRefreshFileItem();
}

void ifpGUIView::unselectAllChildren(QTreeWidgetItem *parentItem)
{
    QTreeWidgetItem *current;
    int cnt;

    for(cnt = 0; cnt < parentItem->childCount(); cnt++)
    {
        current = parentItem->child(cnt);
        if(current != NULL)
        {
            current->setSelected(false);
            if (current->childCount() > 0)
                unselectAllChildren(current);
        }
    }
}

void ifpGUIView::uploadSelectedFile(void)
{
    QString destDir;

    if (ifp_dh == NULL)
    {
        showInfoMsgBox(tr("iFP GUI Upload Selected Files - Error"), iFPNotConnectedMsg);
        return;
    }

    // get the ifp viewer directory
    destDir = ifpview->currentDirectory();
    // upload selected files on local file system to destination
    dirview->fmTransferSelectedFiles(destDir);

    ifpview->setDir(destDir);
    ifpview->fmRefreshFileItem();
}

void ifpGUIView::downloadSelectedFile(void)
{
    QString destDir;

    if (ifp_dh == NULL)
    {
        showInfoMsgBox(tr("iFP GUI Download Selected Files - Error"), iFPNotConnectedMsg);
        return;
    }

    // get the ifp viewer directory
    destDir = dirview->currentDirectory();
    // upload selected files on local file system to destination
    ifpview->fmTransferSelectedFiles(destDir);

    dirview->setDir(destDir);
    dirview->fmRefreshFileItem();
}


void ifpGUIView::deleteSelectedFile(void)
{
    if (ifpview->hasFocus())
    {
        if (ifp_dh == NULL)
        {
            showInfoMsgBox(tr("iFP GUI Delete Selected Files - Error"), iFPNotConnectedMsg);
            return;
        }
        ifpview->fmDeleteFile();
    }
    else if (dirview->hasFocus())
    {
        dirview->fmDeleteFile();
    }
}

int ifpGUIView::getBatteryStatus()
{
    int ret;

    if (ifp_dh == NULL)
        return(-1);

    ret = ifpgui_ifp_battery(&ifp_dev);
    return(ret);
}

QString ifpGUIView::getFirmwareRev()
{
    char firm_str[20];
    int  firm_val;

    if (ifp_dh == NULL)
        return("");

    firm_val = ifpgui_ifp_firmware_version(&ifp_dev);
    sprintf(firm_str, "%x.%02x", firm_val/0x0100, firm_val%0x100);
    return(QString(firm_str));
}

int ifpGUIView::getCapacity()
{
    if (ifp_dh == NULL)
        return(-1);

    return(ifpgui_ifp_capacity(&ifp_dev));
}

int ifpGUIView::getFreeSpace()
{
    if (ifp_dh == NULL)
        return(-1);

    return(ifpgui_ifp_freespace(&ifp_dev));
}

QString ifpGUIView::getifpType()
{
    char model_str[20];

    if (ifp_dh == NULL)
        return("");

    ifpgui_ifp_model(&ifp_dev, model_str, 20);
    return(QString(model_str));
}

void ifpGUIView::formatIfpDevice()
{
    int mbret, tret;
    QString str;
    QString Message;
    TiFPreadFileTree t_ifpTree;
    TiFPformatDevice t_ifpFormat;

    if (ifp_dh == NULL)
    {
        showInfoMsgBox(tr("iFP GUI Format iRiver Device - Error"), iFPNotConnectedMsg);
        return;
    }

    str = tr("Format IFP Device.\n\nWARNING---\n\nAnswering 'Yes' will erase all data on this device!\n\n");

    QMessageBox mb(tr("Format IFP Device?"),
                   str,
                   QMessageBox::Question,
                   QMessageBox::No  | QMessageBox::Escape,
                   QMessageBox::Yes,
                   QMessageBox::NoButton);
    mbret = mb.exec();
    if (mbret == QMessageBox::No)
    {
        return;
    }
    str = tr("Please Wait while formatting IFP device.  This may take 15 to 30 seconds.");
    setFileName(str);
    showTransferProgress();


    enableWindow(false);

    setTotalProgressTarget(-1);
    setTotalProgress(0);
    setFileProgressTarget(0);
    setFileProgress(0);
    hideCancelBtn();

    qApp->processEvents();

    t_ifpFormat.iFPformatDevice();

    setFileProgressTarget(100);
    setFileProgress(100);
    qApp->processEvents();

    enableWindow(true);

    showInfoMsgBox(tr("Format Device Status"), tr("Format Complete."));

    showCancelBtn();
    hideTransferProgress();
    tret = t_ifpTree.iFPreadFileTree();
    if (tret != IO_INTF_OK)
    {
        Message = QObject::tr("Could not get the file list.  Check your battery.  Try replugging the USB cable.");
        showInfoMsgBox(QObject::tr("iRiver Upload File Status"), Message);
    }

    ifpview->setDir("/");
    ifpview->fmRefreshFileItem();
}

void ifpGUIView::UpdateFirmware()
{
    QString str;
    QString msg;
    QString filename;
    QByteArray qcfilename;

    if (ifp_dh == NULL)
    {
        showInfoMsgBox(tr("iFP GUI Update Firmware - Error"), iFPNotConnectedMsg);
        return;
    }

    str = tr("Update Firmware");
    msg  = tr("Update firmware?\n\n");
    msg += tr(" Make sure your battery is full.\n");
    msg += tr(" DO NOT disconnect your USB cable until this process is finished!");

    filename = QFileDialog::getOpenFileName(this,
                                            tr("Choose a Firmware HEX file"),
                                            QDir::homePath(),
                                            tr("IFP Firmware Files  (*.hex; *.HEX)"));
    if (filename != "")
    {
        str = tr("Please Wait while updating firmware.");

        if (filename.length() > IFP_MAXPATHLEN)
        {
            showInfoMsgBox(tr("Firmware Update Status"), tr("Path to firmware file is too long."));
            return;
        }
        qcfilename = filename.toUtf8();
        ifpgui_ifp_update_firmware(&ifp_dev, qcfilename, progress, NULL);
        usleep(10000000); // sleep 10 seconds

        msg  = tr("Update Done\n\n");
        msg += tr(" DO NOT touch the device until it has turned itself off!\n\n");
        msg += tr(" Afterwards, you may have to turn the device back on.");
        showInfoMsgBox(tr("Firmware Update Status"), msg);
    }

}

void ifpGUIView::setRadioFrequency()
{
    RadioFreqWindow *rfdialog;

    if (ifp_dh == NULL)
    {
        showInfoMsgBox(tr("iFP GUI Tuner Settings - Error"), iFPNotConnectedMsg);
        return;
    }

    rfdialog = new RadioFreqWindow();
    rfdialog->setModal(TRUE);
    rfdialog->exec();
    delete rfdialog;
}

bool ifpGUIView::isProcessing()
{
    bool processingDirview = dirview->isProcessing();
    bool processingIfpview = ifpview->isProcessing();

    return(processingDirview || processingIfpview);
}

void ifpGUIView::create_directory()
{
    if (ifpview->hasFocus())
    {
        ifpview->fmCreateDirectory();
    }
    else if (dirview->hasFocus())
    {
        dirview->fmCreateDirectory();
    }
}

void ifpGUIView::rename_file()
{
    if (ifpview->hasFocus())
    {
        ifpview->fmRenameFile();
    }
    else if (dirview->hasFocus())
    {
        dirview->fmRenameFile();
    }
}

void ifpGUIView::delete_file()
{
    if (ifpview->hasFocus())
    {
        ifpview->fmDeleteFile();
    }
    else if (dirview->hasFocus())
    {
        dirview->fmDeleteFile();
    }
}

void ifpGUIView::file_properties()
{
}

void ifpGUIView::align_horizontal()
{
    split->setOrientation(Qt::Horizontal);
}

void ifpGUIView::align_vertical()
{
    split->setOrientation(Qt::Vertical);
}


void ifpGUIView::btnCancel_clicked(bool checked)
{
    if (checked)
    {
        // satisfy compiler
    }

    cancel_progress = true;
    hideTransferProgress();
}

/*!
    Creates a string with ... like "Trollte..." or "...olltech", depending on the alignment.
    I took this from the base qt source code.
*/
static QString qEllipsisText(const QString &org, const QFontMetrics &fm, int width, int align)
{
    int ellWidth = fm.width("...");
    QString text = QString::fromLatin1("");
    int i = 0;
    int len = org.length();
    int offset = (align & Qt::AlignRight) ? (len-1) - i : i;

    while (i < len && fm.width(text + org[offset]) + ellWidth < width)
    {
        if (align & Qt::AlignRight)
            text.prepend(org[offset]);
        else
            text += org[offset];
        offset = (align & Qt::AlignRight) ? (len-1) - ++i : ++i;
    }
    if (text.isEmpty())
        text = (align & Qt::AlignRight) ? org.right(1) : text = org.left(1);
    if (align & Qt::AlignRight)
        text.prepend("...");
    else
        text += "...";
    return text;
}

//=============================================================================
// take a File Name and truncate it leaving the extension intact
//=============================================================================
static QString truncateFileName(const QString &org, const QFontMetrics &fm, int width)
{
    int pos, extlen;
    QString tmpFileName = QString::fromLatin1("");
    QString fileExt = QString::fromLatin1("");
    int extWidth;

    // first extract the file extension from the filename
    pos = org.lastIndexOf('.');
    if (pos > 0)
    {
        // file name has an extension
        extlen = org.length() - pos;
        fileExt = org.mid(pos, extlen);
        tmpFileName = org.mid(0, pos);
    }
    else
    {
        // file name does not have an extension
        extlen = 0;
        tmpFileName = org;
    }
    extWidth = fm.width(fileExt);
    tmpFileName = qEllipsisText(tmpFileName, fm, width - extWidth, Qt::AlignLeft) + fileExt;
    return(tmpFileName);
}

//=============================================================================
// take a file path and remove whole parts replacing them with "..."
//=============================================================================
static QString truncateFilePath(QString &fileStr)
{
    int pos;
    int cnt, middle;
    QStringList lst;
    QStringList::Iterator lst_it;
    QString truncFileName, tmpFileName, fileExt;

    pos = fileStr.lastIndexOf('\\');
    if (pos >= 0)
        lst = fileStr.split("\\");
    else
        lst = fileStr.split("/");

    // remove any "..." elements
    lst_it = lst.begin();
    while(lst_it != lst.end())
    {
        if (*lst_it == "...")
            lst_it = lst.erase(lst_it);
        else
            lst_it++;
    }

    middle = lst.size() / 2;

    // build truncated string
    if (pos >= 0)
        truncFileName = "\\";
    else
        truncFileName = "/";

    cnt = 1;
    lst_it = lst.begin();
    for ( ; lst_it != lst.end(); ++lst_it)
    {
        // replace middlemost element with a right side emphesis
        if (cnt != middle)
        {
            truncFileName += *lst_it;
            if (cnt != lst.size())
            {
                if (pos >= 0)
                    truncFileName += "\\";
                else
                    truncFileName += "/";
            }
        }
        else
        {
            truncFileName += "...";
            if (pos >= 0)
                truncFileName += "\\";
            else
                truncFileName += "/";
        }
        cnt++;
    }
    return(truncFileName);
}

void ifpGUIView::setFileName(const QString &fileName)
{
    int width, targetWidth;
    QString truncFileName, tmpFileName;

    if (fileName.isEmpty() || fileName.isNull() || (fileName == lastFileName))
        return;

    lblFilename->setText(fileName);
    return;


    tmpFileName = fileName;

    targetWidth = ifpTransferToolBar->width() - (transferLayout->margin() * 2);
    QFontMetrics fm = this->fontMetrics();
    width = fm.width(tmpFileName);
    if (targetWidth > 20)
    {
        do
        {
            truncFileName = truncateFilePath(tmpFileName);
            if (truncFileName == tmpFileName)
            {
                truncFileName = truncateFileName(tmpFileName, fm, targetWidth);
            }
            width = fm.width(truncFileName);
            tmpFileName = truncFileName;
        } while(width > targetWidth);
    }

    lblFilename->setText(truncFileName);
    lastFileName = truncFileName;
}

void ifpGUIView::setFileProgress(int progress)
{
    fileprogress->setValue(progress);
}

void ifpGUIView::setFileProgressTarget(int totalSteps)
{
    fileprogress->setMaximum(totalSteps);
}

void ifpGUIView::setTotalProgress(int progress)
{
    if (totalprogress->isHidden() == FALSE)
    {
        totalprogress->setValue(progress);
    }
}

void ifpGUIView::setTotalProgressTarget(int totalSteps)
{
    if (totalSteps == -1)
    {
        totalprogress->hide();
    }
    else
    {
        totalprogress->show();
        totalprogress->setMaximum(totalSteps);
    }
}

void ifpGUIView::hideCancelBtn(void)
{
    btnCancel->hide();
}

void ifpGUIView::showCancelBtn(void)
{
    btnCancel->show();
}

void ifpGUIView::showTransferProgress()
{
    ifpTransferInfo.show_progress = true;
    ifpTransferToolBar->setVisible(true);
}

void ifpGUIView::hideTransferProgress()
{
    ifpTransferInfo.show_progress       = false;
    ifpTransferInfo.file_bytes          = 0;
    ifpTransferInfo.file_total          = 0;
    ifpTransferInfo.is_status_batch     = false;
    ifpTransferInfo.batch_total         = 0;
    ifpTransferInfo.batch_bytes         = 0;
    ifpTransferInfo.file_name           = "";

    ifpTransferInfo.is_batch            = false;
    ifpTransferInfo.files_count         = 0;
    ifpTransferInfo.files_total         = 0;
    ifpTransferInfo.total_to_transfer   = 0;
    ifpTransferInfo.total_transfered    = 0;

    ifpTransferInfo.delete_success      = false;
    ifpTransferInfo.auto_convert        = false;

    ifpTransferToolBar->setVisible(false);
}

bool ifpGUIView::isTransferProgressVisible()
{
    return(ifpTransferToolBar->isVisible());
}

void ifpGUIView::updateProgress(ifp_progress_info *pg_info)
{
    setFileName(pg_info->file_name);

    if (pg_info->is_batch)
    {
        setFileProgressTarget(pg_info->file_total);
        setFileProgress(pg_info->file_bytes);

        if (pg_info->is_status_batch)
        {
            setTotalProgressTarget(pg_info->total_to_transfer);
            setTotalProgress(pg_info->total_transfered + pg_info->batch_bytes);// + pg_info->file_bytes);
        }
        else
        {
            setTotalProgressTarget(pg_info->total_to_transfer);
            setTotalProgress(pg_info->total_transfered + pg_info->file_bytes);
        }
    }
    else if (pg_info->is_status_batch)
    {
        setFileProgressTarget(pg_info->file_total);
        setFileProgress(pg_info->file_bytes);

        setTotalProgressTarget(pg_info->batch_total);
        setTotalProgress(pg_info->batch_bytes);// + pg_info->file_bytes);
    }
    else
    {
        setTotalProgressTarget(-1);

        if (pg_info->file_total > 0)
            setFileProgressTarget(pg_info->file_total);

        if (pg_info->file_bytes > 0)
            setFileProgress(pg_info->file_bytes);
    }
    qApp->processEvents();
}

int progress(void *context, struct ifp_transfer_status *status)
{
    ifp_progress_info *prog_info;

    if (context == NULL)
        return(0); // this is really a bad error

    if (MainIfpGuiWindow->cancel_progress == 1)
        return(1);

    prog_info = (ifp_progress_info*)context;

    ifpTransferInfo.show_progress       = true;
    ifpTransferInfo.file_bytes          = status->file_bytes;
    ifpTransferInfo.file_total          = status->file_total;
    ifpTransferInfo.is_status_batch     = status->is_batch;
    ifpTransferInfo.batch_total         = status->batch_total;
    ifpTransferInfo.batch_bytes         = status->batch_bytes;
    ifpTransferInfo.file_name           = QString(status->file_name);

    ifpTransferInfo.is_batch            = prog_info->is_batch;
    ifpTransferInfo.files_count         = prog_info->files_count;
    ifpTransferInfo.files_total         = prog_info->files_total;
    ifpTransferInfo.total_to_transfer   = prog_info->total_to_transfer;
    ifpTransferInfo.total_transfered    = prog_info->total_transfered;

    // misc
    ifpTransferInfo.delete_success      = prog_info->delete_success;
    ifpTransferInfo.auto_convert        = prog_info->auto_convert;


    return(0);
}
