//*****************************************************************************
//                                 TextCtrl.cpp                               *
//                                --------------                              *
//  Started     : 21/06/2004                                                  *
//  Last Update : 29/10/2008                                                  *
//  Copyright   : (C) 2004 by MSWaters                                        *
//  Email       : M.Waters@bom.gov.au                                         *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "utility/TextCtrl.hpp"

//*****************************************************************************
// Initialize static variables for class

int  TextCtrl::m_iLinesMax = TCTL_LINESDEF;
int  TextCtrl::m_iLinesDsp = TCTL_DISPDEF;

//*****************************************************************************
// Constructor.

TextCtrl::TextCtrl( void ) : wxTextCtrl( )
{
  bSetInitMsg( TCTL_INITMSG );

  bClear( );
}

//*****************************************************************************
// Destructor.

TextCtrl::~TextCtrl( )
{
}

//*****************************************************************************
// Create an instantance of this object.
//
// Argument List :
//   poWin  - The parent window
//   oWinID - The window identifier
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  TextCtrl::bCreate( wxWindow * poWin, wxWindowID oWinID )
{
  long  lStyle;

  // Check if the object has already been created
  if( bIsCreated( ) ) return( TRUE );

  lStyle = wxTE_MULTILINE | wxTE_READONLY | wxSUNKEN_BORDER | wxTE_DONTWRAP |
           wxHSCROLL;
  if( ! Create( poWin,oWinID,wxT(""),wxDefaultPosition,wxDefaultSize,lStyle ) )
    return( FALSE );

  SetFont( FONT_MONO );

  return( bClear( ) );
}

//*****************************************************************************
// Clear the text control.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  TextCtrl::bClear( void )
{
  long  lStyle;

  m_iLinesCnt = 0;

  // Check if the object has already been created
  if( ! bIsCreated( ) ) return( FALSE );

  // Left-justify the text control contents
  lStyle = GetWindowStyle( );
  lStyle &= ~( wxTE_LEFT | wxTE_CENTER | wxTE_RIGHT );
  lStyle |= wxTE_LEFT;
  SetWindowStyle( lStyle );

  Clear( );
  SetEditable( FALSE );

  return( TRUE );
}

//*****************************************************************************
// Clear the text control and display a message to the user.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  TextCtrl::bInitialize( void )
{
  int  i1;

  // Check if the object has already been created
  if( ! bIsCreated( ) ) return( FALSE );

  bClear( );

  if( ! m_osInitMsg.IsEmpty( ) )
  {
    // Center the default message horizontally
    SetWindowStyle( GetWindowStyle( ) | wxTE_CENTRE );

    // Display the requisite number of blank lines
    for( i1=1; i1<TCTL_INITLINES; i1++ )
      AppendText( wxT("\n") );

    // Display the default message
    AppendText( m_osInitMsg );
  }

  return( TRUE );
}

//*****************************************************************************
// Is the text control empty (ie. contains nothing or just the initialization
// message).
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  TextCtrl::bIsEmpty( void )
{
  long  li1;

  if( wxTextCtrl::IsEmpty( ) )                                return( TRUE );

  for( li1=1; li1<TCTL_INITLINES; li1++ )
    if( ! GetLineText( li1-1 ).IsEmpty( ) )                   return( FALSE );

  if( GetLineText( li1 ).Find( m_osInitMsg ) == wxNOT_FOUND ) return( FALSE );

  return( TRUE );
}

//*****************************************************************************
// Set the initialization message to be displayed.
//
// Argument List :
//   rosMsg - The initialization message
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  TextCtrl::bSetInitMsg( const wxString & rosMsg )
{
  if( rosMsg.Length( ) > 60 ) return( FALSE );

  m_osInitMsg = rosMsg;

  return( TRUE );
}

//*****************************************************************************
// Set the maximum number of lines to be loaded into the control.
//
// Argument List :
//   iLines - The maximum number of lines to be displayed
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  TextCtrl::bSetLinesMax( int iLines )
{
  if( iLines<TCTL_LINESMIN || iLines>TCTL_LINESMAX ) return( FALSE );

  m_iLinesMax = iLines;

  return( TRUE );
}

//*****************************************************************************
// Set the number of lines to be displayed in the control.
//
// Argument List :
//   iLines - The number of lines to be displayed
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  TextCtrl::bSetLinesDsp( int iLines )
{
  if( iLines<TCTL_DISPMIN || iLines>TCTL_DISPMAX ) return( FALSE );

  m_iLinesDsp = iLines;

  return( TRUE );
}


//*****************************************************************************
// Append a line of text to the text control.
//
// Argument List :
//   rosLine - The line of text to append
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  TextCtrl::bAppendLine( const wxString & rosLine )
{
  // Check if the object has already been created
  if( ! bIsCreated( ) )            return( FALSE );

  // Check that the max. lines hasn't been reached
  if( m_iLinesCnt >= m_iLinesMax ) return( FALSE );

  // Append this text and a line terminator
  *this << rosLine << wxT('\n');
  m_iLinesCnt++;

  return( TRUE );
}

//*****************************************************************************
// Append the contents of a file to the text control.
//
// Argument List :
//   roFName - The file name to append
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  TextCtrl::bAppendFile( const wxString & roFName )
{
  wxString  os1;
  size_t    szt1;

  // Check if the object has already been created
  if( ! bIsCreated( ) ) return( FALSE );

  // Open the file
  wxTextFile  oFile( roFName );
  if( ! oFile.Open( ) )
  {
    *this << wxT("Couldn't load the file containing the process output :\n  ");
    if( roFName.IsEmpty( ) )
         *this << wxT("The log file name is empty!");
    else *this << roFName;
    return( FALSE );
  }

  // Check that the file isn't empty
  if( oFile.GetLineCount( ) > 0 )
  { // Append the file contents to the text control
    for( szt1=0; szt1<oFile.GetLineCount( ); szt1++ )
    {
      os1 = oFile.GetLine( szt1 );
      if( ! bAppendLine( os1 ) )
      {
        *this << wxT("\nText control maximum line count (ie. ") << m_iLinesMax
              << wxT(") reached.\n");
        break;
      }
    }
  }
  else *this << wxT("There is no process output.");

  oFile.Close( );    // Close the log file

  SetInsertionPoint( 0 ); // Go to the top of the text control

  return( TRUE );
}

//*****************************************************************************
// Load the contents of a file into the text control.
//
// Argument List :
//   roFName - The file name to load
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  TextCtrl::bLoadFile( const wxString & roFName )
{
  // Check if the object has already been created
  if( ! bIsCreated( ) ) return( FALSE );

  bClear( ); // Clear the text control

  return( bAppendFile( roFName ) );
}

//*****************************************************************************
