/*
 * Author: Andrei Zavada <johnhommer@gmail.com>
 *         building on original work by Thomas Nowotny <tnowotny@ucsd.edu>
 *
 * License: GPL-2+
 *
 * Initial version: 2008-08-02
 *
 */


#ifndef LIBCN_BASE_UNIT_H
#define LIBCN_BASE_UNIT_H

#include <fstream>
#include <cstring>
#include <vector>
#include <list>

#include "types.hh"
#include "sources.hh"

#include "config.h"


using namespace std;
using namespace CNRun;
namespace CNRun {

// this gets referenced in case of out-of-bounds idx or misspelled sym
// in param accessors
extern double __cn_dummy_double;



// forward decls
class CModel;
class C_BaseUnit;






// for all units
#define CN_UERROR			(1 << 0)
#define CN_UOWNED			(1 << 1)
#define CN_UHASPARAMRANGE		(1 << 2)
#define CN_ULISTENING_MEM		(1 << 3)
#define CN_ULISTENING_DISK		(1 << 4)
#define CN_ULISTENING_1VARONLY		(1 << 5)
#define CN_ULISTENING_DEFERWRITE	(1 << 6)
#define CN_ULISTENING_BINARY		(1 << 7)
//#define CN_NDYNPARAMS			(1 << 8)

// only for neurons
#define CN_NFIRING			(1 << 9)  // firing now
#define CN_NREFRACT			(1 << 10)  // in refractory phase now


#define CN_MAX_LABEL_SIZE 40


// the base unit provides the methods for the following:
// * classification;
// * access to parameters, tape reader and range interface;
// * attachment to the mother model;
// * listening, i.e., keeping a history of vars along a timeline;
class C_BaseUnit {

    private:
	C_BaseUnit();  // not callable

    protected:
	TUnitType
		_type;  // will look up p, pno and vno from __CNUDT using _type as index
    public:
	TUnitType
		type() const		{  return _type;   }

      // classification
	const int	traits()	const {  return __CNUDT[_type].traits;			}
	const bool	is_hostable()	const {  return __CNUDT[_type].traits & UT_HOSTED;		}
	const bool	is_ddtbound()	const {  return __CNUDT[_type].traits & UT_DDTSET;		}
	const bool	is_neuron()	const {  return _type >= NT_FIRST && _type <= NT_LAST;	}
	const bool	is_synapse()	const {  return _type >= YT_FIRST && _type <= YT_LAST;	}
	const bool	is_oscillator()	const {  return __CNUDT[_type].traits & UT_OSCILLATOR;	}
	const bool	is_conscious()	const {  return is_oscillator();				}

	const char *class_name() const
		{  return is_neuron() ? "Neuron" : "Synapse";	}
	const char *species() const
		{  return __CNUDT[_type].species;		}
	const char *family() const
		{  return __CNUDT[_type].family;		}
	const char *type_description() const
		{  return __CNUDT[_type].description;		}

      // parameter & variable names and symbols
	const char *const param_name( size_t i)	const { return __CNUDT[_type].stock_param_names[i]; }
	const char *const param_sym( size_t i)	const { return __CNUDT[_type].stock_param_syms[i];  }
	int param_idx_by_sym( const char*) const __attribute__ ((pure));
	const char *const var_name( size_t i)	const { return __CNUDT[_type].stock_var_names[i];   }
	const char *const var_sym( size_t i)	const { return __CNUDT[_type].stock_var_syms[i];    }
	int var_idx_by_sym( const char*) const __attribute__ ((pure));
	unsigned short v_no() const	{ return __CNUDT[_type].vno; }
	unsigned short p_no() const	{ return __CNUDT[_type].pno; }

    protected:
      // identification
	unsigned long
		_serial_id;  // assigned incrementally as read by import_NetworkML
	char	_label[CN_MAX_LABEL_SIZE];
    public:
	unsigned long serial() const
		{  return _serial_id;  }
	const char *label() const  // for synapses, it is "%s:%d", src->label, targets.size()
		{  return _label;  }
	void set_label( const char *new_label)
		{  strncpy( _label, new_label, CN_MAX_LABEL_SIZE-1); }

      // status bitfield & properties
    protected:
	int	_status;
    public:
	int	status()	{  return _status; }

      // ctor & dtor
    protected:
	C_BaseUnit( TUnitType, const char *label,
		    CModel*, int s_mask);
    public:
	virtual ~C_BaseUnit();  // surely virtual

      // parent model
	friend class CModel;
	friend class SSpikeloggerService;
    protected:
	CModel	*M;
    public:
	const CModel&
		parent_model() const	{ return *M; }
	bool	is_owned() const	{ return _status & CN_UOWNED; }
	const double&
		model_time() const;  // defined in model.h

    public:
      // private copy of params
	vector<double> P;
	double get_param_value( size_t p) const
		{  return P[p];  }
	double get_param_value( const char *sym) const
		{
			int id = param_idx_by_sym( sym);
			return (id == -1) ? __cn_dummy_double : P[id];
		}
	double &param_value( size_t p)	{  return P[p];  }
	double &param_value( const char *sym)
		{
			int id = param_idx_by_sym( sym);
			return (id == -1) ? __cn_dummy_double : P[id];
		}
	void reset_params()
		{
			P.resize( p_no());
			memcpy( P.data(), __CNUDT[_type].stock_param_values,
				sizeof(double) * p_no());
			param_changed_hook();
		}

      // purity checks
	bool is_not_altered() const
		{
			return (memcmp( P.data(), __CNUDT[_type].stock_param_values,
				       sizeof (double) * p_no()) == 0) &&
				!has_sources();
		}
	bool has_same_params( const C_BaseUnit &rv) const
		{
			return _type == rv._type &&
				memcmp( P.data(), rv.P.data(), sizeof (double) * p_no()) == 0;
		}
	bool has_sources() const __attribute__ ((pure))
		{
			return not sources.empty();
		}
	bool has_same_sources( const C_BaseUnit &rv) const __attribute__ ((pure))
		{
			return sources == rv.sources;
			// not sure taking the order of otherwise identical sources should matter
		}
	bool is_identical( const C_BaseUnit &rv) const __attribute__ ((pure))
		{
			return	_type == rv._type && has_same_params(rv) &&
				((has_sources() && has_same_sources(rv)) ||
				 (!has_sources() && !rv.has_sources()));
		}
	virtual void dump( bool with_params = false, FILE *strm = stdout) const;


      // Source interface
	enum TSinkType { SINK_PARAM, SINK_VAR };

	template <class T>
	struct SSourceInterface {
	    friend class C_BaseUnit;
	    friend class CModel;
	    private:
		C_BaseSource *source;
		TSinkType sink_type;
		unsigned short idx;

		SSourceInterface( T *insource, TSinkType insink_type, unsigned short inidx)
		      : source (insource), sink_type (insink_type), idx (inidx)
			{}
	    public:
		bool operator== ( const SSourceInterface &rv) const
			{  return source == rv.source && sink_type == rv.sink_type && idx == rv.idx;  }
	};
	list <SSourceInterface <C_BaseSource> > sources;
	template <class T> void attach_source( T *s, TSinkType t, unsigned short idx);

	void detach_source( C_BaseSource*, TSinkType, unsigned short idx);

	void apprise_from_sources();
	virtual void param_changed_hook()
		{}


      // access to state variables: differs per hosted or standalone
	virtual double &var_value( size_t) = 0;
	virtual const double &get_var_value( size_t) const = 0;
	virtual void reset_vars() = 0;
	virtual void reset_state();

      // state history
	bool is_listening() const
		{
			return _status & (CN_ULISTENING_DISK | CN_ULISTENING_MEM);
		}
	void start_listening( int mask = 0 | CN_ULISTENING_DISK);
	void stop_listening();
	void restart_listening()
		{
			int lbits = _status & (CN_ULISTENING_DISK | CN_ULISTENING_MEM
					       | CN_ULISTENING_1VARONLY | CN_ULISTENING_DEFERWRITE);
			stop_listening();
			start_listening( lbits);
		}
	void pause_listening();
	void resume_listening();

    private:
      // where vars are written by tell()
	int _binwrite_handle;
	ofstream *_listener_disk;
      // ... and/or stored, in a diskless model
	vector<double> *_listener_mem;
    public:
      // by this method
	void tell();
	const vector<double> *listener_mem() const	{ return _listener_mem; }

	unsigned short precision;

      // one common method for all descendants
};



extern unsigned short __cn_default_unit_precision;

extern int __cn_verbosely;




class __C_BaseUnitCompareByLabel {
    public:
	bool operator () ( C_BaseUnit *&lv, C_BaseUnit *&rv)
		{
			return strcmp( lv->label(), rv->label()) < 0;
		}
};


}

#endif

// EOF
