// Licensed to the Apache Software Foundation (ASF) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The ASF licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

#pragma once

// TODO(wesm): IR compilation does not have any include directories set
#include "../../arrow/vendored/datetime/date.h"

bool is_leap_year(int yy);
bool did_days_overflow(arrow_vendored::date::year_month_day ymd);
int last_possible_day_in_month(int month, int year);

// A point of time measured in millis since epoch.
class EpochTimePoint {
 public:
  explicit EpochTimePoint(std::chrono::milliseconds millis_since_epoch)
      : tp_(millis_since_epoch) {}

  explicit EpochTimePoint(int64_t millis_since_epoch)
      : EpochTimePoint(std::chrono::milliseconds(millis_since_epoch)) {}

  int TmYear() const { return static_cast<int>(YearMonthDay().year()) - 1900; }

  int TmMon() const { return static_cast<unsigned int>(YearMonthDay().month()) - 1; }

  int TmYday() const {
    auto to_days = arrow_vendored::date::floor<arrow_vendored::date::days>(tp_);
    auto first_day_in_year = arrow_vendored::date::sys_days{
        YearMonthDay().year() / arrow_vendored::date::jan / 1};
    return (to_days - first_day_in_year).count();
  }

  int TmMday() const { return static_cast<unsigned int>(YearMonthDay().day()); }

  int TmWday() const {
    auto to_days = arrow_vendored::date::floor<arrow_vendored::date::days>(tp_);
    return (arrow_vendored::date::weekday{to_days} -  // NOLINT
            arrow_vendored::date::Sunday)
        .count();
  }

  int TmHour() const { return static_cast<int>(TimeOfDay().hours().count()); }

  int TmMin() const { return static_cast<int>(TimeOfDay().minutes().count()); }

  int TmSec() const {
    // TODO(wesm): UNIX y2k issue on int=gdv_int32 platforms
    return static_cast<int>(TimeOfDay().seconds().count());
  }

  EpochTimePoint AddYears(int num_years) const {
    auto ymd = YearMonthDay() + arrow_vendored::date::years(num_years);
    return EpochTimePoint((arrow_vendored::date::sys_days{ymd} +  // NOLINT
                           TimeOfDay().to_duration())
                              .time_since_epoch());
  }

  EpochTimePoint AddMonths(int num_months) const {
    auto ymd = YearMonthDay() + arrow_vendored::date::months(num_months);

    EpochTimePoint tp = EpochTimePoint((arrow_vendored::date::sys_days{ymd} +  // NOLINT
                                        TimeOfDay().to_duration())
                                           .time_since_epoch());

    if (did_days_overflow(ymd)) {
      int days_to_offset =
          last_possible_day_in_month(static_cast<int>(ymd.year()),
                                     static_cast<unsigned int>(ymd.month())) -
          static_cast<unsigned int>(ymd.day());
      tp = tp.AddDays(days_to_offset);
    }
    return tp;
  }

  EpochTimePoint AddDays(int num_days) const {
    auto days_since_epoch = arrow_vendored::date::sys_days{YearMonthDay()} +  // NOLINT
                            arrow_vendored::date::days(num_days);
    return EpochTimePoint(
        (days_since_epoch + TimeOfDay().to_duration()).time_since_epoch());
  }

  EpochTimePoint ClearTimeOfDay() const {
    return EpochTimePoint((tp_ - TimeOfDay().to_duration()).time_since_epoch());
  }

  bool operator==(const EpochTimePoint& other) const { return tp_ == other.tp_; }

  int64_t MillisSinceEpoch() const { return tp_.time_since_epoch().count(); }

  arrow_vendored::date::time_of_day<std::chrono::milliseconds> TimeOfDay() const {
    auto millis_since_midnight =
        tp_ - arrow_vendored::date::floor<arrow_vendored::date::days>(tp_);
    return arrow_vendored::date::time_of_day<std::chrono::milliseconds>(
        millis_since_midnight);
  }

 private:
  arrow_vendored::date::year_month_day YearMonthDay() const {
    return arrow_vendored::date::year_month_day{
        arrow_vendored::date::floor<arrow_vendored::date::days>(tp_)};  // NOLINT
  }

  std::chrono::time_point<std::chrono::system_clock, std::chrono::milliseconds> tp_;
};
