#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gtk/gtk.h>

#include "callbacks.h"
#include "interface.h"
#include "support.h"
#include "common.h"
#include "cat.h"
#include "decode.h"
#include "shared.h"
#include "sound.h"

/* Pop up error dialog */
static GtkWidget *error_dialog = NULL;

/*------------------------------------------------------------------------*/

/*  Error_Dialog()
 *
 *  Opens an error dialog box
 */

  void
Error_Dialog( char *message )
{
  GtkWidget *label;

  if( error_dialog != NULL ) return;
  error_dialog = create_error_dialog();
  label = lookup_widget( error_dialog, "error_message" );
  gtk_label_set_text( GTK_LABEL(label), message );
  gtk_widget_show( error_dialog );
}


void
on_main_window_destroy(
	GtkObject       *object,
	gpointer         user_data)
{
  Cleanup();
  gtk_main_quit();
}


gboolean
on_main_window_delete_event(
	GtkWidget       *widget,
	GdkEvent        *event,
	gpointer         user_data)
{
  return FALSE;
}


  gboolean
on_scope_expose_event(
	GtkWidget       *widget,
	GdkEventExpose  *event,
	gpointer         user_data)
{
  /* Draw scope background */
  cairo_t *cr = gdk_cairo_create( gbl_scope->window );
  cairo_set_source_rgb( cr, 0.0, .3, 0.0 );
  cairo_rectangle(
	  cr, 0.0, 0.0,
	  (double)gbl_scope_width,
	  (double)gbl_scope_height );
  cairo_fill( cr );
  cairo_destroy( cr );

  gtk_label_set_text( GTK_LABEL(gbl_scope_label), _("Signal Input") );

  return TRUE;
}


  void
on_speed_changed(
	GtkEditable     *editable,
	gpointer         user_data)
{
  rc_data.unit_elem = gtk_spin_button_get_value_as_int( gbl_speed );
  rc_data.unit_elem =
	(60 * rc_data.tone_freq) /
	(50 * CYCLES_PER_FRAG * rc_data.unit_elem);
}


void
on_squelch_changed(
	GtkEditable     *editable,
	gpointer         user_data)
{
  rc_data.det_squelch =
	gtk_spin_button_get_value_as_int( gbl_squelch );
}


void
on_ratio_changed                       (GtkEditable     *editable,
                                        gpointer         user_data)
{
  rc_data.det_ratio =
	gtk_spin_button_get_value( gbl_ratio );
}


void
on_err_quit_clicked(
	GtkButton       *button,
	gpointer         user_data)
{
  Cleanup();
  gtk_main_quit();
}


void
on_clear_clicked(
	GtkButton       *button,
	gpointer         user_data)
{
  gtk_text_buffer_set_text( gbl_rx_text_buffer, "", -1 );
}


gboolean
on_waterfall_expose_event(
	GtkWidget       *widget,
	GdkEventExpose  *event,
	gpointer         user_data)
{
  if( gbl_wfall_pixbuf == NULL )
	return( TRUE );

  gdk_draw_pixbuf(
	  widget->window, NULL,
	  gbl_wfall_pixbuf,
	  event->area.x, event->area.y,
	  event->area.x, event->area.y,
	  event->area.width, event->area.height,
	  GDK_RGB_DITHER_NONE, 0, 0 );

  return TRUE;
}


  gboolean
on_waterfall_button_press_event(
	GtkWidget       *widget,
	GdkEventButton  *event,
	gpointer         user_data)
{
  if( event->button == 1 )
  {
	if( isFlagSet(CAT_SETUP) )
	  Tune_Tcvr( event->x );
  }
  return TRUE;
}


  void
on_err_ok_button_clicked(
	GtkButton       *button,
	gpointer         user_data)
{
  gtk_widget_destroy( error_dialog );
}


void
on_wpm_toggled                         (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  if( gtk_toggle_button_get_active(togglebutton) )
	Set_Flag( ADAPT_SPEED );
  else
	Clear_Flag( ADAPT_SPEED );
}


void
on_error_dialog_destroy                (GtkObject       *object,
                                        gpointer         user_data)
{
  error_dialog = NULL;
}


gboolean
on_error_dialog_delete_event           (GtkWidget       *widget,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
  return TRUE;
}


void
on_waterfall_size_allocate             (GtkWidget       *widget,
                                        GdkRectangle    *allocation,
                                        gpointer         user_data)
{
  int idx;

  /* Destroy existing pixbuff */
  if( gbl_wfall_pixbuf != NULL )
  {
	g_object_unref( G_OBJECT(gbl_wfall_pixbuf) );
	gbl_wfall_pixbuf = NULL;
  }

  /* Create waterfall pixbuf */
  gbl_wfall_pixbuf = gdk_pixbuf_new(
	  GDK_COLORSPACE_RGB, FALSE, 8,
	  allocation->width, allocation->height );
  if( gbl_wfall_pixbuf == NULL )
  {
	Error_Dialog( _("Failed to create pixbuf for waterfall") );
	return;
  }

  gbl_wfall_pixels = gdk_pixbuf_get_pixels( gbl_wfall_pixbuf );
  gbl_wfall_width  = gdk_pixbuf_get_width ( gbl_wfall_pixbuf );
  gbl_wfall_height = gdk_pixbuf_get_height( gbl_wfall_pixbuf );
  gbl_wfall_rowstride  = gdk_pixbuf_get_rowstride( gbl_wfall_pixbuf );
  gbl_wfall_n_channels = gdk_pixbuf_get_n_channels( gbl_wfall_pixbuf );
  gdk_pixbuf_fill( gbl_wfall_pixbuf, 0 );

  /* Allocate average bin value buffer */
  if( !mem_realloc((void *)&gbl_bin_ave,
		(size_t)gbl_wfall_width * sizeof(int)) )
	return;
  for( idx = 0; idx < gbl_wfall_width; idx++ )
	gbl_bin_ave[idx] = 0;

  /* Initialize fft */
  Ifft_Init(FFT_INPUT_SIZE, gbl_wfall_width);
}


  void
on_rx_togglebutton_toggled			(GtkToggleButton *togglebutton,
									 gpointer         user_data)
{
  if( gtk_toggle_button_get_active(togglebutton) &&
	  isFlagSet(RCCONFIG_OK) )
  {
	char mesg[MESG_SIZE];
	int error = 0;
	mesg[0]   = '\0';

	/* Enable receiving Morse code */
	if( !Setup_Sound_Card(mesg, &error) )
	{
	  if( error )
	  {
		Strlcat( mesg, _("\nError: "), sizeof(mesg) );
		Strlcat( mesg, snd_strerror(error), sizeof(mesg) );
	  }
	  Cleanup();
	  Error_Dialog( mesg );
	}
	else if( isFlagSet(ENABLE_RECEIVE) )
	{
	  if( isFlagSet(ENABLE_CAT) ) Open_Tcvr_Serial();
	  gtk_idle_add( Print_Character, "RX" );
	}
	else
	  Error_Dialog( _("Failed to initialize signal detector") );
  }
  else
  {
	if( isFlagSet(ENABLE_RECEIVE) )
	{
	  g_idle_remove_by_data( "RX" );
	  if( isFlagSet(ENABLE_CAT) ) Close_Tcvr_Serial();
	  Close_Capture_Handle();
	}
  }
}


  void
on_scope_size_allocate                 (GtkWidget       *widget,
										GdkRectangle    *allocation,
										gpointer         user_data)
{
  gbl_scope_width  = allocation->width  - 2;
  gbl_scope_height = allocation->height - 2;
}


void
on_centerline_checkbutton_toggled      (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  if( gtk_toggle_button_get_active(togglebutton) )
	Set_Flag( CENTERLINE );
  else
	Clear_Flag( CENTERLINE );
}


void
on_ratio_radiobutton_toggled           (GtkToggleButton *togglebutton,
										gpointer         user_data)
{
  if( gtk_toggle_button_get_active(togglebutton) )
  {
	gtk_label_set_text( GTK_LABEL(gbl_scope_label), _("Lead/Trail Ratio") );
	Clear_Flag(
		SELECT_LEVEL   |
		DISPLAY_LEVEL  |
		DISPLAY_SIGNAL);
	Set_Flag( DISPLAY_RATIO );
  }
}


  void
on_level_radiobutton_toggled           (GtkToggleButton *togglebutton,
										gpointer         user_data)
{
  if( gtk_toggle_button_get_active(togglebutton) )
  {
	gtk_label_set_text( GTK_LABEL(gbl_scope_label), _("Squelch Level") );
	Set_Flag( SELECT_LEVEL );
	Set_Flag( DISPLAY_LEVEL );
	Clear_Flag(	DISPLAY_RATIO |	DISPLAY_SIGNAL );
  }
}


void
on_stop_radiobutton_toggled            (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  if( gtk_toggle_button_get_active(togglebutton) )
  {
	gtk_label_set_text( GTK_LABEL(gbl_scope_label), _("Hold Display") );
	Clear_Flag(
		DISPLAY_SIGNAL |
		SELECT_LEVEL   |
		DISPLAY_LEVEL  |
		DISPLAY_RATIO );
  }
}


void
on_signal_radiobutton_toggled          (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  if( gtk_toggle_button_get_active(togglebutton) )
  {
	gtk_label_set_text( GTK_LABEL(gbl_scope_label), _("Signal Detector") );
	Clear_Flag(
		SELECT_LEVEL  |
		DISPLAY_LEVEL |
		DISPLAY_RATIO );
	Set_Flag( DISPLAY_SIGNAL );
  }
}

