///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
///
# include "rheolef/csr.h"

# include "rheolef/outml.h"
# include "rheolef/outhb.h"
# include "rheolef/outmm.h"
# include "rheolef/outps.h"
# include "rheolef/out-sparse-matlab.h"
# include "rheolef/inhb.h"
# include "rheolef/iorheo.h"
# include "rheolef/catchmark.h"
# include "rheolef/asr.h"
using namespace rheolef;
using namespace std;

namespace rheolef { 
// =====================================[ IOS ]==================================
// matlab-like sparse pattern visualization
template <class T> void spy(const csr<T>& a) {
 string basename = iorheo::getbasename(clog);
 bool verbose = iorheo::getverbose(clog);
 bool clean   = iorheo::getclean(clog);
 bool execute = iorheo::getexecute(clog);
 std::ofstream o ((basename+".ps").c_str());
 o << ps;
 o << a;
 o.close();
 if (verbose) cerr << "! file \"" << basename << ".ps\" created." << endl;
 if (execute) {
    string command = "gv "+basename+".ps";
    int status = system (command.c_str());
 }
 if (clean) {
    string command = "rm -f "+basename+".ps";
    int status = system (command.c_str());
 }
}
template <class T>
ostream& 
operator << (ostream& s, const csr<T>& a)
{
    typedef typename csr<T>::size_type size_type;

    iorheo::flag_type fmt = (iorheo::flags(s) & iorheo::format_field);
	
    if (fmt [iorheo::dump]) {

        ml_pref.reset();
        ml_pref.bank_format = false;
        ml_pref.pr_as_read_syntax = false;
        print_matlab (s, a);
        return s;
    }
    if (fmt [iorheo::ml]) {

        ml_pref.reset();
        print_matlab (s, a);
        return s;
    }
    if (fmt [iorheo::ps]) {

	ps_pref.reset();
	ps_pref.color = true;
	print_postscript (s, a.nrow(), a.ncol(), a.nnz(),
		a.a().begin(), a.ja().begin(), a.ia().begin());
	return s;
	}
    if (fmt [iorheo::hb]) {

        // get hb options in stream:
        bool do_transpose = iorheo::gettranspose(s);
        size_type nrhs = iorheo::getnrhs(s);
        string rhs_type = iorheo::getrhstype(s);
        csr<T> b;
        if (! do_transpose) {
    	    // this is the default
    	    // convert CSR to CSC by transpoition
    	    // and then write CSC matrix in HB
    	    // TODO: do not transpose if a is symmetric !
            b = trans(a);
        } else {
    	    // interpret HB output as in CSC
    	    // so save transposed matrix in file
            b = a;
        }
        print_harwell_boeing (s, 
    	        b.nrow(), b.ncol(), b.nnz(), 
           	b.a().begin(), b.ia().begin(), b.ja().begin(),
    	        nrhs, 
    	        "RUA", "1Harwell-Boeing matrix file generated by sparskit++", "SKIT++",
    	        rhs_type.c_str());
        if (nrhs != 0) {
    	    // reset vector-counter 
    	    iorheo::setivec(s, 0);
        }
        return s;
    }
    if (fmt [iorheo::matrix_market]) {
        s << "%%MatrixMarket matrix coordinate real general" << endl
          << a.nrow() << " " << a.ncol() << " " << a.nnz() << endl;
        csr_output_matrix_market (s, a.ia().begin(), a.ia().end(), a.ja().begin(), a.a().begin());
        return s;
    }
    if (fmt [iorheo::sparse_matlab]) {
        string name = iorheo::getbasename(s);
        s << name << " = zeros(" << a.nrow() << "," << a.ncol() << ");" << endl;
        csr_output_sparse_matlab (s, name, a.ia().begin(), a.ia().end(), a.ja().begin(), a.a().begin());
        return s;
    }
    fatal_macro ("format not implemented");
    return s;
}
template <class T>
istream&
get_hb(istream& s, csr<T>& a)
{
    typedef typename csr<T>::size_type size_type;

    static const bool verbose_hb = false;
    // read HB
    char title[73], key[9], type[4];
    size_type nptr, nidx, nnz, neltvl;
    char ptrfmt[17], idxfmt[17], valfmt[21], rhsfmt [21];
    size_type totcrd, ptrcrd, idxcrd, valcrd, rhscrd;
    char rhstype [4]; size_type nrhs,nrhsix;
    size_type line_no = 1;
    
    // read header
    bool status = read_harwell_boeing (
    	s, title, key, type, 
    	nptr, nidx, nnz, neltvl,
        ptrfmt, idxfmt, valfmt, rhsfmt, 
        totcrd, ptrcrd, idxcrd, valcrd, rhscrd, 
        rhstype, nrhs, nrhsix,
    	0, line_no);
    if (!status) return s;
    if (verbose_hb) {
        trace_harwell_boeing (
    	    cerr, title, key, type,
    	    nptr, nidx, nnz, neltvl,
            ptrfmt, idxfmt, valfmt, rhsfmt,
            totcrd, ptrcrd, idxcrd, valcrd, rhscrd,
            rhstype, nrhs, nrhsix);
    }
    csr<T> b(nptr,nidx,nnz);
    
    status = read_harwell_boeing (
    	s, title, key, type,
    	nptr, nidx, nnz, neltvl,
        ptrfmt, idxfmt, valfmt, rhsfmt,
        totcrd, ptrcrd, idxcrd, valcrd, rhscrd,
        rhstype, nrhs, nrhsix,
    	0, line_no,
    
        // output values already allocated [nptr+1,nnz,nnz]
        b.a().begin(), b.ia().begin(), b.ja().begin());
    
    // set HB info in iorheo data-structure associated to stream 's'
    // -> usefull for rhs, guesses and exact solutions
    // set HB info in iorheo data-structure associated to stream 's'
    iorheo::setnptr(s,nptr);
    iorheo::setnidx(s,nidx);
    iorheo::setnrhs(s,nrhs);
    iorheo::setrhsfmt(s,rhsfmt);
    iorheo::setrhstype(s,rhstype);
    iorheo::setline_no(s,line_no);
    iorheo::setivec(s,0);
    
    if (! status) return s;
    
    bool do_transpose = iorheo::gettranspose(s);
    
    if (! do_transpose) {
    	trace_macro("CSC->CSR");
    	// convert to CSR: transpose and nrow :=: ncol
    	// using in-place transposition
    	// TODO: do not transpose if a is symmetric !
            a = trans(b);
    } else {
    	if (! b.is_sorted()) {
    	    trace_macro("CSR SORT");
    	    a = b.sort();
    	} else {
    	    a = b;
    	}
    }
    return s;
}
template <class T>
istream&
get_mm(istream& s, csr<T>& a)
{
    typedef typename csr<T>::size_type size_type;
    check_macro (scatch(s,"MatrixMarket"), "input stream does not contains a matrix-market.");
    char c = s.peek();
    while (s.good() && (c != '\n')) { 
	    s.get(c); 
    }
    check_macro(s.good(), "input stream does not contains a matrix-market.");
    size_type nrow;
    size_type ncol;
    size_type nnz;
    s >> nrow >> ncol >> nnz;
    if (!s.good()) return s;
    asr<T> x(nrow,ncol);
    for (size_type p = 0; p < nnz && s.good(); p++) {
       size_type i, j;
       T val;
       s >> i >> j >> val;
       x.entry(i-1,j-1) = val;
    }
    check_macro(s.good(), "input stream does not contains a matrix-market.");
    a = csr<T>(x);
    return s;
}
template <class T>
istream&
operator >> (istream& s, csr<T>& a)
{
    iorheo::flag_type fmt = (iorheo::flags(s) & iorheo::format_field);
    if (fmt [iorheo::hb]) {
       return get_hb(s,a);
    } else if (fmt [iorheo::matrix_market]) {
       return get_mm(s,a);
    }
    fatal_macro ("format not implemented");
    return s;
}
// =====================[ INSTANCIATION IN LIBRARY ]=============================
template istream& operator >> (istream&, csr<Float>&);
template ostream& operator << (ostream&, const csr<Float>&);
template void spy (const csr<Float>&);

}// namespace rheolef
