% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/page.R
\name{page_fillable}
\alias{page_fillable}
\title{A screen-filling page layout}
\usage{
page_fillable(
  ...,
  padding = NULL,
  gap = NULL,
  fillable_mobile = FALSE,
  title = NULL,
  theme = bs_theme(),
  lang = NULL
)
}
\arguments{
\item{...}{UI elements for the page. Named arguments become HTML attributes.}

\item{padding}{Padding to use for the body. This can be a numeric vector
(which will be interpreted as pixels) or a character vector with valid CSS
lengths. The length can be between one and four. If one, then that value
will be used for all four sides. If two, then the first value will be used
for the top and bottom, while the second value will be used for left and
right. If three, then the first will be used for top, the second will be
left and right, and the third will be bottom. If four, then the values will
be interpreted as top, right, bottom, and left respectively.}

\item{gap}{A \link[htmltools:validateCssUnit]{CSS length unit} defining the
\code{gap} (i.e., spacing) between elements provided to \code{...}.}

\item{fillable_mobile}{Whether or not the page should fill the viewport's
height on mobile devices (i.e., narrow windows).}

\item{title}{The browser window title (defaults to the host URL of the page)}

\item{theme}{A \code{\link[=bs_theme]{bs_theme()}} object.}

\item{lang}{ISO 639-1 language code for the HTML page, such as "en" or "ko".
This will be used as the lang in the \code{<html>} tag, as in \code{<html lang="en">}.
The default (NULL) results in an empty string.}
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#experimental}{\figure{lifecycle-experimental.svg}{options: alt='[Experimental]'}}}{\strong{[Experimental]}}

A Bootstrap-based page layout whose contents fill the full height and width
of the browser window.
}
\examples{
\dontshow{if (rlang::is_interactive()) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}

library(shiny)
library(ggplot2)

ui <- page_fillable(
  h1("Example", code("mtcars"), "dashboard"),
  layout_columns(
    card(
      full_screen = TRUE,
      card_header("Number of forward gears"),
      plotOutput("gear")
    ),
    card(
      full_screen = TRUE,
      card_header("Number of carburetors"),
      plotOutput("carb")
    )
  ),
  card(
    full_screen = TRUE,
    card_header("Weight vs. Quarter Mile Time"),
    layout_sidebar(
      sidebar = sidebar(
        varSelectInput("var_x", "Compare to qsec:", mtcars[-7], "wt"),
        varSelectInput("color", "Color by:", mtcars[-7], "cyl"),
        position = "right"
      ),
      plotOutput("var_vs_qsec")
    )
  )
)

server <- function(input, output) {
  for (var in c("cyl", "vs", "am", "gear", "carb")) {
    mtcars[[var]] <- as.factor(mtcars[[var]])
  }

  output$gear <- renderPlot({
    ggplot(mtcars, aes(gear)) + geom_bar()
  })

  output$carb <- renderPlot({
    ggplot(mtcars, aes(carb)) + geom_bar()
  })

  output$var_vs_qsec <- renderPlot({
    req(input$var_x, input$color)

    ggplot(mtcars) +
      aes(y = qsec, x = !!input$var_x, color = !!input$color) +
      geom_point()
  })
}

shinyApp(ui, server)
\dontshow{\}) # examplesIf}
}
\references{
\itemize{
\item \href{https://rstudio.github.io/bslib/articles/filling/index.html}{Filling Layouts}
on the bslib website.
\item \href{https://rstudio.github.io/bslib/articles/dashboards/index.html}{Getting Started with Dashboards}
on the bslib website.
}
}
\seealso{
\code{\link[=layout_columns]{layout_columns()}} and \code{\link[=layout_column_wrap]{layout_column_wrap()}} for laying out content
into rows and columns.

\code{\link[=layout_sidebar]{layout_sidebar()}} for 'floating' sidebar layouts.

\code{\link[=accordion]{accordion()}} for grouping related input controls in the \code{sidebar}.

\code{\link[=card]{card()}} for wrapping outputs in the 'main' content area.

\code{\link[=value_box]{value_box()}} for highlighting values.

Other Dashboard page layouts: 
\code{\link{page_navbar}()},
\code{\link{page_sidebar}()}
}
\concept{Dashboard page layouts}
