/* 
 * Copyright (C) 2000-2004 by Oswald Buddenhagen <puf@ossi.cjb.net>
 * based on puf 0.1.x (C) 1999,2000 by Anders Gavare <gavare@hotmail.com>
 *
 * You may modify and distribute this code under the terms of the GPL.
 * There is NO WARRANTY of any kind. See COPYING for details.
 *
 * puf.c - startup code and global functions
 *
 */

#include "puf.h"


int verbose;
char *progname;
#ifdef DEBUG
int debug;
#endif


#ifndef HAVE_STRDUP
char *
strdup(const char *s)
{
    int l = strlen(s) + 1;
    char *rt = malloc(l);
    if (rt)
	memcpy(rt, s, l);
    return rt;
}
#endif

#define FMTLEN 128

static const char *
expand_url(char *buf, url_t *u, const char *msg)
{
    char *pt;
    unsigned l, i, fl;

    if ((pt = strstr(msg, "$u"))) {
	l = print_url(buf + SHORTSTR + FMTLEN, SHORTSTR, u, 0);
	if (l >= SHORTSTR) {
	    memcpy(buf + 2 * SHORTSTR + FMTLEN - 4, "...", 4);
	    l = SHORTSTR - 1;
	}
	fl = (unsigned)(pt - msg);
	memcpy(buf, msg, fl);
	for (i = 0; i < l; i++) {
	    if (buf[i + SHORTSTR + FMTLEN] == '%')
		buf[fl++] = '%';
	    buf[fl++] = buf[i + SHORTSTR + FMTLEN];
	}
	strcpy(buf + fl, pt + 2);
	return buf;
    } else
	return msg;
}

int 
verrm(url_t *u, const char *msg, va_list va)
{
    int rtv, lv;
    char tbuf[2 * SHORTSTR + FMTLEN], fmt[SHORTSTR];

    num_errors++;
    if (*msg == '!') {
	msg++;
	num_urls_fail++;
	rtv = RT_GIVEUP;
	lv = ERR;
    } else if (++u->attempt >= (unsigned)u->parm->opt->max_attempts) {
	num_urls_fail++;
	rtv = RT_GIVEUP;
	lv = ERR;
    } else {
	rtv = RT_RETRY;
	lv = WRN;
    }

    if (lv <= verbose) {
	snprintf(fmt, sizeof(fmt), 
		 rtv == RT_RETRY ? "%s warning: %s (try %d)\n" : "%s: %s\n", 
		 progname, msg, u->attempt);
	vfprintf(stderr, expand_url(tbuf, u, fmt), va);
    }

    return rtv;
}

int 
errm(url_t *u, const char *msg, ...)
{
    va_list va;
    int rtv;

    va_start(va, msg);
    rtv = verrm(u, msg, va);
    va_end(va);
    return rtv;
}

void 
dbp(const char *msg, ...)
{
    va_list va;
    char fmt[SHORTSTR];

    snprintf(fmt, SHORTSTR, "%s debug: %s", progname, msg);
    va_start(va, msg);
    vfprintf(stderr, fmt, va);
    va_end(va);
}

void 
dbpe(const char *msg, ...)
{
    va_list va;

    va_start(va, msg);
    vfprintf(stderr, msg, va);
    va_end(va);
}

void 
dbpu(url_t *u, const char *msg, ...)
{
    va_list va;
    char tbuf[2 * SHORTSTR + FMTLEN], fmt[SHORTSTR];

    snprintf(fmt, SHORTSTR, "%s debug: %s", progname, msg);
    va_start(va, msg);
    vfprintf(stderr, expand_url(tbuf, u, fmt), va);
    va_end(va);
}

void 
dbpeu(url_t *u, const char *msg, ...)
{
    va_list va;
    char tbuf[2 * SHORTSTR + FMTLEN];

    va_start(va, msg);
    vfprintf(stderr, expand_url(tbuf, u, msg), va);
    va_end(va);
}

static const char *prfs[] = {" fatal", "", " warning", " info", " debug"};

void 
prx(int lev, const char *msg, ...)
{
    va_list va;
    char fmt[SHORTSTR];

    if (lev <= verbose) {
	snprintf(fmt, SHORTSTR, "%s%s: %s", progname, prfs[lev], msg);
	va_start(va, msg);
	vfprintf(stderr, fmt, va);
	va_end(va);
    }
}

void 
prxu(int lev, url_t *u, const char *msg, ...)
{
    va_list va;
    char tbuf[2 * SHORTSTR + FMTLEN], fmt[SHORTSTR];

    if (lev <= verbose) {
	snprintf(fmt, SHORTSTR, "%s%s: %s", progname, prfs[lev], msg);
	va_start(va, msg);
	vfprintf(stderr, expand_url(tbuf, u, fmt), va);
	va_end(va);
    }
}

/*  print an error message and terminate  */
void 
die(int ret, const char *msg, ...)
{
    char fmt[SHORTSTR];
    va_list va;

    snprintf(fmt, SHORTSTR, "\n%s: %s\n", progname, msg);
    va_start(va, msg);
    vfprintf(stderr, fmt, va);
    va_end(va);
    exit(ret);
}


/*  allocate memory or ask user what to do, if none available  */
void *
mrealloc(void *ptr, size_t size)
{
    void *t;
    static int asked = 0;

    if (size <= 0 || size > 100000)
	dbg(MEM, ("suspicious memory allocation: %lu bytes\n", size));
    if (!(t = realloc(ptr, size))) {
	if (!asked && isatty(0) && isatty(2)) {
	    prx(0, "Out of memory. Strange things may happen. Continue [n]? ");
	    if (getchar() != 'y')
		byebye("aborting");
	    asked++;
	} else
	    prx(ERR, "Allocation of %d bytes failed!\n", size);
    }
    return t;
}

void *
mmalloc(size_t size)
{
    return mrealloc(0, size);
}

static long crc32_tab[256];

static void 
init_hash(void)
{
    int i, b;
    long by;

    for (i = 0; i < 256; i++) {
	for (by = i, b = 0; b < 8; b++)
	    by = (by >> 1) ^ (-(by & 1) & 0xedb88320);
	crc32_tab[i] = by;
    }
}


/*  this is the normal crc32 algorithm.  */
int 
calc_nhash(const u_char *data, int len)
{
    int i;
    long crc = 0xc0debabe;

    for (i = 0; i < len; i++, data++)
	crc = (crc >> 8) ^ crc32_tab[(crc & 255) ^ *data];
    return (int)crc;
}

int 
calc_hash(const u_char *data)
{
    long crc = 0xc0debabe;

    for (; *data; data++)
	crc = (crc >> 8) ^ crc32_tab[(crc & 255) ^ *data];
    return (int)crc;
}

/*  Returns non-0 if string is matched by pattern. Does case folding.  */
int
patternMatch(const char *string, int slen, const char *pattern)
{
    int	p, sp;

    for (sp = 0;; sp++) {
	switch (p = *pattern++) {
	case '*':
	    if (!*pattern)
		return 1;
	    for (; sp < slen; sp++)
		if (patternMatch(string + sp, slen - sp, pattern))
		    return 1;
	    return 0;
	case '?':
	    if (sp >= slen)
		return 0;
	    break;
	case '\0':
	    return sp >= slen;
	case '\\':
	    p = *pattern++;
	    /* fall through */
	default:
	    if (sp >= slen)
		return 0;
	    if (tolower(p) != tolower(string[sp]))
		return 0;
	}
    }
}

int 
main(int argc, char *argv[])
{
#ifdef __DMALLOC_H__
    url_t *u, *nu;
    host_t *h, *nh;
    hinfo_t *hi;
#endif
    
    if (!isatty(1))
	show_stat = 0;
    else
	printf(PACKAGE " v" VERSION
	   "  Copyright (C) 2000-2004 by Oswald Buddenhagen\n"
	   "  based on puf v0.1.x  "
	     "Copyright (C) 1999,2000 by Anders Gavare\n");

    progname = strrchr(argv[0], '/');
    if (progname)
	progname++;
    else
	progname = argv[0];

    bind_addr.sin_family = AF_INET;	/*  the rest is already 0  */

    srand(time(0));

    init_hash();

    init_user_agent();

    getopts(argc, argv);

    setlocale(LC_TIME, "C");	/*  some voodoo for strftime  */

    fetch_all();

    return 0;
}
