/*
 *  PlanFacile (Easy plan, in french) is a small tool to help people to
 *  write a document on a particular subject.
 *  Copyright (C) 2005  Julien BRUGUIER
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef __COMMANDEMACRO__
#define __COMMANDEMACRO__

#include <src/global/global.h>

typedef struct commandemacro CONTENEUR(CommandeMacro);

typedef struct commandemacroparametres CONTENEUR(CommandeMacroParametres);

#include <src/donnees/flux/flux.h>
#include <src/donnees/flux/actionflux.h>
#include <src/donnees/commandes/localisationfichier.h>

#define TAILLEINIT	5
#define MULTTAILLE	2

struct commandemacroparametres
{
	STOCKAGE_SCALAIRE(Taille) memoire;
	//Taille rserve en mmoire.
	STOCKAGE_SCALAIRE(Taille) taille;
	//Taille rellement utilise.
	//Cette valeur donne le nombre de paramtres
	//passs  la macro lors de l'appel.
	TABLEAU(STOCKAGE(Flux)) parametre;
	//Paramtres de la macro.
	//Ces flux seront de type FLUX_MACRO_PARAMETRE.
};
/* Cette structure sert  contenir un nombre arbitraire
 * de paramtres de la macro.
 */

Resultat commandemacroparametres_initialisation(TRAVAIL(CommandeMacroParametres) commandemacroparametres);
/* Cre une structure destine  recevoir les diffrents
 * paramtres d'un appel de macro.
 * Renvoie RESULTAT_ERREUR_MEMOIRE en cas d'chec d'une allocation.
 */

Resultat commandemacroparametres_ajout_parametre(TRAVAIL(CommandeMacroParametres) commandemacroparametres , TRAVAIL(Flux) parametre);
/* Ajoute un parametre  la liste des paramtres.
 * Renvoie RESULTAT_ERREUR_MEMOIRE si l'allocation ventuelle choue.
 * Renvoie RESULTAT_ERREUR si la liste commandemacroparametres est NULL.
 */

Resultat commandemacroparametres_destruction(TRAVAIL(CommandeMacroParametres) commandemacroparametres);
/* Dtruit le contenu d'une liste de paramtres.
 */

struct commandemacro
{
	STOCKAGE(LocalisationFichier) localisation;
	//Localisation de l'appel de macro dans le source.
	STOCKAGE_SCALAIRE(NomMacro) nom;
	//Nom de la macro appelle.
	STOCKAGE(CommandeMacroParametres) parametres;
	//Paramtres de la macro.
};
/* Cette commande est destine  recevoir les diffrents lments permettant
 * de raliser un appel de macro. Contrairement  la commande #define, cette commande
 * sera explicitement prsente dans le flux de commandes.
 */

Resultat commandemacro_initialisation(TRAVAIL(CommandeMacro) commandemacro);
/* Cre une commande d'appel de macro vide.
 * Renvoie RESULTAT_ERREUR_MEMOIRE si une allocation choue.
 */

Resultat commandemacro_definition_localisationfichier(TRAVAIL(CommandeMacro) commandemacro , TRAVAIL(LocalisationFichier) localisationfichier);
/* Indique la position d'une commande d'appel de macro.
 * Renvoie RESULTAT_ERREUR si commandemacro est NULL.
 * Renvoie RESULTAT_ERREUR_MEMOIRE si une allocation choue.
 */

Resultat commandemacro_definition_nom(TRAVAIL(CommandeMacro) commandemacro , TRAVAIL_SCALAIRE(NomMacro) nom);
/* Assigne un nom  l'appel de macro.
 * Renvoie RESULTAT_ERREUR si commandemacro est NULL.
 * Renvoie RESULTAT_ERREUR_MEMOIRE si une allocation choue.
 */

Resultat commandemacro_ajout_parametre(TRAVAIL(CommandeMacro) commandemacro , TRAVAIL(Flux) parametre);
/* Ajoute un paramtre  l'appel de macro.
 * Renvoie RESULTAT_ERREUR si commandemacro est NULL.
 * Renvoie RESULTAT_ERREUR_MEMOIRE si une allocation choue.
 */

Resultat commandemacro_definition_parametre(TRAVAIL(CommandeMacro) commandemacro , TRAVAIL_SCALAIRE(Indice) indice , TRAVAIL(Flux) parametre);
/* Assigne un paramtre  un appel de macro.
 * Renvoie RESULTAT_ERREUR si commandemacro est NULL.
 * Renvoie RESULTAT_ERREUR_MEMOIRE si une allocation choue.
 * Renvoie RESULTAT_ERREUR_DEPASSEMENT si l'indice est incorrect.
 */

Resultat commandemacro_definition_parametres(TRAVAIL(CommandeMacro) commandemacro , TRAVAIL(CommandeMacroParametres) commandemacroparametres);
/* Assigne une liste de paramtres  un appel de macro.
 * Renvoie RESULTAT_ERREUR si commandemacro est NULL.
 * Renvoie RESULTAT_ERREUR_MEMOIRE si une allocation choue.
 */

Resultat commandemacro_lecture_localisationfichier(TRAVAIL(CommandeMacro) commandemacro , REFERENCE(LocalisationFichier) localisationfichier);
/* Indique la position d'une commande d'appel de macro.
 * Renvoie RESULTAT_ERREUR si commandemacro est NULL.
 */

Resultat commandemacro_lecture_nom(TRAVAIL(CommandeMacro) commandemacro , REFERENCE_SCALAIRE(NomMacro) nom);
/* Lit un nom  l'appel de macro.
 * Renvoie RESULTAT_ERREUR si commandemacro est NULL.
 */

Resultat commandemacro_lecture_tailleparametre(TRAVAIL(CommandeMacro) commandemacro , REFERENCE_SCALAIRE(Taille) tailleparametre);
/* Lit le nombre de paramtres de l'appel de macro.
 * Renvoie RESULTAT_ERREUR si commandemacro est NULL.
 */

Resultat commandemacro_lecture_parametre(TRAVAIL(CommandeMacro) commandemacro , TRAVAIL_SCALAIRE(Indice) indice , REFERENCE(Flux) parametre);
/* Lit un paramtre  l'appel de macro.
 * Renvoie RESULTAT_ERREUR si commandemacro est NULL.
 * Renvoie RESULTAT_ERREUR_DEPASSEMENT si l'indice est incorrect.
 */

Resultat commandemacro_decalage_parametres(TRAVAIL(CommandeMacro) commandemacro);
/* Dcale les paramtres d'un indice. Le paramtre 0 ajout
 * sera NULL.
 * Renvoie RESULTAT_ERREUR si commandemacro est NULL.
 * Renvoie RESULTAT_ERREUR_MEMOIRE si une allocation choue.
 */

Resultat commandemacro_parcours(TRAVAIL(CommandeMacro) commandemacro , TRAVAIL(ActionFlux) action , TRAVAIL(General) general);
/* Ralise un parcours des flux inclus dans la commande spcifie.
 * Renvoie les mmes erreurs que flux_parcours,
 * ainsi que RESULTAT_ERREUR si commande est NULL,
 * rien ne se passe si action est NULL.
 */

Resultat commandemacro_copie(TRAVAIL(CommandeMacro) commandemacro , TRAVAIL(CommandeMacro) copie);
/* Cre une copie de la commande macro.
 * Renvoie RESULTAT_ERREUR si commandemacro est NULL, et
 * RESULTAT_ERREUR_MEMOIRE si une allocation choue.
 * Si la copie est non vide, elle est au pralable supprime.
 */

Resultat commandemacro_destruction(TRAVAIL(CommandeMacro) commandemacro);
/* Dtruit une commande d'appel de macro.
 */

#endif
