/*	Conductor_Definition

PIRL CVS ID: Conductor_Definition.java,v 1.14 2012/04/16 06:04:11 castalia Exp

Copyright (C) 2008-2012	 Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package	PIRL.Conductor.Maestro;

import	PIRL.Conductor.Conductor;
import	PIRL.Messenger.Message;
import	PIRL.PVL.Parameter;
import	PIRL.PVL.Parser;
import	PIRL.PVL.PVL_Exception;


/**	A <i>Conductor_Definition</i> contains the information that defines a
	Conductor instance.
<p>
	A Conductor_Definition encapsulates the information necessary to
	instantiate a Conductor object.
<h4>
	Definition Name
</h4><p>
	The {@link #Name() name} of the definition is used to refer to the
	entire definition.Typically this name is expected to be unique in a
	set of Conductor_Definition objects such that if refers to a
	combination of definition parameter values that does not {@link
	#Matches(Message) match} the combination of parameter values in any
	other Conductor_Definition object of the set.
<h4>
	Definition Parameters
</h4><p>
	A Conductor_Defintion may contain the following parameters:
<dl>
<dt>{@link #PIPELINE_PARAMETER_NAME}
<dd>The pipeline name used by a Conductor instantiation. This parameter
	is required. If it is not present when the Conductor_Definition is
	constructed the parameter is provided using the definition name for
	its value.

<dt>{@link #CONFIGURATION_PARAMETER_NAME}
<dd>The source of the Configuration used by a Conductor instantiation.
	This parameter is required. If it is not present, or the alternative
	when the Conductor_Definition is constructed the parameter is
	provided using the value of the {@link
	#CONFIGURATION_SOURCE_PARAMETER_NAME} or, if this is not present, the
	{@link #DEFAULT_CONFIGURATION_FILENAME} for its value.

<dt>{@link #SERVER_PARAMETER_NAME}
<dd>The name of the database server access parameters group in the
	configuration file used by the Conductor instantiation. This
	parameter is optional. If it is not present the Conductor will use
	the parameter of the same name to choose the default database server
	group from the Configuration.

<dt>{@link #CATALOG_PARAMETER_NAME}
<dd>The name of the catalog in the database server where the pipeline
	tables are located. This parameter is optional. If it is not present
	the Conductor will the parameter of the same name from the
	Configuration.
<p>
	@author	Michael Wendell and Bradford Castalia, UA/PIRL
	@version 1.14
	@see	Conductor
*/
public class Conductor_Definition
	extends Message
{
/**	Class identification name with source code version and date.
*/
public static final String 
	ID = "PIRL.Conductor.Maestro.Conductor_Definition (1.14 2012/04/16 06:04:11)";


/**	The name of the parameter that specifies the pipeline name.
*/
public static final String
	PIPELINE_PARAMETER_NAME			= Conductor.PIPELINE_PARAMETER;

/**	The name of the parameter that specifies the Configuration source.
<p>
	@see	#CONFIGURATION_SOURCE_PARAMETER_NAME
*/
public static final String
	CONFIGURATION_PARAMETER_NAME	= "Configuration";

/**	The name of the parameter that specifies the Configuration source
	in a Conductor {@link Conductor#Identity()}.
<p>
	@see	#CONFIGURATION_PARAMETER_NAME
*/
public static final String
	CONFIGURATION_SOURCE_PARAMETER_NAME
		= Conductor.CONFIGURATION_SOURCE_PARAMETER;

/**	The default Configuration source filename.
*/
public static final String
	DEFAULT_CONFIGURATION_FILENAME	= Conductor.DEFAULT_CONFIGURATION_FILENAME;

/**	The name of the parameter that specifies the database server access
	information parameters group name in the {@link
	#CONFIGURATION_PARAMETER_NAME Configuration source}.
*/
public static final String
	SERVER_PARAMETER_NAME			= Conductor.DATABASE_SERVER_PARAMETER;

/**	The name of the parameter that specifies the database catalog
	containing the pipeline tables.
*/
public static final String
	CATALOG_PARAMETER_NAME			= Conductor.CATALOG_PARAMETER;

/**	The name of the parameter that specifies the {@link
	#Start_Conductor_Message(boolean, int)} action.
*/
public static final String
	ACTION_PARAMETER_NAME			= Message.ACTION_PARAMETER_NAME;

/**	The value of the {@link #ACTION_PARAMETER_NAME} parameter.
*/
public static final String
	START_CONDUCTORS_ACTION			= Stage_Manager.START_CONDUCTORS_ACTION;

/**	The name of the parameter that is used to tell a Conductor instantiation
	to wait to start.
<p>
	The Conductor will remain in the {@link Conductor#WAITING}.processing
	state if this parameter is included in the Conductor_Definition; otherwise
	it will immediately enter the {@link Conductor#RUNNING} processing
	state.
*/
public static final String
	WAIT_TO_START_PARAMETER_NAME	= "Wait_to_Start";

/**	The name of the parameter used to specify the number of Conductor
	instances to run in a Stage_Manger {@link
	#Start_Conductor_Message(boolean, int)}.
<p>
	<b>N.B.</b>: This parameter is not a Conductor command line option.
*/
public static final String
	COUNT_PARAMETER_NAME			= "Count";


private static final String 
	NL								= System.getProperty ("line.separator");


// Debug control.
private static final int
	DEBUG_OFF					= 0,
	DEBUG_CONSTRUCTOR			= 1 << 1,
	DEBUG_MATCHES				= 1 << 2,
	DEBUG_ALL					= -1,

	DEBUG						= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/** Creates a Conductor_Definition from a Message.
<p>
	The Message parameters and name are copied and then the contents
	are {@link #Validate() validated}.
<p>
	@param	message	A Message containg Conductor definition parameters.
	@throws PVL_Exception	If the definition could not be copied into
		this object and {@link #Validate() validated}.
	@throws	IllegalArgumentException	If the Message content does not
		produce a {@link #Validate() valid} Conductor_Definition.
*/
public Conductor_Definition 
	(
	Message		message
	)
	throws
		PVL_Exception,
		IllegalArgumentException
{
super (message);
Validate ();
}

/** Creates a Conductor_Definition from a Parameter.
<p>
	If the Parameter is an Assignment it becomes the sole
	Parameter in this Message. If it is an Aggregate its Parameter list
	becomes the list of Parameters in this Message and its {@link #Name()
	name} becomes the name of this Message.
<p>
	The contents of the new Conductor_Definition are {@link #Validate()
	validated}.
<p>
	@throws PVL_Exception	If the Parameter could not be copied into
		this object and {@link #Validate() validated}.
	@throws	IllegalArgumentException	If the Parameter content does not
		produce a {@link #Validate() valid} Conductor definition.
*/
public Conductor_Definition 
	(
	Parameter	parameter
	)
	throws
		PVL_Exception,
		IllegalArgumentException
{
super (parameter);
Validate ();
}

/** Creates a Conductor_Definition from a pipeline name.
<p>
	The resulting definition will have its {@link #Name() name} and
	{@link #PIPELINE_PARAMETER_NAME} parameter value set to the pipeline
	name. The {@link #CONFIGURATION_PARAMETER_NAME} will be set with the
	{@link #DEFAULT_CONFIGURATION_FILENAME} value. No other parameters
	will be present.
<p>
	@param	pipeline	The name of the Conductor pipeline.
	@throws IllegalArgumentException	If the pipeline name is null, empty
		or the special {@link Parser#CONTAINER_NAME}.
*/
public Conductor_Definition 
	(
	String		pipeline
	)
	throws IllegalArgumentException
{
if (pipeline == null)
	throw new IllegalArgumentException (ID + NL
		+ "Can't construct from a null pipeline name.");
if (pipeline.length () == 0 ||
	pipeline.equals (Parser.CONTAINER_NAME))
	throw new IllegalArgumentException (ID + NL
		+ "Invalid \"" + pipeline + "\" pipeline name for constructor.");
Name (pipeline);
Set (PIPELINE_PARAMETER_NAME, pipeline);
Set (CONFIGURATION_PARAMETER_NAME, DEFAULT_CONFIGURATION_FILENAME);
}

/**	A Conductor_Definition can not be constructed without arguments.
*/
private Conductor_Definition ()
{}

/*==============================================================================
	Accessors
*/
/** Test if a Conductor Definition Message matches this Conductor_Definition.
<p>
	The match criteria are:
<p>
	If the {@link #PIPELINE_PARAMETER_NAME} parameter is present in the
	message its value must match the value of the same parameter in this
	Conductor_Definition. If it is not present in the message the message
	{@link Message#Name() name} must match the parameter value in this
	Conductor_Definition. The Conductor_Definition will always contain
	this parameter; the Message must either contain the parameter with
	the same value or be named the same.
<p>
	If the {@link #CONFIGURATION_PARAMETER_NAME} or {@link
	#CONFIGURATION_SOURCE_PARAMETER_NAME} parameter is present in
	the message its value must match the value of the same parameter in
	this Conductor_Definition. If it is not present in the message the
	parameter value in this Conductor_Definition must be {@link
	#DEFAULT_CONFIGURATION_FILENAME}. The Conductor_Definition will
	always contain this parameter; it is optional for the Message.
<p>
	If the {@link #SERVER_PARAMETER_NAME} parameter is present in the
	message the same parameter must be present in this
	Conductor_Definition with the same value. If the parameter is not
	present in the message it must not be present in this
	Conductor_Definition. This parameter is optional for both the
	Conductor_Definition and the Message.
<p>
	If the {@link #CATALOG_PARAMETER_NAME} parameter is present in the
	message the same parameter must be present in this
	Conductor_Definition with the same value. If the parameter is not
	present in the message it must not be present in this
	Conductor_Definition. This parameter is optional for both the
	Conductor_Definition and the Message.
<p>
	<b>N.B.</b>: The {@link #Name() name} of the Message does not need
	to match the name of this Conductor_Definition. Also, the Message
	may contain additional parameters other than those used to determine
	the match; the match is only against the Conductor_Definition
	parameters.
<p>
	@param	message	A Message. If null, false is returned.
	@return	true if the Message contains a Conductor definition that
		logically matches this Conductor_Definition.
*/
public boolean Matches
	(
	Message		message
	)
{
if ((DEBUG & DEBUG_MATCHES) != 0)
	System.out.println
		(">>> Conductor_Definition.Matches:" + NL
		+"    this definition -" + NL
		+ toString () + NL
		+"    that message -" + NL
		+ message);
if (message == null)
	{
	if ((DEBUG & DEBUG_MATCHES) != 0)
		System.out.println
			("<<< Conductor_Definition.Matches: false");
	return false;
	}

String
	that_value;
if ((that_value = message.Get (PIPELINE_PARAMETER_NAME)) == null)
	 that_value = message.Name ();
if (! that_value.equals (Get (PIPELINE_PARAMETER_NAME)))
	{
	if ((DEBUG & DEBUG_MATCHES) != 0)
		System.out.println
		("    " + PIPELINE_PARAMETER_NAME
			+ " " + Get (PIPELINE_PARAMETER_NAME)
			+ " != " + that_value + NL
		+"<<< Conductor_Definition.Matches: false");
	return false;
	}

if ((that_value = message.Get (CONFIGURATION_PARAMETER_NAME)) == null &&
	(that_value = message.Get (CONFIGURATION_SOURCE_PARAMETER_NAME)) == null)
	that_value = DEFAULT_CONFIGURATION_FILENAME;
if (! that_value.equals (Get (CONFIGURATION_PARAMETER_NAME)))
	{
	if ((DEBUG & DEBUG_MATCHES) != 0)
		System.out.println
		("    " + CONFIGURATION_PARAMETER_NAME
			+ " " + Get (CONFIGURATION_PARAMETER_NAME)
			+ " != " + ((message.Get (CONFIGURATION_PARAMETER_NAME) != null) ?
			CONFIGURATION_PARAMETER_NAME : CONFIGURATION_SOURCE_PARAMETER_NAME)
			+ " " + that_value + NL
		+"<<< Conductor_Definition.Matches: false");
	return false;
	}

String
	this_value;
this_value = Get (SERVER_PARAMETER_NAME);
that_value = message.Get (SERVER_PARAMETER_NAME);
if (that_value != null &&
	! that_value.equals (this_value))
	{
	if ((DEBUG & DEBUG_MATCHES) != 0)
		System.out.println
		("    " + SERVER_PARAMETER_NAME
			+ " " + this_value
			+ " != " + that_value + NL
		+"<<< Conductor_Definition.Matches: false");
	return false;
	}
else
if (this_value != null &&
	! this_value.equals (that_value))
	{
	if ((DEBUG & DEBUG_MATCHES) != 0)
		System.out.println
		("    " + SERVER_PARAMETER_NAME
			+ " " + this_value
			+ " != " + that_value + NL
		+"<<< Conductor_Definition.Matches: false");
	return false;
	}

this_value = Get (CATALOG_PARAMETER_NAME);
that_value = message.Get (CATALOG_PARAMETER_NAME);
if (that_value != null &&
	! that_value.equals (this_value))
	{
	if ((DEBUG & DEBUG_MATCHES) != 0)
		System.out.println
		("    " + CATALOG_PARAMETER_NAME
			+ " " + this_value
			+ " != " + that_value + NL
		+"<<< Conductor_Definition.Matches: false");
	return false;
	}
else
if (this_value != null &&
	! this_value.equals (that_value))
	{
	if ((DEBUG & DEBUG_MATCHES) != 0)
		System.out.println
		("    " + CATALOG_PARAMETER_NAME
			+ " " + this_value
			+ " != " + that_value + NL
		+"<<< Conductor_Definition.Matches: false");
	return false;
	}

if ((DEBUG & DEBUG_MATCHES) != 0)
	System.out.println
		("<<< Conductor_Definition.Matches: true");
return true;
}

/** Generate a Message used by the Stage_Manager to instantiate a
	Conductor.
<p>
	A {@link #Validate() validated} copy of this Conductor_Definition is
	filtered for use as a Stage_Manager Conductor start request Message.
<p>
	The Conductor_Definition must contain a {@link
	#PIPELINE_PARAMETER_NAME} parameter. The new Message will have an
	{@link #ACTION_PARAMETER_NAME} set to {@link
	#START_CONDUCTORS_ACTION}.
<p>
	@param	wait_to_start	If true a {@link
		#WAIT_TO_START_PARAMETER_NAME} token parameter will be included
		in the returned Message.
	@param	count	If the value of this integer is greater than 1 a
		{@link #COUNT_PARAMETER_NAME} parameter having the count value
		will be included in the returned Message.
	@return	A Message suitable for use as a Stage_Manager Conductor start
		request message.
	@throws	IllegalArgumentException	If this Conductor_Description
		is not {@link #Validate() valid}.
	@throws	PVL_Exception	If this Conductor_Definition could not be
		copied into a new Message.
*/
public Message Start_Conductor_Message
	(
	boolean	wait_to_start,
	int		count
	)
	throws
		IllegalArgumentException,
		PVL_Exception
{
//	Get a validated copy of this definition.
Message 
	message = new Conductor_Definition (this);

message.Set (ACTION_PARAMETER_NAME, START_CONDUCTORS_ACTION, 0);

if (wait_to_start)
	message.Set_Token (WAIT_TO_START_PARAMETER_NAME);

if (count > 1)
	message.Set (COUNT_PARAMETER_NAME, count);

return message;
}

/** Generate a Message used by the Stage_Manager to instantiate
	a Conductor.
<p>
	This is a convenience interface to the {@link
	#Start_Conductor_Message(boolean, int)} method.
<p>
	@param	wait_to_start	If true a {@link
		#WAIT_TO_START_PARAMETER_NAME} token parameter will be included
		in the returned Message.
	@param	count	If the value of this Integer is greater than 1 a
		{@link #COUNT_PARAMETER_NAME} parameter having the count value
		will be included in the returned Message.
	@return	A Message suitable for use as a Stage_Manager Conductor start
		request message.
	@throws	PVL_Exception	If this Conductor_Definition could not be
		copied into a new Message.
	@throws	IllegalArgumentException	If this Conductor_Description
		does not contain a {@link #PIPELINE_PARAMETER_NAME} parameter.
	@see	#Start_Conductor_Message(boolean, int)
*/
public Message Start_Conductor_Message
	(
	boolean	wait_to_start,
	Integer	count
	)
	throws
		PVL_Exception,
		IllegalArgumentException
{return Start_Conductor_Message (wait_to_start,
	((count == null) ? 1 : count.intValue ()));}

/*==============================================================================
	Manipulators
*/
/** Validate this Conductor_Definition.
<p>
	The values of all Conductor definition parameters are obtained and
	checked for definition rule violations:
<p>
	Each Conductor_Definition parameter, if present, must have a
	non-empty value.
<p>
	If the {@link #PIPELINE_PARAMETER_NAME} parameter is not present the
	{@link #Name() name} of the definition must not be the {@link
	Parser#CONTAINER_NAME} or empty. The default value for this required
	parameter is the name of the definition so the name must be valid if
	the parameter is not present.
<p>
	All of the current parameters are then removed and the Conductor
	definition parameters are set to appropriate values:
<p>
	If the {@link #PIPELINE_PARAMETER_NAME} parameter was not present it
	is set to the definition name; the default value for this required
	parameter is the name of the definition. If the parameter was present
	and the name of the defintion is the {@link Parser#CONTAINER_NAME} or
	empty the definition name is set to the value of the parameter. This
	parameter will be the first entry in the Conductor_Definition.
<p>
	If neither the {@link #CONFIGURATION_PARAMETER_NAME} parameter nor
	{@link #CONFIGURATION_SOURCE_PARAMETER_NAME} parameter was present
	the {@link #CONFIGURATION_PARAMETER_NAME} parameter is set to the
	{@link #DEFAULT_CONFIGURATION_FILENAME}. This parameter will be the
	third entry in the Conductor_Definition.
<p>
	If the {@link #SERVER_PARAMETER_NAME} and {@link
	#CATALOG_PARAMETER_NAME} parameters were present they are restored.
	They are not entered if they were not originally present.
<p>
	<b>N.B.</b>: Any other parameters are not relevant to a
	Conductor_Definition and so are removed.
<p>
	@throws	IllegalArgumentException	If the {@link
		#PIPELINE_PARAMETER_NAME} parameter isn't present and the {@link
		#Name() name} of the definition is the {@link
		Parser#CONTAINER_NAME}.
*/
public void Validate ()
	throws IllegalArgumentException
{
String
	pipeline		= Parameter_Value (PIPELINE_PARAMETER_NAME),
	configuration	= Parameter_Value (CONFIGURATION_PARAMETER_NAME),
	server			= Parameter_Value (SERVER_PARAMETER_NAME),
	catalog			= Parameter_Value (CATALOG_PARAMETER_NAME);
if (configuration == null)
	configuration 	= Parameter_Value (CONFIGURATION_SOURCE_PARAMETER_NAME);

if (pipeline == null &&
	(Name ().equals (Parser.CONTAINER_NAME) ||
	 Name ().length () == 0))
	throw new IllegalArgumentException (ID + NL
		+ "Invalid Conductor definition." + NL
		+ "No definition name or \"" + PIPELINE_PARAMETER_NAME
			+ "\" parameter -" + NL
		+ Description ());

//	Start with a clean slate.
Remove_All ();

if (pipeline == null)
	//	The name will be valid from check above.
	pipeline = Name ();
else
if (Name ().equals (Parser.CONTAINER_NAME) ||
	Name ().length () == 0)
	Name (pipeline);

Set (PIPELINE_PARAMETER_NAME, pipeline);

if (configuration == null)
	configuration = DEFAULT_CONFIGURATION_FILENAME;
Set (CONFIGURATION_PARAMETER_NAME, configuration);

if (server != null)
	Set (SERVER_PARAMETER_NAME, server);
if (catalog != null)
	Set (CATALOG_PARAMETER_NAME, catalog);
}


private String Parameter_Value
	(
	String	parameter_name
	)
{
String
	value = Get (parameter_name);
if (value != null &&
	value.length () == 0)
	throw new IllegalArgumentException (ID + NL
		+ "Invalid Conductor definition." + NL
		+ "Empty \"" + parameter_name
			+ "\" parameter value -" + NL
		+ Description ());
return value;
}


}
