/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment unit test for pgmviewportfactory.c
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <tests/check/common/pgmcheck.h>
#include <pgm/pgm.h>
#include "pgmplugintestsuite.h"

/* test plugin loading */
PGM_START_TEST (test_factory_load)
{
  PgmViewportFactory *factory = NULL;
  PgmViewport *viewport = NULL;
  gboolean success;
  PgmError ret= PGM_ERROR_OK;

  success = pgm_init_check (NULL, NULL);
  fail_unless (success, "init_check returned FALSE");

  /* Try to load an unavailable plugin */
  factory = pgm_viewport_factory_new ("blah blah");
  ret =  pgm_viewport_factory_create (factory, &viewport);
  fail_if ( PGM_ERROR_X != ret,
            "viewport_factory_create did not detect the plugin doesn't exist");

  g_object_unref (factory);

  /* Try to load an available plugin */
  factory = pgm_viewport_factory_new (PGM_TEST_PLUGIN);
  fail_unless (factory != NULL, "viewport_factory_new returned FALSE with our "
               "plugin");
  gst_object_unref (factory);

  pgm_deinit ();
}
PGM_END_TEST;

PGM_START_TEST (test_factory_multi_load)
{
  PgmViewportFactory *factory1 = NULL, *factory2 = NULL;
  PgmViewport *viewport = NULL;
  gboolean success;

  success = pgm_init_check (NULL, NULL);
  fail_unless (success, "init_check returned FALSE");

  /* Load the plugin */
  factory1 = pgm_viewport_factory_new (PGM_TEST_PLUGIN);
  fail_unless (factory1 != NULL, "viewport_factory_new returned FALSE with our "
               "plugin");
  factory2 = pgm_viewport_factory_new (PGM_TEST_PLUGIN);
  fail_unless (factory2 != NULL, "viewport_factory_new returned FALSE with our "
               "plugin on the second call");
  fail_unless (factory1 == factory2, "two factories for the same plugin should"
               "be equal");
  ASSERT_OBJECT_REFCOUNT (factory1, "factory1", 3);
  ASSERT_OBJECT_REFCOUNT (factory2, "factory2", 3);

  pgm_viewport_factory_create (factory1, &viewport);
  fail_unless (viewport != NULL, "viewport creation failed");
  gst_object_unref (viewport);
  viewport = NULL;

  ASSERT_OBJECT_REFCOUNT (factory1, "factory1 after viewport creation/deletion", 3);

  gst_object_unref (factory1);
  factory1 = NULL;
  ASSERT_OBJECT_REFCOUNT (factory2, "factory2 after unref on factory1", 2);
  gst_object_unref (factory2);

  factory1 = pgm_viewport_factory_new (PGM_TEST_PLUGIN);
  fail_unless (factory1 != NULL, "viewport recreation failed");
  ASSERT_OBJECT_REFCOUNT (factory1, "factory1", 2);

  pgm_viewport_factory_create (factory1, &viewport);
  fail_unless (viewport != NULL, "viewport creation from a recreated factory failed");
  gst_object_unref (viewport);
  viewport = NULL;

  gst_object_unref (factory1);

  pgm_deinit ();
}
PGM_END_TEST;

/* test available plugin */
PGM_START_TEST (test_factory_desc)
{
  PgmViewportFactory *factory = NULL;
  gchar *string = NULL;
  gboolean success;
  PgmError ret;

  success = pgm_init_check (NULL, NULL);
  fail_unless (success, "init_check returned FALSE");

  factory = pgm_viewport_factory_new (PGM_TEST_PLUGIN);
  fail_unless (factory != NULL, "viewport_factory_new returned FALSE our "
               "plugin");

  /* Check the description */
  ret = pgm_viewport_factory_get_description (factory, &string);
  fail_if (ret != PGM_ERROR_OK, "viewport_factory_get_description error");

  if (!g_ascii_strcasecmp (PGM_TEST_PLUGIN, "boilerplate"))
    {
      success = !g_ascii_strcasecmp ("A boilerplate viewport used for tests",
                                string);
    }
  else if (!g_ascii_strcasecmp (PGM_TEST_PLUGIN, "opengl"))
    {
      success = !g_ascii_strcasecmp ("An OpenGL viewport", string);
    }
  else
    {
      success = TRUE;
    }

  fail_unless (success, "plugin returns a bad description");
  g_free (string);

  /* Check the license */
  ret = pgm_viewport_factory_get_license (factory, &string);
  fail_if (ret != PGM_ERROR_OK, "viewport_factory_get_license error");
  success = g_ascii_strcasecmp ("LGPL", string);
  fail_unless (!success, "plugin returns a bad license");
  g_free (string);

  /* Check the origin */
  ret = pgm_viewport_factory_get_origin (factory, &string);
  fail_if (ret != PGM_ERROR_OK, "viewport_factory_get_origin error");
  success = g_ascii_strcasecmp ("https://code.fluendo.com/pigment/trac",
                                string);
  fail_unless (!success, "plugin returns a bad origin");
  g_free (string);

  /* Check the author */
  ret = pgm_viewport_factory_get_author (factory, &string);
  fail_if (ret != PGM_ERROR_OK, "viewport_factory_get_author error");
  success = g_ascii_strcasecmp ("Loïc Molinari <loic@fluendo.com>",
                                string);
  fail_unless (!success, "plugin returns a bad author");
  g_free (string);

  gst_object_unref (factory);
  pgm_deinit ();
}
PGM_END_TEST;

/* test viewport creation with make */
PGM_START_TEST (test_factory_make)
{
  PgmViewport *viewport = NULL;
  gboolean success;
  PgmError ret;

  success = pgm_init_check (NULL, NULL);
  fail_unless (success, "init_check returned FALSE");

  /* Create a viewport with factory_make */
  ret = pgm_viewport_factory_make (PGM_TEST_PLUGIN, &viewport);
  fail_if (ret != PGM_ERROR_OK, "viewport_factory_create error");
  fail_unless (PGM_IS_VIEWPORT (viewport), "viewport_factory_create returns "
               "a bad viewport");
  ASSERT_OBJECT_REFCOUNT (viewport, "viewport", 1);

  gst_object_unref (viewport);
  pgm_deinit ();
}
PGM_END_TEST;


/* test 2 viewport creations with new/create */
PGM_START_TEST (test_factory_create)
{
  PgmViewportFactory *factory = NULL;
  PgmViewport *vp1 = NULL, *vp2 = NULL;
  gboolean success;
  PgmError ret;

  success = pgm_init_check (NULL, NULL);
  fail_unless (success, "init_check returned FALSE");

  /* Create the factory */
  factory = pgm_viewport_factory_new (PGM_TEST_PLUGIN);
  fail_unless (factory != NULL, "viewport_factory_new returned FALSE with our "
               "plugin");
  ASSERT_OBJECT_REFCOUNT (factory, "factory", 2);

  /* Instantiate a first viewport, increase the factory ref count */
  ret = pgm_viewport_factory_create (factory, &vp1);
  fail_if (ret != PGM_ERROR_OK, "viewport_factory_create error");
  fail_unless (PGM_IS_VIEWPORT (vp1), "viewport_factory_create returns a bad "
               "viewport");
  ASSERT_OBJECT_REFCOUNT (vp1, "vp1", 1);
  ASSERT_OBJECT_REFCOUNT (factory, "factory", 3);

  /* Instantiate a second viewport, increase the factory ref count */
  ret = pgm_viewport_factory_create (factory, &vp2);
  fail_if (ret != PGM_ERROR_OK, "viewport_factory_create error");
  fail_unless (PGM_IS_VIEWPORT (vp2), "viewport_factory_create returns a bad "
               "viewport");
  ASSERT_OBJECT_REFCOUNT (vp2, "vp1", 1);
  ASSERT_OBJECT_REFCOUNT (factory, "factory", 4);

  /* Unref the viewports */
  gst_object_unref (vp1);
  ASSERT_OBJECT_REFCOUNT (factory, "factory", 3);
  gst_object_unref (vp2);
  ASSERT_OBJECT_REFCOUNT (factory, "factory", 2);

  gst_object_unref (factory);
  pgm_deinit ();
}
PGM_END_TEST;

Suite *
pgm_viewport_factory_suite (void)
{
  Suite *s = suite_create ("PgmViewportFactory");
  TCase *tc_chain = tcase_create ("pgmviewportfactory tests");

  suite_add_tcase (s, tc_chain);
  tcase_add_test (tc_chain, test_factory_load);
  tcase_add_test (tc_chain, test_factory_desc);
  tcase_add_test (tc_chain, test_factory_make);
  tcase_add_test (tc_chain, test_factory_create);
  tcase_add_test (tc_chain, test_factory_multi_load);

  return s;
}

