/*
 *  Copyright (c) 2010 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef _OPENGTL_GRAY_COLOR_CONVERTER_P_H_
#define _OPENGTL_GRAY_COLOR_CONVERTER_P_H_

#include "AbstractColorConverter.h"
#include "StdTypes.h"
#include "Color.h"

#include "RgbColorConverter_p.h"

namespace GTLCore {
  template<typename _Channel_Type_, bool _has_alpha>
  class GTLCORE_EXPORT GrayColorConverter : public AbstractColorConverter
  {
  private:
    typedef GTLCore::GammaToLinearFloat<_Channel_Type_> GammaToLinearFloat;
    typedef GTLCore::IntegerToFloat<_Channel_Type_>     IntegerToFloat;
    typedef GTLCore::LinearToGammaFloat<_Channel_Type_> LinearToGammaFloat;
    typedef GTLCore::FloatToInteger<_Channel_Type_>     FloatToInteger;
    GTL_NO_COPY(GrayColorConverter)
  public:
    GrayColorConverter(double _gamma, const GTLCore::PixelDescription& _pixelDescription) : AbstractColorConverter(_pixelDescription), m_gamma(_gamma),
        m_pixelToRgb( RgbColorConverterTraits<_Channel_Type_>::createPixelToRgbA(_gamma)),
        m_rgbToPixel( RgbColorConverterTraits<_Channel_Type_>::createRgbAToPixel(_gamma)),
        m_vectorToRgb(_gamma),
        m_rgbToVector(_gamma)
    {
    }
  public:
    virtual ~GrayColorConverter()
    {
    }
    virtual void pixelToRgba(const char* _data, RgbaF* _rgba) const
    {
      const _Channel_Type_* data = reinterpret_cast<const _Channel_Type_*>(_data);
      float v = m_pixelToRgb(data[0]);
      _rgba->r = v;
      _rgba->g = v;
      _rgba->b = v;
      if(_has_alpha) {
        _rgba->a = ChannelMaths<_Channel_Type_, float>::scaleToA(data[1]);
      } else {
        _rgba->a = 1.0;
      }
    }
    virtual void rgbaToPixel(const RgbaF* _rgba, char* _data) const
    {
      _Channel_Type_* data = reinterpret_cast<_Channel_Type_*>(_data);
      data[0] = m_rgbToPixel((_rgba->r + _rgba->g + _rgba->b) / 3.0);
      if(_has_alpha) {
        data[1] = ChannelMaths<float, _Channel_Type_>::scaleToA(_rgba->a);
      }
    }
    virtual void vectorToRgba(const float* _data, RgbaF* _rgba) const
    {
      float v = m_vectorToRgb(_data[0]);
      _rgba->r = v;
      _rgba->g = v;
      _rgba->b = v;
      if(_has_alpha) {
        _rgba->a = _data[1];
      } else {
        _rgba->a = 1.0;
      }
    }
    virtual void rgbaToVector(const RgbaF* _rgba, float* _data) const
    {
      _data[0] = m_rgbToVector((_rgba->r + _rgba->g + _rgba->b) / 3.0);
      if(_has_alpha) {
        _data[1] = _rgba->a;
      }
    }
  private:
    // Those are placeholders for the future conversion to/from HSV and to/from LAB
    virtual void placeholder1() {}
    virtual void placeholder2() {}
    virtual void placeholder3() {}
    virtual void placeholder4() {}
    virtual void placeholder5() {}
    virtual void placeholder6() {}
    virtual void placeholder7() {}
    virtual void placeholder8() {}
  private:
    double m_gamma;
    typename RgbColorConverterTraits<_Channel_Type_>::PixelToRgb m_pixelToRgb;
    typename RgbColorConverterTraits<_Channel_Type_>::RgbToPixel m_rgbToPixel;
    GammaToLinearFloat m_vectorToRgb;
    LinearToGammaFloat m_rgbToVector;
  };
}

#endif
