'use strict';

require('mocha');
const assert = require('assert').strict;
const bashPath = require('bash-path');
const cp = require('child_process');
const braces = require('..');

const bash = input => {
  return cp.spawnSync(bashPath(), ['-c', `echo ${input}`])
    .stdout.toString()
    .split(/\s+/)
    .filter(Boolean);
};

const equal = (input, expected = bash(input), options) => {
  assert.deepEqual(braces.expand(input, options), expected);
};

/**
 * Bash 4.3 unit tests with `braces.expand()`
 */

describe('bash - expanded brace ranges', () => {
  describe('large numbers', () => {
    it('should expand large numbers', () => {
      equal('{2147483645..2147483649}', ['2147483645', '2147483646', '2147483647', '2147483648', '2147483649']);
    });

    it('should throw an error when range exceeds rangeLimit', () => {
      assert.throws(() => braces.expand('{214748364..2147483649}'));
    });
  });

  describe('escaping / invalid ranges', () => {
    it('should not try to expand ranges with decimals', () => {
      equal('{1.1..2.1}', ['{1.1..2.1}']);
      equal('{1.1..~2.1}', ['{1.1..~2.1}']);
    });

    it('should escape invalid ranges:', () => {
      equal('{1..0f}', ['{1..0f}']);
      equal('{1..10..ff}', ['{1..10..ff}']);
      equal('{1..10.f}', ['{1..10.f}']);
      equal('{1..10f}', ['{1..10f}']);
      equal('{1..20..2f}', ['{1..20..2f}']);
      equal('{1..20..f2}', ['{1..20..f2}']);
      equal('{1..2f..2}', ['{1..2f..2}']);
      equal('{1..ff..2}', ['{1..ff..2}']);
      equal('{1..ff}', ['{1..ff}']);
      equal('{1.20..2}', ['{1.20..2}']);
    });

    it('weirdly-formed brace expansions -- fixed in post-bash-3.1', () => {
      equal('a-{b{d,e}}-c', ['a-{bd}-c', 'a-{be}-c']);
      equal('a-{bdef-{g,i}-c', ['a-{bdef-g-c', 'a-{bdef-i-c']);
    });

    it('should not expand quoted strings.', () => {
      equal('{"klklkl"}{1,2,3}', ['{klklkl}1', '{klklkl}2', '{klklkl}3']);
      equal('{"x,x"}', ['{x,x}']);
    });

    it('should escaped outer braces in nested non-sets', () => {
      equal('{a-{b,c,d}}', ['{a-b}', '{a-c}', '{a-d}']);
      equal('{a,{a-{b,c,d}}}', ['a', '{a-b}', '{a-c}', '{a-d}']);
    });

    it('should escape imbalanced braces', () => {
      equal('a-{bdef-{g,i}-c', ['a-{bdef-g-c', 'a-{bdef-i-c']);
      equal('abc{', ['abc{']);
      equal('{abc{', ['{abc{']);
      equal('{abc', ['{abc']);
      equal('}abc', ['}abc']);
      equal('ab{c', ['ab{c']);
      equal('ab{c', ['ab{c']);
      equal('{{a,b}', ['{a', '{b']);
      equal('{a,b}}', ['a}', 'b}']);
      equal('abcd{efgh', ['abcd{efgh']);
      equal('a{b{c{d,e}f}g}h', ['a{b{cdf}g}h', 'a{b{cef}g}h']);
      equal('f{x,y{{g,z}}h}', ['fx', 'fy{g}h', 'fy{z}h']);
      equal('z{a,b},c}d', ['za,c}d', 'zb,c}d']);
      equal('a{b{c{d,e}f{x,y{{g}h', ['a{b{cdf{x,y{{g}h', 'a{b{cef{x,y{{g}h']);
      equal('f{x,y{{g}h', ['f{x,y{{g}h']);
      equal('f{x,y{{g}}h', ['f{x,y{{g}}h']);
      equal('a{b{c{d,e}f{x,y{}g}h', ['a{b{cdfxh', 'a{b{cdfy{}gh', 'a{b{cefxh', 'a{b{cefy{}gh']);
      equal('f{x,y{}g}h', ['fxh', 'fy{}gh']);
      equal('z{a,b{,c}d', ['z{a,bd', 'z{a,bcd']);
    });
  });

  describe('positive numeric ranges', () => {
    it('should expand numeric ranges', () => {
      equal('a{0..3}d', ['a0d', 'a1d', 'a2d', 'a3d']);
      equal('x{10..1}y', ['x10y', 'x9y', 'x8y', 'x7y', 'x6y', 'x5y', 'x4y', 'x3y', 'x2y', 'x1y']);
      equal('x{3..3}y', ['x3y']);
      equal('{1..10}', ['1', '2', '3', '4', '5', '6', '7', '8', '9', '10']);
      equal('{1..3}', ['1', '2', '3']);
      equal('{1..9}', ['1', '2', '3', '4', '5', '6', '7', '8', '9']);
      equal('{10..1}', ['10', '9', '8', '7', '6', '5', '4', '3', '2', '1']);
      equal('{10..1}y', ['10y', '9y', '8y', '7y', '6y', '5y', '4y', '3y', '2y', '1y']);
      equal('{3..3}', ['3']);
      equal('{5..8}', ['5', '6', '7', '8']);
    });
  });

  describe('negative ranges', () => {
    it('should expand ranges with negative numbers', () => {
      equal('{-10..-1}', ['-10', '-9', '-8', '-7', '-6', '-5', '-4', '-3', '-2', '-1']);
      equal('{-20..0}', ['-20', '-19', '-18', '-17', '-16', '-15', '-14', '-13', '-12', '-11', '-10', '-9', '-8', '-7', '-6', '-5', '-4', '-3', '-2', '-1', '0']);
      equal('{0..-5}', ['0', '-1', '-2', '-3', '-4', '-5']);
      equal('{9..-4}', ['9', '8', '7', '6', '5', '4', '3', '2', '1', '0', '-1', '-2', '-3', '-4']);
    });
  });

  describe('alphabetical ranges', () => {
    it('should expand alphabetical ranges', () => {
      equal('{a..F}', ['a', '`', '_', '^', ']', '\\', '[', 'Z', 'Y', 'X', 'W', 'V', 'U', 'T', 'S', 'R', 'Q', 'P', 'O', 'N', 'M', 'L', 'K', 'J', 'I', 'H', 'G', 'F']);
      equal('0{a..d}0', ['0a0', '0b0', '0c0', '0d0']);
      equal('a/{b..d}/e', ['a/b/e', 'a/c/e', 'a/d/e']);
      equal('{1..f}', ['1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f']);
      equal('{a..A}', ['a', '`', '_', '^', ']', '\\', '[', 'Z', 'Y', 'X', 'W', 'V', 'U', 'T', 'S', 'R', 'Q', 'P', 'O', 'N', 'M', 'L', 'K', 'J', 'I', 'H', 'G', 'F', 'E', 'D', 'C', 'B', 'A']);
      equal('{A..a}', ['A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a']);
      equal('{a..e}', ['a', 'b', 'c', 'd', 'e']);
      equal('{A..E}', ['A', 'B', 'C', 'D', 'E']);
      equal('{a..f}', ['a', 'b', 'c', 'd', 'e', 'f']);
      equal('{a..z}', ['a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z']);
      equal('{E..A}', ['E', 'D', 'C', 'B', 'A']);
      equal('{f..1}', ['f', 'e', 'd', 'c', 'b', 'a', '`', '_', '^', ']', '\\', '[', 'Z', 'Y', 'X', 'W', 'V', 'U', 'T', 'S', 'R', 'Q', 'P', 'O', 'N', 'M', 'L', 'K', 'J', 'I', 'H', 'G', 'F', 'E', 'D', 'C', 'B', 'A', '@', '?', '>', '=', '<', ';', ':', '9', '8', '7', '6', '5', '4', '3', '2', '1']);
      equal('{f..a}', ['f', 'e', 'd', 'c', 'b', 'a']);
      equal('{f..f}', ['f']);
    });

    it('should expand multiple ranges:', () => {
      equal('a/{b..d}/e/{f..h}', ['a/b/e/f', 'a/b/e/g', 'a/b/e/h', 'a/c/e/f', 'a/c/e/g', 'a/c/e/h', 'a/d/e/f', 'a/d/e/g', 'a/d/e/h']);
    });
  });

  describe('combo', () => {
    it('should expand numerical ranges - positive and negative', () => {
      equal('a{01..05}b', ['a01b', 'a02b', 'a03b', 'a04b', 'a05b']);
      equal('0{1..9}/{10..20}', ['01/10', '01/11', '01/12', '01/13', '01/14', '01/15', '01/16', '01/17', '01/18', '01/19', '01/20', '02/10', '02/11', '02/12', '02/13', '02/14', '02/15', '02/16', '02/17', '02/18', '02/19', '02/20', '03/10', '03/11', '03/12', '03/13', '03/14', '03/15', '03/16', '03/17', '03/18', '03/19', '03/20', '04/10', '04/11', '04/12', '04/13', '04/14', '04/15', '04/16', '04/17', '04/18', '04/19', '04/20', '05/10', '05/11', '05/12', '05/13', '05/14', '05/15', '05/16', '05/17', '05/18', '05/19', '05/20', '06/10', '06/11', '06/12', '06/13', '06/14', '06/15', '06/16', '06/17', '06/18', '06/19', '06/20', '07/10', '07/11', '07/12', '07/13', '07/14', '07/15', '07/16', '07/17', '07/18', '07/19', '07/20', '08/10', '08/11', '08/12', '08/13', '08/14', '08/15', '08/16', '08/17', '08/18', '08/19', '08/20', '09/10', '09/11', '09/12', '09/13', '09/14', '09/15', '09/16', '09/17', '09/18', '09/19', '09/20']);
      equal('{-10..10}', ['-10', '-9', '-8', '-7', '-6', '-5', '-4', '-3', '-2', '-1', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '10']);
    });
  });

  describe('steps > positive ranges', () => {
    it('should expand ranges using steps:', () => {
      equal('{1..10..1}', ['1', '2', '3', '4', '5', '6', '7', '8', '9', '10']);
      equal('{1..10..2}', ['1', '3', '5', '7', '9']);
      equal('{1..20..20}', ['1']);
      equal('{1..20..2}', ['1', '3', '5', '7', '9', '11', '13', '15', '17', '19']);
      equal('{10..0..2}', ['10', '8', '6', '4', '2', '0']);
      equal('{10..1..2}', ['10', '8', '6', '4', '2']);
      equal('{100..0..5}', ['100', '95', '90', '85', '80', '75', '70', '65', '60', '55', '50', '45', '40', '35', '30', '25', '20', '15', '10', '5', '0']);
      equal('{2..10..1}', ['2', '3', '4', '5', '6', '7', '8', '9', '10']);
      equal('{2..10..2}', ['2', '4', '6', '8', '10']);
      equal('{2..10..3}', ['2', '5', '8']);
      equal('{a..z..2}', ['a', 'c', 'e', 'g', 'i', 'k', 'm', 'o', 'q', 's', 'u', 'w', 'y']);
    });

    it('should expand positive ranges with negative steps:', () => {
      equal('{10..0..-2}', ['10', '8', '6', '4', '2', '0']);
    });
  });

  describe('steps > negative ranges', () => {
    it('should expand negative ranges using steps:', () => {
      equal('{-1..-10..-2}', ['-1', '-3', '-5', '-7', '-9']);
      equal('{-1..-10..2}', ['-1', '-3', '-5', '-7', '-9']);
      equal('{-10..-2..2}', ['-10', '-8', '-6', '-4', '-2']);
      equal('{-2..-10..1}', ['-2', '-3', '-4', '-5', '-6', '-7', '-8', '-9', '-10']);
      equal('{-2..-10..2}', ['-2', '-4', '-6', '-8', '-10']);
      equal('{-2..-10..3}', ['-2', '-5', '-8']);
      equal('{-50..-0..5}', ['-50', '-45', '-40', '-35', '-30', '-25', '-20', '-15', '-10', '-5', '0']);
      equal('{10..1..-2}', ['10', '8', '6', '4', '2']);
      equal('{100..0..-5}', ['100', '95', '90', '85', '80', '75', '70', '65', '60', '55', '50', '45', '40', '35', '30', '25', '20', '15', '10', '5', '0']);
    });
  });

  describe('steps > alphabetical ranges', () => {
    it('should expand alpha ranges with steps', () => {
      equal('{a..e..2}', ['a', 'c', 'e']);
      equal('{E..A..2}', ['E', 'C', 'A']);
      equal('{a..z}', ['a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z']);
      equal('{a..z..2}', ['a', 'c', 'e', 'g', 'i', 'k', 'm', 'o', 'q', 's', 'u', 'w', 'y']);
      equal('{z..a..-2}', ['z', 'x', 'v', 't', 'r', 'p', 'n', 'l', 'j', 'h', 'f', 'd', 'b']);
    });

    it('should expand alpha ranges with negative steps', () => {
      equal('{z..a..-2}', ['z', 'x', 'v', 't', 'r', 'p', 'n', 'l', 'j', 'h', 'f', 'd', 'b']);
    });
  });

  describe('padding', () => {
    it('unwanted zero-padding -- fixed post-bash-4.0', () => {
      equal('{10..0..2}', ['10', '8', '6', '4', '2', '0']);
      equal('{10..0..-2}', ['10', '8', '6', '4', '2', '0']);
      equal('{-50..-0..5}', ['-50', '-45', '-40', '-35', '-30', '-25', '-20', '-15', '-10', '-5', '0']);
    });
  });

  describe('ranges', () => {
    const fixtures = [
      'should expand ranges',

      ['a{b,c{1..50}/{d,e,f}/,g}h/i', {}, ['abh/i', 'ac1/d/h/i', 'ac1/e/h/i', 'ac1/f/h/i', 'ac2/d/h/i', 'ac2/e/h/i', 'ac2/f/h/i', 'ac3/d/h/i', 'ac3/e/h/i', 'ac3/f/h/i', 'ac4/d/h/i', 'ac4/e/h/i', 'ac4/f/h/i', 'ac5/d/h/i', 'ac5/e/h/i', 'ac5/f/h/i', 'ac6/d/h/i', 'ac6/e/h/i', 'ac6/f/h/i', 'ac7/d/h/i', 'ac7/e/h/i', 'ac7/f/h/i', 'ac8/d/h/i', 'ac8/e/h/i', 'ac8/f/h/i', 'ac9/d/h/i', 'ac9/e/h/i', 'ac9/f/h/i', 'ac10/d/h/i', 'ac10/e/h/i', 'ac10/f/h/i', 'ac11/d/h/i', 'ac11/e/h/i', 'ac11/f/h/i', 'ac12/d/h/i', 'ac12/e/h/i', 'ac12/f/h/i', 'ac13/d/h/i', 'ac13/e/h/i', 'ac13/f/h/i', 'ac14/d/h/i', 'ac14/e/h/i', 'ac14/f/h/i', 'ac15/d/h/i', 'ac15/e/h/i', 'ac15/f/h/i', 'ac16/d/h/i', 'ac16/e/h/i', 'ac16/f/h/i', 'ac17/d/h/i', 'ac17/e/h/i', 'ac17/f/h/i', 'ac18/d/h/i', 'ac18/e/h/i', 'ac18/f/h/i', 'ac19/d/h/i', 'ac19/e/h/i', 'ac19/f/h/i', 'ac20/d/h/i', 'ac20/e/h/i', 'ac20/f/h/i', 'ac21/d/h/i', 'ac21/e/h/i', 'ac21/f/h/i', 'ac22/d/h/i', 'ac22/e/h/i', 'ac22/f/h/i', 'ac23/d/h/i', 'ac23/e/h/i', 'ac23/f/h/i', 'ac24/d/h/i', 'ac24/e/h/i', 'ac24/f/h/i', 'ac25/d/h/i', 'ac25/e/h/i', 'ac25/f/h/i', 'ac26/d/h/i', 'ac26/e/h/i', 'ac26/f/h/i', 'ac27/d/h/i', 'ac27/e/h/i', 'ac27/f/h/i', 'ac28/d/h/i', 'ac28/e/h/i', 'ac28/f/h/i', 'ac29/d/h/i', 'ac29/e/h/i', 'ac29/f/h/i', 'ac30/d/h/i', 'ac30/e/h/i', 'ac30/f/h/i', 'ac31/d/h/i', 'ac31/e/h/i', 'ac31/f/h/i', 'ac32/d/h/i', 'ac32/e/h/i', 'ac32/f/h/i', 'ac33/d/h/i', 'ac33/e/h/i', 'ac33/f/h/i', 'ac34/d/h/i', 'ac34/e/h/i', 'ac34/f/h/i', 'ac35/d/h/i', 'ac35/e/h/i', 'ac35/f/h/i', 'ac36/d/h/i', 'ac36/e/h/i', 'ac36/f/h/i', 'ac37/d/h/i', 'ac37/e/h/i', 'ac37/f/h/i', 'ac38/d/h/i', 'ac38/e/h/i', 'ac38/f/h/i', 'ac39/d/h/i', 'ac39/e/h/i', 'ac39/f/h/i', 'ac40/d/h/i', 'ac40/e/h/i', 'ac40/f/h/i', 'ac41/d/h/i', 'ac41/e/h/i', 'ac41/f/h/i', 'ac42/d/h/i', 'ac42/e/h/i', 'ac42/f/h/i', 'ac43/d/h/i', 'ac43/e/h/i', 'ac43/f/h/i', 'ac44/d/h/i', 'ac44/e/h/i', 'ac44/f/h/i', 'ac45/d/h/i', 'ac45/e/h/i', 'ac45/f/h/i', 'ac46/d/h/i', 'ac46/e/h/i', 'ac46/f/h/i', 'ac47/d/h/i', 'ac47/e/h/i', 'ac47/f/h/i', 'ac48/d/h/i', 'ac48/e/h/i', 'ac48/f/h/i', 'ac49/d/h/i', 'ac49/e/h/i', 'ac49/f/h/i', 'ac50/d/h/i', 'ac50/e/h/i', 'ac50/f/h/i', 'agh/i']],
      ['0{1..9} {10..20}', {}, ['01 10', '01 11', '01 12', '01 13', '01 14', '01 15', '01 16', '01 17', '01 18', '01 19', '01 20', '02 10', '02 11', '02 12', '02 13', '02 14', '02 15', '02 16', '02 17', '02 18', '02 19', '02 20', '03 10', '03 11', '03 12', '03 13', '03 14', '03 15', '03 16', '03 17', '03 18', '03 19', '03 20', '04 10', '04 11', '04 12', '04 13', '04 14', '04 15', '04 16', '04 17', '04 18', '04 19', '04 20', '05 10', '05 11', '05 12', '05 13', '05 14', '05 15', '05 16', '05 17', '05 18', '05 19', '05 20', '06 10', '06 11', '06 12', '06 13', '06 14', '06 15', '06 16', '06 17', '06 18', '06 19', '06 20', '07 10', '07 11', '07 12', '07 13', '07 14', '07 15', '07 16', '07 17', '07 18', '07 19', '07 20', '08 10', '08 11', '08 12', '08 13', '08 14', '08 15', '08 16', '08 17', '08 18', '08 19', '08 20', '09 10', '09 11', '09 12', '09 13', '09 14', '09 15', '09 16', '09 17', '09 18', '09 19', '09 20']],
      ['a{0..3}d', {}, ['a0d', 'a1d', 'a2d', 'a3d']],
      ['x{10..1}y', {}, ['x10y', 'x9y', 'x8y', 'x7y', 'x6y', 'x5y', 'x4y', 'x3y', 'x2y', 'x1y']],
      ['x{3..3}y', {}, ['x3y']],
      ['{0..10,braces}', {}, ['0..10', 'braces']],
      ['{3..3}', {}, ['3']],
      ['{5..8}', {}, ['5', '6', '7', '8']],
      ['**/{1..5}/a.js', {}, ['**/1/a.js', '**/2/a.js', '**/3/a.js', '**/4/a.js', '**/5/a.js']],
      ['{braces,{0..10}}', {}, ['braces', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '10']],
      ['./\\{x,y}/{a..z..3}/', {}, ['./{x,y}/a/', './{x,y}/d/', './{x,y}/g/', './{x,y}/j/', './{x,y}/m/', './{x,y}/p/', './{x,y}/s/', './{x,y}/v/', './{x,y}/y/']],
      ['x{{0..10},braces}y', {}, ['x0y', 'x1y', 'x2y', 'x3y', 'x4y', 'x5y', 'x6y', 'x7y', 'x8y', 'x9y', 'x10y', 'xbracesy']],
      ['{braces,{0..10}}', {}, ['braces', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '10']],
      ['{{0..10},braces}', {}, ['0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '10', 'braces']],
      ['{{1..10..2},braces}', {}, ['1', '3', '5', '7', '9', 'braces']],
      ['{{1..10},braces}', {}, ['1', '2', '3', '4', '5', '6', '7', '8', '9', '10', 'braces']],
      ['{1.1..2.1}', {}, ['{1.1..2.1}']],
      ['{1.1..~2.1}', {}, ['{1.1..~2.1}']],
      ['{1..0f}', {}, ['{1..0f}']],
      ['{1..10..ff}', {}, ['{1..10..ff}']],
      ['{1..10.f}', {}, ['{1..10.f}']],
      ['{1..10f}', {}, ['{1..10f}']],
      ['{1..20..2f}', {}, ['{1..20..2f}']],
      ['{1..20..f2}', {}, ['{1..20..f2}']],
      ['{1..2f..2}', {}, ['{1..2f..2}']],
      ['{1..ff..2}', {}, ['{1..ff..2}']],
      ['{1..ff}', {}, ['{1..ff}']],
      ['{1.20..2}', {}, ['{1.20..2}']],
      ['a{0..3}d', {}, ['a0d', 'a1d', 'a2d', 'a3d']],
      ['x{10..1}y', {}, ['x10y', 'x9y', 'x8y', 'x7y', 'x6y', 'x5y', 'x4y', 'x3y', 'x2y', 'x1y']],
      ['x{3..3}y', {}, ['x3y']],
      ['{1..10}', {}, ['1', '2', '3', '4', '5', '6', '7', '8', '9', '10']],
      ['{1..3}', {}, ['1', '2', '3']],
      ['{1..9}', {}, ['1', '2', '3', '4', '5', '6', '7', '8', '9']],
      ['{10..1}y', {}, ['10y', '9y', '8y', '7y', '6y', '5y', '4y', '3y', '2y', '1y']],
      ['{3..3}', {}, ['3']],
      ['{5..8}', {}, ['5', '6', '7', '8']],
      ['{-10..-1}', {}, ['-10', '-9', '-8', '-7', '-6', '-5', '-4', '-3', '-2', '-1']],
      ['{-20..0}', {}, ['-20', '-19', '-18', '-17', '-16', '-15', '-14', '-13', '-12', '-11', '-10', '-9', '-8', '-7', '-6', '-5', '-4', '-3', '-2', '-1', '0']],
      ['{0..-5}', {}, ['0', '-1', '-2', '-3', '-4', '-5']],
      ['{9..-4}', {}, ['9', '8', '7', '6', '5', '4', '3', '2', '1', '0', '-1', '-2', '-3', '-4']],
      ['0{1..9}/{10..20}', {}, ['01/10', '01/11', '01/12', '01/13', '01/14', '01/15', '01/16', '01/17', '01/18', '01/19', '01/20', '02/10', '02/11', '02/12', '02/13', '02/14', '02/15', '02/16', '02/17', '02/18', '02/19', '02/20', '03/10', '03/11', '03/12', '03/13', '03/14', '03/15', '03/16', '03/17', '03/18', '03/19', '03/20', '04/10', '04/11', '04/12', '04/13', '04/14', '04/15', '04/16', '04/17', '04/18', '04/19', '04/20', '05/10', '05/11', '05/12', '05/13', '05/14', '05/15', '05/16', '05/17', '05/18', '05/19', '05/20', '06/10', '06/11', '06/12', '06/13', '06/14', '06/15', '06/16', '06/17', '06/18', '06/19', '06/20', '07/10', '07/11', '07/12', '07/13', '07/14', '07/15', '07/16', '07/17', '07/18', '07/19', '07/20', '08/10', '08/11', '08/12', '08/13', '08/14', '08/15', '08/16', '08/17', '08/18', '08/19', '08/20', '09/10', '09/11', '09/12', '09/13', '09/14', '09/15', '09/16', '09/17', '09/18', '09/19', '09/20']],
      ['0{a..d}0', {}, ['0a0', '0b0', '0c0', '0d0']],
      ['a/{b..d}/e', {}, ['a/b/e', 'a/c/e', 'a/d/e']],
      ['{1..f}', { minimatch: false }, ['1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f']],
      ['{a..A}', {}, ['a', '`', '_', '^', ']', '\\', '[', 'Z', 'Y', 'X', 'W', 'V', 'U', 'T', 'S', 'R', 'Q', 'P', 'O', 'N', 'M', 'L', 'K', 'J', 'I', 'H', 'G', 'F', 'E', 'D', 'C', 'B', 'A']],
      ['{A..a}', {}, ['A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a']],
      ['{a..e}', {}, ['a', 'b', 'c', 'd', 'e']],
      ['{A..E}', {}, ['A', 'B', 'C', 'D', 'E']],
      ['{a..f}', {}, ['a', 'b', 'c', 'd', 'e', 'f']],
      ['{a..z}', {}, ['a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z']],
      ['{E..A}', {}, ['E', 'D', 'C', 'B', 'A']],
      ['{f..1}', { minimatch: false }, ['f', 'e', 'd', 'c', 'b', 'a', '`', '_', '^', ']', '\\', '[', 'Z', 'Y', 'X', 'W', 'V', 'U', 'T', 'S', 'R', 'Q', 'P', 'O', 'N', 'M', 'L', 'K', 'J', 'I', 'H', 'G', 'F', 'E', 'D', 'C', 'B', 'A', '@', '?', '>', '=', '<', ';', ':', '9', '8', '7', '6', '5', '4', '3', '2', '1']],
      ['{f..a}', {}, ['f', 'e', 'd', 'c', 'b', 'a']],
      ['{f..f}', {}, ['f']],
      ['a/{b..d}/e/{f..h}', {}, ['a/b/e/f', 'a/b/e/g', 'a/b/e/h', 'a/c/e/f', 'a/c/e/g', 'a/c/e/h', 'a/d/e/f', 'a/d/e/g', 'a/d/e/h']],
      ['{-10..10}', {}, ['-10', '-9', '-8', '-7', '-6', '-5', '-4', '-3', '-2', '-1', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '10']],
      ['{1..10..1}', { optimize: false }, ['1', '2', '3', '4', '5', '6', '7', '8', '9', '10']],
      ['{1..10..2}', { optimize: false }, ['1', '3', '5', '7', '9']],
      ['{1..20..20}', { optimize: false }, ['1']],
      ['{1..20..2}', { optimize: false }, ['1', '3', '5', '7', '9', '11', '13', '15', '17', '19']],
      ['{10..0..2}', { optimize: false }, ['10', '8', '6', '4', '2', '0']],
      ['{10..1..2}', { optimize: false }, ['10', '8', '6', '4', '2']],
      ['{100..0..5}', { optimize: false }, ['100', '95', '90', '85', '80', '75', '70', '65', '60', '55', '50', '45', '40', '35', '30', '25', '20', '15', '10', '5', '0']],
      ['{2..10..1}', { optimize: false }, ['2', '3', '4', '5', '6', '7', '8', '9', '10']],
      ['{2..10..2}', { optimize: false }, ['2', '4', '6', '8', '10']],
      ['{2..10..3}', { optimize: false }, ['2', '5', '8']],
      ['{a..z..2}', { optimize: false }, ['a', 'c', 'e', 'g', 'i', 'k', 'm', 'o', 'q', 's', 'u', 'w', 'y']],
      ['{10..0..-2}', { optimize: false }, ['10', '8', '6', '4', '2', '0']],
      ['{-1..-10..-2}', { optimize: false }, ['-1', '-3', '-5', '-7', '-9']],
      ['{-1..-10..2}', { optimize: false }, ['-1', '-3', '-5', '-7', '-9']],
      ['{-10..-2..2}', { optimize: false }, ['-10', '-8', '-6', '-4', '-2']],
      ['{-2..-10..1}', { optimize: false }, ['-2', '-3', '-4', '-5', '-6', '-7', '-8', '-9', '-10']],
      ['{-2..-10..2}', { optimize: false }, ['-2', '-4', '-6', '-8', '-10']],
      ['{-2..-10..3}', { optimize: false }, ['-2', '-5', '-8']],
      ['{-50..-0..5}', { optimize: false }, ['-50', '-45', '-40', '-35', '-30', '-25', '-20', '-15', '-10', '-5', '0']],
      ['{-9..9..3}', { optimize: false }, ['-9', '-6', '-3', '0', '3', '6', '9']],
      ['{10..1..-2}', { optimize: false }, ['10', '8', '6', '4', '2']],
      ['{100..0..-5}', { optimize: false }, ['100', '95', '90', '85', '80', '75', '70', '65', '60', '55', '50', '45', '40', '35', '30', '25', '20', '15', '10', '5', '0']],
      ['{a..e..2}', { optimize: false }, ['a', 'c', 'e']],
      ['{E..A..2}', { optimize: false }, ['E', 'C', 'A']],
      ['{a..z..2}', { optimize: false }, ['a', 'c', 'e', 'g', 'i', 'k', 'm', 'o', 'q', 's', 'u', 'w', 'y']],
      ['{z..a..-2}', { optimize: false }, ['z', 'x', 'v', 't', 'r', 'p', 'n', 'l', 'j', 'h', 'f', 'd', 'b']],
      ['{z..a..-2}', { optimize: false }, ['z', 'x', 'v', 't', 'r', 'p', 'n', 'l', 'j', 'h', 'f', 'd', 'b']],
      ['{10..0..2}', { optimize: false }, ['10', '8', '6', '4', '2', '0']],
      ['{10..0..-2}', { optimize: false }, ['10', '8', '6', '4', '2', '0']],
      ['{-50..-0..5}', { optimize: false }, ['-50', '-45', '-40', '-35', '-30', '-25', '-20', '-15', '-10', '-5', '0']],
      ['../{1..3}/../foo', {}, ['../1/../foo', '../2/../foo', '../3/../foo']],
      ['../{2..10..2}/../foo', { optimize: false }, ['../2/../foo', '../4/../foo', '../6/../foo', '../8/../foo', '../10/../foo']],
      ['../{1..3}/../{a,b,c}/foo', {}, ['../1/../a/foo', '../1/../b/foo', '../1/../c/foo', '../2/../a/foo', '../2/../b/foo', '../2/../c/foo', '../3/../a/foo', '../3/../b/foo', '../3/../c/foo']],
      ['./{a..z..3}/', { optimize: false }, ['./a/', './d/', './g/', './j/', './m/', './p/', './s/', './v/', './y/']],
      ['./{"x,y"}/{a..z..3}/', { minimatch: false, optimize: false }, ['./{x,y}/a/', './{x,y}/d/', './{x,y}/g/', './{x,y}/j/', './{x,y}/m/', './{x,y}/p/', './{x,y}/s/', './{x,y}/v/', './{x,y}/y/']],
      ['a/{x,y}/{1..5}c{d,e}f.{md,txt}', {}, ['a/x/1cdf.md', 'a/x/1cdf.txt', 'a/x/1cef.md', 'a/x/1cef.txt', 'a/x/2cdf.md', 'a/x/2cdf.txt', 'a/x/2cef.md', 'a/x/2cef.txt', 'a/x/3cdf.md', 'a/x/3cdf.txt', 'a/x/3cef.md', 'a/x/3cef.txt', 'a/x/4cdf.md', 'a/x/4cdf.txt', 'a/x/4cef.md', 'a/x/4cef.txt', 'a/x/5cdf.md', 'a/x/5cdf.txt', 'a/x/5cef.md', 'a/x/5cef.txt', 'a/y/1cdf.md', 'a/y/1cdf.txt', 'a/y/1cef.md', 'a/y/1cef.txt', 'a/y/2cdf.md', 'a/y/2cdf.txt', 'a/y/2cef.md', 'a/y/2cef.txt', 'a/y/3cdf.md', 'a/y/3cdf.txt', 'a/y/3cef.md', 'a/y/3cef.txt', 'a/y/4cdf.md', 'a/y/4cdf.txt', 'a/y/4cef.md', 'a/y/4cef.txt', 'a/y/5cdf.md', 'a/y/5cdf.txt', 'a/y/5cef.md', 'a/y/5cef.txt']],
      ['a/{x,{1..5},y}/c{d}e', {}, ['a/x/c{d}e', 'a/1/c{d}e', 'a/2/c{d}e', 'a/3/c{d}e', 'a/4/c{d}e', 'a/5/c{d}e', 'a/y/c{d}e']]
    ];

    fixtures.forEach(arr => {
      if (typeof arr === 'string') {
        return;
      }

      const options = { ...arr[1] };
      const pattern = arr[0];
      const expected = arr[2];

      if (options.skip !== true) {
        it('should compile: ' + pattern, () => equal(pattern, expected, options));
      }
    });
  });
});
