!
!  fitsfind
!
!  Copyright © 2013, 2018 F.Hroch (hroch@physics.muni.cz)
!
!  This file is part of Munipack.
!
!  Munipack is free software: you can redistribute it and/or modify
!  it under the terms of the GNU General Public License as published by
!  the Free Software Foundation, either version 3 of the License, or
!  (at your option) any later version.
!
!  Munipack is distributed in the hope that it will be useful,
!  but WITHOUT ANY WARRANTY; without even the implied warranty of
!  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!  GNU General Public License for more details.
!
!  You should have received a copy of the GNU General Public License
!  along with Munipack.  If not, see <http://www.gnu.org/licenses/>.


module fitsfind

  use fitsio
  use iso_fortran_env

  implicit none

contains

  subroutine fits_find_read(filename,fkeys,data,readns,saturation,status)

    character(len=*), intent(in) :: filename
    character(len=*), dimension(:), intent(in) :: fkeys
    real, intent(out) :: readns,saturation
    real, dimension(:,:), allocatable, intent(out) :: data
    integer, intent(in out) :: status

    integer, parameter :: DIM = 2
    integer :: naxis, bitpix
    integer, dimension(DIM) :: naxes
    real, parameter :: nullval = 0.0
    integer, parameter :: group = 1
    logical :: anyf, satkey
    character(len=FLEN_CARD) :: buf

    if( status /= 0 ) return

    call ftiopn(15,filename,READONLY,status)
    if( status /= 0 ) goto 666

    call ftgipr(15,DIM,bitpix,naxis,naxes,status)
    if( naxis /= 2 ) then
       write(error_unit,*) 'Find: Only 2D frames are supported.'
       goto 666
    end if

    call ftgkye(15,fkeys(1),saturation,buf,status)
    if( status == KEYWORD_NOT_FOUND ) then
       status = 0
       satkey = .false.
    else
       satkey = .true.
    end if

    call ftgkye(15,fkeys(2),readns,buf,status)
    if( status == KEYWORD_NOT_FOUND ) then
       readns = 0
       status = 0
    end if

    if( status /= 0 ) goto 666

    allocate(data(naxes(1),naxes(2)))
    call ftg2de(15,group,nullval,size(data,1),naxes(1),naxes(2), &
         data,anyf,status)

    if( status == 0 .and. .not. satkey ) then
       if( bitpix > 0 ) then
          saturation = 2**bitpix - 1
       else
          saturation = huge(data) * (1.0 - 10*epsilon(data))
       end if
    end if

666 continue

    call ftclos(15,status)

    if( status /= 0 ) then
       call ftrprt('STDERR',status)
       if( allocated(data) ) deallocate(data)
    end if

  end subroutine fits_find_read



  subroutine fits_find_save(filename,output,fkeys, &
       readns_init,satur_init, nstar, &
       fwhm,threshold, saturation,shrplo,shrphi,rndlo,rndhi, readns, &
       lothresh,  lobad, hibad, hmin, skymod, skyerr, skysig, maxsky, status)

    ! results fills new FITS extension

    character(len=*), intent(in) :: filename, output
    character(len=*), dimension(:), intent(in) :: fkeys

    real, intent(in) :: readns_init,satur_init, fwhm, threshold, saturation,&
         shrplo,shrphi,rndlo,rndhi, lothresh, readns, lobad, hibad, hmin, &
         skymod, skyerr, skysig
    integer, intent(in) :: maxsky, nstar
    integer, intent(in out) :: status

    integer, parameter :: extver = 0, frow = 1, felem = 1
    character(len=FLEN_CARD) :: buf
    character(len=FLEN_VALUE), dimension(5) :: ttype, tform, tunit
    real, dimension(:), allocatable :: xcen,ycen,hstar,round,sharp
    integer :: n, hdutype

    if( status /= 0 ) return

    call fits_open_file(15,filename,output,status)
    if( status /= 0 ) goto 666

    ! update values of readnoise and saturation at first image extension
    call ftmahd(15,1,hdutype,status)

    if( satur_init > 0.0 ) &
         call ftukye(15,fkeys(1),satur_init,5,'[counts] saturation',status)

    if( readns_init > 0.0 ) &
         call ftukye(15,fkeys(2),readns_init,-7,'[ADU] read noise',status)

    ! look for the extension
    call ftmnhd(15,BINARY_TBL,FINDEXTNAME,extver,status)
    if( status == BAD_HDU_NUM ) then
       status = 0
    else if( status == 0 ) then
       ! already presented ? remove it !
       call ftdhdu(15,hdutype,status)
    end if
    if( status /= 0 ) goto 666

    ttype(1) = FITS_COL_X
    ttype(2) = FITS_COL_Y
    ttype(3) = FITS_COL_PEAKRATIO
    ttype(4) = FITS_COL_SHARP
    ttype(5) = FITS_COL_ROUND
    tform = '1D'
    tunit = ''

    call ftibin(15,0,size(ttype),ttype,tform,tunit,FINDEXTNAME,0,status)

    call ftukye(15,FITS_KEY_FWHM,fwhm,-2, &
         '[pix] standard FWHM of objects',status)
    call ftukye(15,FITS_KEY_THRESHOLD,threshold,-2, &
         'threshold in sigmas above background',status)
    call ftukye(15,FITS_KEY_LOWBAD,lobad,-3,'[ADU] low good datum',status)
    call ftukye(15,FITS_KEY_HIGHBAD,hibad,-3,'[ADU] high good datum',status)
    call ftukye(15,FITS_KEY_RNDLO,rndlo,-3,'low round',status)
    call ftukye(15,FITS_KEY_RNDHI,rndhi,-3,'high round',status)
    call ftukye(15,FITS_KEY_SHRPLO,shrplo,-3,'low sharp',status)
    call ftukye(15,FITS_KEY_SHRPHI,shrphi,-3,'high sharp',status)

    call ftpcom(15,'Star detection parameters:',status)

    write(buf,*) 'Saturation (ADU)=',saturation,' (see primary HDU)'
    call ftpcom(15,buf,status)

    write(buf,*) 'Read noise (ADU)=',readns,' (see primary HDU)'
    call ftpcom(15,buf,status)

    write(buf,*) 'Lower threshold (sigma)=',lothresh
    call ftpcom(15,buf,status)

    write(buf,*) 'Levels range (ADU) =',lobad, '..',hibad
    call ftpcom(15,buf,status)

    write(buf,*) 'Hmin (ADU) =',hmin
    call ftpcom(15,buf,status)

    write(buf,*) 'Round range =',rndlo, '..',rndhi
    call ftpcom(15,buf,status)

    write(buf,*) 'Sharp range =',shrplo, '..',shrphi
    call ftpcom(15,buf,status)

    write(buf,*) 'Approximate sky value =',skymod,'+-',skyerr
    call ftpcom(15,buf,status)

    write(buf,*) 'Estimated sky sigma =',skysig
    call ftpcom(15,buf,status)

    write(buf,*) 'Pixels used for sky determination =',maxsky
    call ftpcom(15,buf,status)

    allocate(xcen(nstar),ycen(nstar),sharp(nstar),round(nstar),hstar(nstar))
    do n = 1, nstar
       read(3) xcen(n),ycen(n),hstar(n),sharp(n),round(n)
    end do

    ! sort arrays by height above lower threshold
    call sorter(xcen,ycen,hstar,sharp,round)

    call ftpcle(15,1,frow,felem,size(xcen),xcen,status)
    call ftpcle(15,2,frow,felem,size(ycen),ycen,status)
    call ftpcle(15,3,frow,felem,size(hstar),hstar,status)
    call ftpcle(15,4,frow,felem,size(sharp),sharp,status)
    call ftpcle(15,5,frow,felem,size(round),round,status)

    deallocate(xcen,ycen,hstar,round,sharp)

666 continue

    call fits_close_file(15,status)
    call ftrprt('STDERR',status)

  end subroutine fits_find_save


  subroutine sorter(xcen,ycen,hstar,sharp,round)

    use quicksort

    real, dimension(:),intent(in out) :: xcen,ycen,hstar,sharp,round

    integer, parameter :: rp = selected_real_kind(15)
    real(rp), dimension(:), allocatable :: htmp
    real, dimension(:), allocatable :: tmp
    integer, dimension(:), allocatable :: id, idx
    integer :: i,n,m

    n = size(xcen)
    allocate(tmp(n),htmp(n),id(n),idx(n))
    id = [ (i, i = 1,n) ]
    htmp = hstar

    call qsort(htmp,id)
    ! sorted by hstar in that order: low to high

    ! reverse sort
    m = n + 1
    forall( i = 1:n ) hstar(i) = real(htmp(m - i))
    forall( i = 1:n ) idx(i) = id(m - i)

    tmp = xcen
    xcen = tmp(idx)

    tmp = ycen
    ycen = tmp(idx)

    tmp = sharp
    sharp = tmp(idx)

    tmp = round
    round = tmp(idx)

    deallocate(tmp,htmp,id,idx)

  end subroutine sorter


end module fitsfind
