#!/bin/sh

usage() {
    echo "$0 < tag | rpm | deb | tgz | tbz > [latest]"
    echo "      Use option <tag> to tag the release version, the version in *.spec as input"
    echo "      Use option <rpm> to generate rpm packages"
    echo "      Use option <deb> to generate deb packages"
    echo "      Use option <tgz> to generate *.tar.gz packages"
    echo "      Use option <tbz> to generate *.tar.bz2 packages"
    echo "      Use option [latest] to get the latest code in git, instead version tagged one."

    echo "      Please run it with a git repo directory."
}

if [ $# -lt 1 ]; then
    usage
    exit 1
fi

# check package path
if [ ! -d .git ]; then
    echo "It seems current working dir is not the root of mic2 git repo."
    exit 1
fi

FORMAT=$1

CWD=`pwd`
DISTDIR=./dist
PKGNAME=mic2
VERSION=`cat VERSION`

if [ $# -gt 1 ] && [ $2 = 'latest' ]; then
TREE=HEAD
else
TREE=$VERSION
fi

update_version() {
    echo "version = \"$VERSION\"" > mic/__version__.py
}

tag_release() {
    # add git tag
    git tag -a -m "Tag as ${VERSION}" ${VERSION}
    echo "Add tag ${VERSION} to git repo."
    echo "Do NOT forget to modify debian/changelog before generate deb pkg!"
}

create_deb() {
    DEBVER=`head -n1 debian/changelog  | sed 's/.*(\(.*\)).*/\1/'`
    if [ X$DEBVER != X$VERSION ]; then
        echo 'The version in debian/changelog do not match the version in spec.'
        echo 'Please update debian/changelog at first.'
        exit 0
    fi

    dpkg-buildpackage -rfakeroot -uc -us -tc
    /bin/mv ../${PKGNAME}_${VERSION}* $DISTDIR
}

create_rpm() {
    if [ ! -d /tmp/rpm_mic_packaging ]; then
        mkdir /tmp/rpm_mic_packaging
        cd /tmp/rpm_mic_packaging
        mkdir RPMS SRPMS SPECS SOURCES BUILD
        cd $CWD
    fi

    git archive --format=tar --prefix=${PKGNAME}-${VERSION}/ ${TREE} | bzip2 -9 > \
                    /tmp/rpm_mic_packaging/SOURCES/${PKGNAME}-${VERSION}.tar.bz2
    rpmbuild -ba mic2.spec --clean --define "_topdir /tmp/rpm_mic_packaging"
    find /tmp/rpm_mic_packaging -name '*.rpm' -exec mv {} $DISTDIR/ \;

    rm -rf /tmp/rpm_mic_packaging
}

gen_changelog() {
    GIT_DIR=.git git log > .changelog.tmp && mv .changelog.tmp ChangeLog; rm -f .changelog.tmp
    if [ ! -f ChangeLog ]; then
        touch ChangeLog
        echo 'git directory not found: installing possibly empty changelog .'
    fi
}

create_archive() {
    rm -f ChangeLog
    gen_changelog

    git archive --format=tar --prefix=${PKGNAME}-${VERSION}/ ${TREE} > ${PKGNAME}-${VERSION}.tar
    tar --delete -f mic2-${VERSION}.tar mic2-${VERSION}/debian
    tar -zcf debian.tar.gz debian
    mkdir -p mic2-${VERSION}
    mv ChangeLog mic2-${VERSION}/
    tar --append -f mic2-${VERSION}.tar mic2-${VERSION}
    rm -rf mic2-${VERSION}
}

create_tgz() {
    create_archive
    mv debian.tar.gz $DISTDIR/

    gzip mic2-${VERSION}.tar
    mv mic2-${VERSION}.tar.gz $DISTDIR/
}
create_tbz() {
    create_archive
    mv debian.tar.gz $DISTDIR/

    bzip2 mic2-${VERSION}.tar
    mv mic2-${VERSION}.tar.bz2 $DISTDIR/
}

# cleanup dist dir
/bin/rm -rf $DISTDIR/*
if [ ! -d $DISTDIR ]; then
    mkdir $DISTDIR
fi

update_version

case $FORMAT in
    tag)
        tag_release
        exit 0
        ;;
    deb)
        create_deb
        ;;
    rpm)
        create_rpm
        ;;
    tgz)
        create_tgz
        ;;
    tbz)
        create_tbz
        ;;
    *)
        usage
        ;;
esac

echo "All the generated files are placed at $DISTDIR directory"

