/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */
#ifndef MEDIASCANNER_LOCALE_H
#define MEDIASCANNER_LOCALE_H

// Boost C++
#include <boost/locale/encoding.hpp>

// Standard Library
#include <string>
#include <typeinfo>

namespace mediascanner {

namespace internal {
    void PrintLocaleWarning(const std::locale &loc);
}

bool CheckLocaleFacets(const std::locale &loc = std::locale());

std::locale SetupLocale(const std::string &id = std::string());
std::locale SetupLocale(const std::locale &base, const std::string &id);

inline std::wstring ToUnicode(const std::string &str,
                              const std::locale &loc = std::locale()) {
    try {
        return boost::locale::conv::to_utf<wchar_t>(str, loc);
    } catch(std::bad_cast &) {
        internal::PrintLocaleWarning(loc);
        const std::locale new_locale = SetupLocale(loc, std::string());
        return boost::locale::conv::to_utf<wchar_t>(str, new_locale);
    }
}

inline std::string FromUnicode(const std::wstring &str,
                               const std::locale &loc = std::locale()) {
    try {
        return boost::locale::conv::from_utf(str, loc);
    } catch(std::bad_cast &) {
        internal::PrintLocaleWarning(loc);
        const std::locale new_locale = SetupLocale(loc, std::string());
        return boost::locale::conv::from_utf(str, new_locale);
    }
}

} // namespace mediascanner

#endif // MEDIASCANNER_LOCALE_H
