package org.apache.maven.plugin.ear;

/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import org.apache.maven.artifact.Artifact;
import org.apache.maven.plugin.ear.util.ArtifactTypeMappingService;

import java.util.ArrayList;
import java.util.List;

/**
 * Builds an {@link EarModule} based on an <tt>Artifact</tt>.
 *
 * @author <a href="snicoll@apache.org">Stephane Nicoll</a>
 * @version $Id: EarModuleFactory.java 746754 2009-02-22 16:21:05Z snicoll $
 */
public final class EarModuleFactory
{
    public final static List standardArtifactTypes = new ArrayList();

    static
    {
        standardArtifactTypes.add( "jar" );
        standardArtifactTypes.add( "ejb" );
        standardArtifactTypes.add( "ejb3" );
        standardArtifactTypes.add( "par" );
        standardArtifactTypes.add( "ejb-client" );
        standardArtifactTypes.add( "rar" );
        standardArtifactTypes.add( "war" );
        standardArtifactTypes.add( "sar" );
        standardArtifactTypes.add( "wsr" );
        standardArtifactTypes.add( "har" );
    }

    /**
     * Creates a new {@link EarModule} based on the
     * specified {@link Artifact} and the specified
     * execution configuration.
     *
     * @param artifact                the artifact
     * @param defaultLibBundleDir     the default bundle dir for {@link JarModule}
     * @param includeInApplicationXml should {@link JarModule} be included in application Xml
     * @return an ear module for this artifact
     * @throws UnknownArtifactTypeException if the artifact is not handled
     */
    public static EarModule newEarModule( Artifact artifact, String defaultLibBundleDir,
                                          Boolean includeInApplicationXml )
        throws UnknownArtifactTypeException
    {
        // Get the standard artifact type based on default config and user-defined mapping(s)
        final String artifactType = ArtifactTypeMappingService.getInstance().getStandardType( artifact.getType() );

        if ( "jar".equals( artifactType ) )
        {
            return new JarModule( artifact, defaultLibBundleDir, includeInApplicationXml );
        }
        else if ( "ejb".equals( artifactType ) )
        {
            return new EjbModule( artifact );
        }
        else if ( "ejb3".equals( artifactType ) )
        {
            return new Ejb3Module( artifact );
        }
        else if ( "par".equals( artifactType ) )
        {
            return new ParModule( artifact );
        }
        else if ( "ejb-client".equals( artifactType ) )
        {
            return new EjbClientModule( artifact, null );
        }
        else if ( "rar".equals( artifactType ) )
        {
            return new RarModule( artifact );
        }
        else if ( "war".equals( artifactType ) )
        {
            return new WebModule( artifact );
        }
        else if ( "sar".equals( artifactType ) )
        {
            return new SarModule( artifact );
        }
        else if ( "wsr".equals( artifactType ) )
        {
            return new WsrModule( artifact );
        }
        else if ( "har".equals( artifactType ) )
        {
            return new HarModule( artifact );
        }
        else
        {
            throw new IllegalStateException( "Could not handle artifact type[" + artifactType + "]" );
        }
    }

    /**
     * Returns a list of standard artifact types.
     *
     * @return the standard artifact types
     */
    public static List getStandardArtifactTypes()
    {
        return standardArtifactTypes;
    }

    /**
     * Specify whether the specified type is standard artifact
     * type.
     *
     * @param type the type to check
     * @return true if the specified type is a standard artifact type
     */
    public static boolean isStandardArtifactType( final String type )
    {
        return standardArtifactTypes.contains( type );
    }

}
