/*
 * libspe - A wrapper library to adapt the JSRE SPU usage model to SPUFS
 * Copyright (C) 2005 IBM Corp.
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License,
 * or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *  or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
 *  License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public License
 *   along with this library; if not, write to the Free Software Foundation,
 *   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#define _GNU_SOURCE
#include <errno.h>
#include <libspe.h>
#include <pthread.h>
#include <stdint.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include <sys/mman.h>

#include "spe.h"

#define __PRINTF(fmt, args...) { fprintf(stderr,fmt , ## args); }
#ifdef DEBUG
#define DEBUG_PRINTF(fmt, args...) __PRINTF(fmt , ## args)
#else
#define DEBUG_PRINTF(fmt, args...)
#endif

/* Internal function prototypes. only used internaly.
 */

static void *map_mss_area(speid_t speid)
{
		 struct thread_store *spe = speid;

		 if (spe->mss_mmap_base == MAP_FAILED)
		 {
		 		 spe->mss_mmap_base = do_spe_map_ps_area(spe, 0,
		 		 		 		 		 		 		 spe->fd_mss, 0x1000);
		 }

		 if (spe->mss_mmap_base == MAP_FAILED)
		 {
		 		 return NULL;
		 }

		 return spe->mss_mmap_base;
}

static void *map_mfc_area(speid_t speid)
{
	struct thread_store *spe = speid;

	if (spe->mfc_mmap_base == MAP_FAILED)
	{
		 		 spe->mfc_mmap_base = do_spe_map_ps_area(spe, 0x3000,
		 		 		 		 		 		 		 spe->fd_mfc, 0x1000); 
	}
	
	if (spe->mfc_mmap_base == MAP_FAILED)
	{
		return NULL;
	}

	return spe->mfc_mmap_base;
}

static void *map_cntl_area(speid_t speid)
{
	struct thread_store *spe = speid;

	if (spe->cntl_mmap_base == MAP_FAILED)
	{
		 		 spe->cntl_mmap_base = do_spe_map_ps_area(spe, 0x4000,
		 		 		 		 		 		 		  spe->fd_cntl, 0x1000); 
	}
	
	if (spe->cntl_mmap_base == MAP_FAILED)
	{
		return NULL;
	}

	return spe->cntl_mmap_base;
}

static void *map_signal1_area(speid_t speid)
{
	struct thread_store *spe = speid;

	if (spe->signal1_mmap_base == MAP_FAILED)
	{
		 		 spe->signal1_mmap_base = do_spe_map_ps_area(spe, 0x14000,
		 		 		 		 		 		 		     spe->fd_sig1, 0x1000); 
	}

	if (spe->signal1_mmap_base == MAP_FAILED)
	{
		return NULL;
	}
	
	return spe->signal1_mmap_base;
}

static void *map_signal2_area(speid_t speid)
{
	struct thread_store *spe = speid;

	if (spe->signal2_mmap_base == MAP_FAILED)
	{
		 		 spe->signal2_mmap_base = do_spe_map_ps_area(spe, 0x1c000,
		 		 		 		 		 		 		     spe->fd_sig2, 0x1000); 
	}

	return spe->signal2_mmap_base;
}

void *spe_get_ps_area(speid_t speid, enum ps_area area)
{
	void *base_addr;
	struct thread_store *spe = speid;

	if (!srch_thread(speid))
		return NULL;

	if (!(spe->flags & SPE_MAP_PS)) {
		errno = EACCES;
		return NULL;
	}

	if ( area == SPE_MSSYNC_AREA ) {
		 		 base_addr = map_mss_area(speid);
		 		 if (base_addr == NULL) {
		 		 		 perror("spe_map_ps_area: internal error. cant map mss file.");
		 		 		 errno = EIO;
		 		 		 return NULL;
		 		 }
		 		 return base_addr;
	} else if ( area == SPE_MFC_COMMAND_AREA ) {
		base_addr = map_mfc_area(speid);
		if (base_addr == NULL) {
                        perror("spe_map_ps_area: internal error. cant map mfc file.");
                        errno = EIO;
			return NULL;
		}
		 		 return base_addr ;
	} else if ( area == SPE_CONTROL_AREA ) {
		base_addr = map_cntl_area(speid);
		if (base_addr == NULL) {
                        perror("spe_map_ps_area: internal error. cant map control file.");
                        errno = EIO;
			return NULL;
		}
		 		 return base_addr ;
	} else if ( area == SPE_SIG_NOTIFY_1_AREA ) {
		base_addr = map_signal1_area(speid);
		if (base_addr == NULL) {
                        perror("spe_map_ps_area: internal error. cant map signal1 file.");
                        errno = EIO;
			return NULL;
		}
		 		 return base_addr; 
	} else if ( area == SPE_SIG_NOTIFY_2_AREA ) {
		base_addr = map_signal2_area(speid);
		if (base_addr == NULL) {
                        perror("spe_map_ps_area: internal error. cant map signal2 file.");
                        errno = EIO;
			return NULL;
		}
		 		 return base_addr; 
	}
	
	perror("spe_map_ps_area: Unsupported call of spe_map_ps_area.");
	errno = EINVAL;
	return NULL;
}
