/*
 * @(#)IMethodCode.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.codecoverage.v2;

import org.apache.bcel.classfile.LineNumberTable;
import org.apache.bcel.classfile.Method;
import org.apache.bcel.generic.Instruction;


/**
 * Contains the BCEL bytecode for a method, and provides helper methods for
 * generating the marks for certain bytecode instructions.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version   $Date: 2004/04/15 05:48:25 $
 * @since     December 15, 2002
 */
public interface IMethodCode
{
    /**
     * Returns the original BCEL Method object.
     *
     * @return the original BCEL Method object
     */
    public Method getOriginalMethod();
    
    
    /**
     * Returns the line number table for the original BCEL method object.
     *
     * @return the LineNumberTable, or <tt>null</tt> if there isn't one
     *      for this method.
     */
    public LineNumberTable getLineNumberTable();
    
    
    /**
     * A helper to get the method name.
     *
     * @return the method name
     */
    public String getMethodName();
    
    
    /**
     * A helper to get the class name.
     *
     * @return the class name
     */
    public String getClassName();
    
    
    /**
     * Returns the number of bytecode instructions in the method.
     *
     * @return the number of bytecode instructions
     */
    public int getInstructionCount();
    
    
    /**
     * Returns the bytecode instruction at the given index.  If the index
     * is out of range (&lt; 0 or &gt;= <tt>getInstructionCount()</tt>),
     * then a <tt>IndexOutOfBoundsException</tt> is thrown.
     *
     * @param index the 0-based index of the method's instruction list
     * @return the instruction at <tt>index</tt>
     */
    public Instruction getInstructionAt( int index );
    
    
    /**
     * Marks an instruction for coverage analysis.  If the index
     * is out of range (&lt; 0 or &gt;= <tt>getInstructionCount()</tt>),
     * then a <tt>IndexOutOfBoundsException</tt> is thrown.
     *
     * @param index the 0-based index of the method's instruction list
     * @param meta meta-data the analysis module records in association with
     *             the mark.  This cannot be <tt>null</tt>.
     * @exception IllegalArgumentException if <tt>meta</tt> is <tt>null</tt>.
     * @exception IndexOutOfBoundsException if <tt>index</tt> is not within
     *             the valid range.
     */
    public void markInstruction( int index, IAnalysisMetaData meta );
}

