/*
 *  This file is part of GNUDoQ, Copyright (C) 2005-2006 Luc Vo Van
 *
 *   GNUDoQ is free software; you can redistribute it and/or modify it
 *   under the terms of the GNU General Public License as published by the
 *   Free Software Foundation; either version 2, or (at your option) any
 *   later version.
 *
 *   GNUDoQ is distributed in the hope that it will be useful, but WITHOUT
 *   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *   for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with GNUDoQ; see the file COPYING.  If not, write to
 *   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 *   Boston, MA 02111-1307, USA.
 *
 *   Class          :   Main UI
 *   Author         :   Luc Vo Van
 *   Original Date  :   September 2005
 *
 **/


#ifndef GNUDOQ_H
#define GNUDOQ_H

#include "ui_GNUDoQForm.h"
#include "GNUDoQBoxWidget.H"
#include <QMainWindow>
#include <QObject>
#include <QPrinter>

#include "sudoku.H"
using sudoku::Sudoku;
#include "sudoku-solve.H"

/**  The main GNUDoQ UI class
 *  This class handles UI events and links them to the GNUDoku engine
 *  if necessary.
 */
class GNUDoQ : public QWidget 
{
	Q_OBJECT

	public:
		/** Class constructor
		 * Initializes a new instance of GNUDoQ
		 * 
		 * @param parent Parent widget
		 */
		GNUDoQ(QWidget *parent = 0);

        private slots:
		/**
		 * Generates a new puzzle when the Generate button is clicked
		 */
                void on_btnGenerate_clicked();

		/**
		 * Solves the puzzle when the Solve button is clicked
		 */
		void on_btnSolve_clicked();

		/**
		 * Clears the player input when the Clear button is clicked
		 */
		void on_btnClear_clicked();

		/**
		 * Prints out the current puzzle to the printer
		 */
		void on_btnPrint_clicked();
		
		/**
		 * Quits the program
		 */
                void on_btnQuit_clicked();
		
		/**
		 * Generates a puzzle from the puzzle code when RETURN is pressed
		 * on the QLineEdit lePuzzleCode
		 */
		void on_lePuzzleCode_returnPressed();

		/**
		 * Shows the about form when the About button is clicked
		 */
		void on_btnLogoAbout_clicked();

		/**
		 * Changes the difficulty level and displays it
		 */
                void on_sldDifficulty_valueChanged();

		/**
		 * Loads the last saved puzzles with QSettings
		 */
		void LoadPuzzle();

		/**
		 * Saves the current puzzle to disk with QSettings
		 */
		void SavePuzzle();

		/**
		 * Checks if the player has inputted all his moves, in which case
		 * the Solve button will display "Verify" instead of "Solve".
		 */
		void TestSolveOrVerify();

	private:
		/**
		 * Generates a puzzle given a difficulty and a seed
		 * @param difficulty difficulty of the puzzle, from 0.0 to 1.0
		 * @param seed seed from which the puzzle will be constructed, use
		 *             a random number to generate a random puzzle
		 */
		void GeneratePuzzle(float const difficulty, int const seed);

		/**
		 * Solves the currently displayed puzzle, using the current
		 * player inputs
		 */
		void SolvePuzzle();

		/**
		 * Gets the values from the player editable boxes (inputs)
		 * @param top GNUDoku variable
		 * @param flags GNUDoku variable
		 * @param visited GNUDoku variable
		 */
		int GetFromBoxes(Sudoku::attempt *top,
			Sudoku::flags *flags,
			char visited[]);

		/**
		 * Fills the boxes in the grid with the puzzle's values.
		 * @param stack GNUDoku variable
		 * @param visited GNUDoku variable
		 */
		void FillBoxes(const Sudoku::attempt stack[],
			       const char visited[]);

		/**
		 * Disconnects all receivers of the grid's boxes signals. This
		 * is used to speed up complete updates of the grid.
		 */
		void DisconnectBoxesValueChanged();

		/**
		 * Connects all receivers to the grid's boxes signals.
		 */
		void ConnectBoxesValueChanged();

		/**
		 * Clears all the boxes of the grid.
		 */
		void ClearBoxes();
		
		/**
		 * Resets the player's inputs to 0. Equivalent to restarting
		 * the puzzle.
		 */
		void ResetPuzzle();

		/**
		 * Returns a QString representation of the grid.
		 * @return QString representation of the grid
		 */
		QString DumpToString();

		// The original code for this puzzle
		QString m_sCurrentCode;

		// The UI made with Designer
		Ui::GNUDoQForm ui;
		
		// The QWidgets that make up the grid
		GNUDoQBoxWidget* m_Boxes[9][9];

		// The printing object
		QPrinter printer;

		// The GNUDoQ version
		static QString VERSION;
};

#endif
