/*
 * Copyright 1999-2006 University of Chicago
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * Wherever applicable, SQLRowCount() acts as a check that the previous
 * SQLExecDirect() modified >0 rows. In particular, this enables working
 * with ODBC 3.0 spec where SQLExecDirect() returns SQ_NO_DATA if no
 * rows were affected, whereas previously, SQL_ERROR was returned.
 * Fixes bug number (3403)
 */

/*
 * The ODBC specification defines flags for SQL_C_CHAR and SQL_C_LONG among
 * other C type identifiers. The above identifiers indicate ODBC C types of
 * SQLCHAR* and SQLINTEGER respectively. To this point, we have used variables
 * of char* and int, respectively, instead of the ODBC C types. This has not
 * caused a problem on 32-bit platforms but when moving to 64-bit platforms
 * the use of int instead of SQLINTEGER identified by SQL_C_LONG now causes
 * a problem. Unfortunately, at the time of this fix the open source software
 * (namely iodbc, MySQL Connector/ODBC, and psqlODBC) fail to implement integer
 * types consistently (See associated bug record for details). To workaround
 * these inconsistencies, we must use long variables instead of SQLINTEGER
 * variables in conjunction with the SQL_C_LONG identifier. At some future
 * point in time, when inconsistencies are worked out of the ODBC open source
 * software, it would be ideal to change these long variables to SQLINTEGER
 * variables. Though it does not cause a problem, we will change char*
 * variables to SQLCHAR*, where applicable.
 * Fixes bug number (3846)
 */

#include "globus_rls_client.h"
#include "misc.h"
#include "db.h"
#include "bloom.h"
#include <sys/time.h>
#include <sql.h>
#include <sqlext.h>
#include <syslog.h>
#include <string.h>
#include <ctype.h>

#ifdef COUNTDB
#define SQLExecDirect(s, b, f)	mysqlexecdirect((s), (b), (f))
static SQLRETURN		mysqlexecdirect(SQLHSTMT s, SQLCHAR *b, int f);
int				dbstatements = 0;
#endif

#define BIGBUF	10000

#define PS(s)		((s) ? (s) : "NULL")

#define SQLOK(r)	((r) == SQL_SUCCESS || (r) == SQL_SUCCESS_WITH_INFO)
#define SQLNODATAOK(r)	((r) == SQL_SUCCESS || (r) == SQL_SUCCESS_WITH_INFO || (r) == SQL_NO_DATA || (r) == SQL_NO_DATA_FOUND)

extern int		loglevel;
static globus_mutex_t	rlidbmtx;

static char		*objtype2table(globus_rls_obj_type_t objtype,
				       int *nferr);
static char		*valuetype2table(globus_rls_attr_type_t type);

typedef struct {
  SQLHENV		env;
  HDBC			dbc;
  SQLHSTMT		stmt;
} db_handle_t;

#define IDLISTSIZE	10000
typedef struct idlist_ {
  SQLINTEGER	ids[IDLISTSIZE];
  SQLINTEGER	vals[IDLISTSIZE]; /* Used when removing attrs	*/
  int			n;
  struct idlist_	*nxt;
} IDLIST;

static void		begintran(db_handle_t *h, int *adjcount);
static int		endtran(db_handle_t *h, SQLSMALLINT comptype,
				char *errmsg, int *adjcount);
static int		query_ret1(db_handle_t *h, char *buf, char *key,
				   db_str1_callback_t cb,void *a,char *errmsg);
static int		query_ret2(db_handle_t *h, char *buf, char *key,
				   db_str2_callback_t cb,void *a,char *errmsg);
static int		query_ret3(db_handle_t *h, char *buf, char *key,
				   db_str3_callback_t cb,void *a,char *errmsg);
static int		query_ret4(db_handle_t *h, char *buf, char *key,
				   db_str4_callback_t cb,void *a,char *errmsg);
static int		updateref(db_handle_t *h, int tidx, char *name,
				  int adjust, int insert, SQLINTEGER *id,
				  SQLINTEGER *ref, char *errmsg, int *adjcount);
static void		gettablecount(db_handle_t *h, int tidx);
static SQLRETURN	deleteattrvals(db_handle_t *h,
				globus_rls_obj_type_t objtype, SQLINTEGER obj_id);
static int		attrinfo(db_handle_t *h, globus_rls_obj_type_t objtype,
			  char *name, char *key, SQLINTEGER *attr_id,
			  globus_rls_attr_type_t *type, SQLINTEGER *obj_id,
			  char **valtab, char *errmsg);
static int		getid(db_handle_t *h, char *s, char *tab, int nferr,
			      SQLINTEGER *id, char *errmsg);
static IDLIST		*newidlist(IDLIST *list);
static void		addid(IDLIST **listp, SQLINTEGER id, SQLINTEGER val);
static void		freeidlist(IDLIST *list);
static int		seterr_insert(db_handle_t *h, SQLRETURN r, char *table,
				      char *name, int err, char *errmsg);
static int		seterr_insertattr(db_handle_t *h, SQLRETURN r, char *table,
					  int obj_id, int attr_id, int err,
					  char *errmsg);
static int		seterr_insertmap(db_handle_t *h, SQLRETURN r, char *c1,
				int id1, char *c2, int id2, char *c3, int id3,
				char *errmsg);
static char		*odbcerr(SQLRETURN rc, SQLSMALLINT htype,
				 SQLHANDLE h, char *errmsg);
static char		*sqllimit(int offset, int reslimit, char *buf);
static char		*todate(char *datestr, char *buf);
static char		*tochar(char *field, char *buf);

/*
 * Table information (name, error if not found, rec count, must match
 * T_xxx defines in db.h.
 */
TABLE			table[] = {
  { "t_lfn", GLOBUS_RLS_LFN_NEXIST, 0 },
  { "t_pfn", GLOBUS_RLS_PFN_NEXIST, 0 },
  { "t_map", GLOBUS_RLS_MAPPING_NEXIST, 0 },
  { "t_lfn", GLOBUS_RLS_LFN_NEXIST, 0 },
  { "t_lrc", GLOBUS_RLS_LRC_NEXIST, 0 },
  { "t_sender", GLOBUS_RLS_LRC_NEXIST, 0 },
  { "t_map", GLOBUS_RLS_MAPPING_NEXIST, 0 },
};
#define T_NUM	(sizeof(table) / sizeof(TABLE))

static globus_mutex_t	tablemtx;

static char	dbmsname[256];
static int	firsttime = 1;
static int	ismysql = 0;
static int	ispostgres = 0;
static int	isoracle = 0;

void
db_init()

{
  globus_mutex_init(&tablemtx, GLOBUS_NULL);
  globus_mutex_init(&rlidbmtx, GLOBUS_NULL);
}

int
db_open(char *dbname, char *user, char *pwd, int lrc_server, void **hp,
	char *errmsg)

{
  db_handle_t	*h;
  SQLRETURN	r;
  int		connected = 0;
  static int	lrccounts = 0;
  static int	rlicounts = 0;
  SWORD		len;
  int		i;

  if (loglevel > 1)
    logit(LOG_DEBUG, "db_open: %s %s", dbname, user);
  if ((h = (db_handle_t *) globus_libc_malloc(sizeof(db_handle_t))) == NULL) {
    if (loglevel)
      logit(LOG_DEBUG, "db_open(%s): No memory", dbname);
    return GLOBUS_RLS_NOMEMORY;
  }

  h->env = NULL;
  h->dbc = NULL;
  h->stmt = NULL;

  r = SQLAllocHandle(SQL_HANDLE_ENV, SQL_NULL_HANDLE, &h->env);
  if (!SQLOK(r)) {
    odbcerr(r, SQL_HANDLE_ENV, SQL_NULL_HANDLE, errmsg);
    goto error;
  }
  SQLSetEnvAttr(h->env, SQL_ATTR_ODBC_VERSION, (SQLPOINTER) SQL_OV_ODBC3,
		SQL_IS_UINTEGER);

  r = SQLAllocHandle(SQL_HANDLE_DBC, h->env, &h->dbc);
  if (!SQLOK(r)) {
    odbcerr(r, SQL_HANDLE_DBC, h->env, errmsg);
    goto error;
  }

  r = SQLConnect(h->dbc, (u_char *) dbname, SQL_NTS, (u_char *) user,
		 SQL_NTS, (u_char *) pwd, SQL_NTS);
  if (!SQLOK(r)) {
    odbcerr(r, SQL_HANDLE_DBC, h->dbc, errmsg);
    goto error;
  }
  connected++;

  r = SQLAllocHandle(SQL_HANDLE_STMT, h->dbc, &h->stmt);
  if (!SQLOK(r)) {
    odbcerr(r, SQL_HANDLE_STMT, h->dbc, errmsg);
    goto error;
  }

  globus_mutex_lock(&tablemtx);
  if (firsttime) {
    firsttime = 0;
    r = SQLGetInfo(h->dbc, SQL_DBMS_NAME, dbmsname, sizeof(dbmsname), &len);
    if (!SQLOK(r)) {
      odbcerr(r, SQL_HANDLE_STMT, h->dbc, errmsg);
      logit(LOG_INFO, "db_open: Unable to get DBMS name: %s", errmsg);
    } else {
      if (loglevel)
	logit(LOG_DEBUG, "db_open: DBMS %s", dbmsname);
      for (i = 0; dbmsname[i]; i++)
	if (isupper(dbmsname[i]))
	  dbmsname[i] = tolower(dbmsname[i]);
      if (strstr(dbmsname, "mysql"))
	ismysql = 1;
      else if (strstr(dbmsname, "postgres"))
	ispostgres = 1;
      else if (strstr(dbmsname, "oracle"))
	isoracle = 1;
    }
  }

  if (lrc_server) {
    if (lrccounts++ == 0) {
      gettablecount(h, T_LRCLFN);
      gettablecount(h, T_LRCPFN);
      gettablecount(h, T_LRCMAP);
    }
  } else {
    if (rlicounts++ == 0) {
      gettablecount(h, T_RLILFN);
      gettablecount(h, T_RLILRC);
      gettablecount(h, T_RLISENDER);
      gettablecount(h, T_RLIMAP);
    }
  }
  globus_mutex_unlock(&tablemtx);

  *hp = h;
  return GLOBUS_RLS_SUCCESS;

 error:
  if (h->stmt)
    SQLFreeHandle(SQL_HANDLE_STMT, h->stmt);
  if (connected)
    SQLDisconnect(h->dbc);
  if (h->dbc)
    SQLFreeHandle(SQL_HANDLE_DBC, h->dbc);
  if (h->env)
    SQLFreeHandle(SQL_HANDLE_ENV, h->env);
  globus_libc_free(h);
  if (loglevel)
    logit(LOG_DEBUG, "db_open(%s): %s", dbname, errmsg);
  return GLOBUS_RLS_DBERROR;
}

void
db_close(void *hv)

{
  db_handle_t	*h = (db_handle_t *) hv;
  SQLRETURN	r;
  char		errmsg[MAXERRMSG];

  if (loglevel > 1)
    logit(LOG_DEBUG, "db_close:");
  r = SQLFreeHandle(SQL_HANDLE_STMT, h->stmt);
  if (!SQLOK(r))
    logit(LOG_WARNING, "db_close: %s",
	  odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg));
  r = SQLDisconnect(h->dbc);
  if (!SQLOK(r))
    logit(LOG_WARNING, "db_close: %s",
	  odbcerr(r, SQL_HANDLE_DBC, h->dbc, errmsg));
  r = SQLFreeHandle(SQL_HANDLE_DBC, h->dbc);
  if (!SQLOK(r))
    logit(LOG_WARNING, "db_close: %s",
	  odbcerr(r, SQL_HANDLE_DBC, h->dbc, errmsg));
  r = SQLFreeHandle(SQL_HANDLE_ENV, h->env);
  if (!SQLOK(r))
    logit(LOG_WARNING, "db_close: %s",
	  odbcerr(r, SQL_HANDLE_ENV, h->env, errmsg));
  globus_libc_free(h);
}

int
db_is_alive(void *hv)

{
  db_handle_t	*h = (db_handle_t *) hv;
  SQLRETURN     r;
  char          errmsg[MAXERRMSG];
  SQLINTEGER    connection_dead;
  SQLINTEGER    buflen;

  r = SQLGetConnectAttr(h->dbc, SQL_ATTR_CONNECTION_DEAD,
          &connection_dead, SQL_IS_POINTER, &buflen);
  if (!SQLOK(r)) {
    logit(LOG_WARNING, "db_is_alive: %s",
            odbcerr(r, SQL_HANDLE_ENV, h->env, errmsg));
    return GLOBUS_RLS_DBERROR;
  }

  if (loglevel > 1)
    if (connection_dead == SQL_CD_TRUE)
      logit(LOG_DEBUG, "db_is_alive: connection dead");

  /* 
   * SQL_CD_TRUE  == connection is dead
   * SQL_CD_FALSE == connection is alive
   */
  return connection_dead == SQL_CD_TRUE ?
      GLOBUS_RLS_DBERROR : GLOBUS_RLS_SUCCESS;
}

int
db_update_add(void *hv, char *rli_url, int flags, char *pattern, char *errmsg)

{
  db_handle_t	*h = (db_handle_t *) hv;
  char		    buf[BIGBUF];
  SQLINTEGER	rli_id = 0;
  int		    rc;
  SQLRETURN	    r;

  if (loglevel > 1)
    logit(LOG_DEBUG, "db_update_add: %s %X %s", rli_url, flags, PS(pattern));

  begintran(h, NULL);
  rc = getid(h, rli_url, "t_rli", GLOBUS_RLS_RLI_NEXIST, &rli_id, errmsg);
  if (rc == GLOBUS_RLS_RLI_NEXIST) {
    snprintf(buf, BIGBUF, "insert into t_rli (name, flags) values('%s',%d)",
	    rli_url, flags);
    r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
    if (!SQLOK(r)) {
      odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
      rc = GLOBUS_RLS_DBERROR;
    } else
      rc = getid(h, rli_url, "t_rli", GLOBUS_RLS_RLI_NEXIST, &rli_id, errmsg);
  } else if (rc == GLOBUS_RLS_SUCCESS && !pattern) {
    rc = GLOBUS_RLS_RLI_EXIST;
    strncpy(errmsg, rli_url, MAXERRMSG);
    goto error;
  }

  if (rc != GLOBUS_RLS_SUCCESS)
    goto error;

  if (pattern) {
    snprintf(buf, BIGBUF, "insert into t_rlipartition (rli_id, pattern) values (%d, '%s')", (int) rli_id, pattern);
    r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
    if (!SQLOK(r)) {
      odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
      rc = GLOBUS_RLS_DBERROR;
      goto error;
    }
  }

  return endtran(h, SQL_COMMIT, errmsg, NULL);

 error:
  endtran(h, SQL_ROLLBACK, NULL, NULL);
  return rc;
}

int
db_update_delete(void *hv, char *rli_url, char *pattern, char *errmsg)

{
  db_handle_t	*h = (db_handle_t *) hv;
  char		buf[BIGBUF];
  SQLINTEGER rli_id;
  int		rc;
  SQLRETURN	r;
  SQLLEN	rows    = 0;

  if (loglevel > 1)
    logit(LOG_DEBUG, "db_update_delete: %s %s", rli_url, PS(pattern));

  begintran(h, NULL);
  if ((rc = getid(h, rli_url, "t_rli", GLOBUS_RLS_RLI_NEXIST, &rli_id,
		  errmsg)) != GLOBUS_RLS_SUCCESS)
    goto error;
  if (pattern)
    snprintf(buf, BIGBUF, "delete from t_rlipartition where rli_id = %d and pattern = '%s'", (int) rli_id, pattern);
  else
    snprintf(buf, BIGBUF, "delete from t_rlipartition where rli_id = %d", (int) rli_id);
  r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
  if (!(SQLNODATAOK(r))) {
    odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
    rc = GLOBUS_RLS_DBERROR;
    goto error;
  }
  if (pattern) {
    if (SQLOK(r)) {
      SQLRowCount(h->stmt, &rows);
    } else {
      rows = 0;
    }
    if (rows != 1) {
      snprintf(errmsg, MAXERRMSG, "%s: %s", rli_url, pattern);
      rc = GLOBUS_RLS_RLI_NEXIST;
      goto error;
    }
  } else {	/* If no pattern remove rli from t_rli	*/
    snprintf(buf, BIGBUF, "delete from t_rli where id = %d", (int) rli_id);
    r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
    if (!(SQLOK(r))) {
      odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
      rc = GLOBUS_RLS_DBERROR;
      goto error;
    }
    SQLRowCount(h->stmt, &rows);
    if (rows != 1) {
      strncpy(errmsg, rli_url, MAXERRMSG);
      rc = GLOBUS_RLS_RLI_NEXIST;
      goto error;
    }
  }
  return endtran(h, SQL_COMMIT, errmsg, NULL);

 error:
  endtran(h, SQL_ROLLBACK, NULL, NULL);
  return rc;
}

int
db_update_get_part(void *hv, char *rli, char *pattern, db_str3_callback_t cb,
		    void *a, char *errmsg)

{
  db_handle_t	*h = (db_handle_t *) hv;
  char		buf[BIGBUF];
  char		key[BIGBUF];

  if (loglevel > 1)
    logit(LOG_DEBUG, "db_update_get_part: %s %s", PS(rli), PS(pattern));

  if (rli)
    if (pattern)	/* Search for both rli and pattern	*/
      snprintf(buf, BIGBUF, "select t_rli.name, t_rli.flags, t_rlipartition.pattern from t_rli, t_rlipartition where t_rli.name = '%s' and t_rlipartition.pattern = '%s' and t_rli.id = t_rlipartition.rli_id", rli, pattern);
    else		/* Search for rli			*/
      snprintf(buf, BIGBUF, "select t_rli.name, t_rli.flags, t_rlipartition.pattern from t_rli left join t_rlipartition on t_rli.id = t_rlipartition.rli_id where t_rli.name = '%s'", rli);
  else			/* Search for pattern			*/
    if (pattern)
      snprintf(buf, BIGBUF, "select t_rli.name, t_rli.flags, t_rlipartition.pattern from t_rli, t_rlipartition where t_rlipartition.pattern = '%s' and t_rli.id = t_rlipartition.rli_id", pattern);
    else		/* Return all records			*/
      snprintf(buf, BIGBUF, "select t_rli.name, t_rli.flags, t_rlipartition.pattern from t_rli left join t_rlipartition on t_rli.id = t_rlipartition.rli_id");

  snprintf(key, BIGBUF, "%s,%s", PS(rli), PS(pattern));
  return query_ret3(h, buf, key, cb, a, errmsg);
}

/*
 * Add attribute value to database.
 */
int
db_attr_add(void *hv, char *key, globus_rls_obj_type_t objtype,
	    globus_rls_attr_type_t type, char *name, char *sval, char *errmsg)

{
  db_handle_t			*h = (db_handle_t *) hv;
  char				buf[BIGBUF];
  SQLRETURN			r;
  int				rc;
  SQLINTEGER		attr_id;
  globus_rls_attr_type_t	atype;
  SQLINTEGER		obj_id;
  char				*valtab;
  char				dbuf[100];

  if (loglevel > 1)
    logit(LOG_DEBUG, "db_attr_add: %s %d %d %s %s", key, objtype, type,
	  name, sval);

  if (type == globus_rls_attr_type_str && strlen(sval) + 1 > MAXDBSTR) {
    strncpy(errmsg, sval, MAXERRMSG);
    return GLOBUS_RLS_BADARG;
  }

  begintran(h, NULL);

  rc = attrinfo(h, objtype, name, key, &attr_id, &atype, &obj_id, &valtab,
		errmsg);
  if (rc != GLOBUS_RLS_SUCCESS)
    goto error;
  /*
   * The type argument was passed by the user and presumably is how the value
   * was encoded in sval.  Check that the type of this attribute in the
   * DB matches the encoding type.
   */
  if (atype != type) {
    sprintf(errmsg, "specified %d should be %d", type, atype);
    rc = GLOBUS_RLS_INV_ATTR_TYPE;
    goto error;
  }
    
  switch (type) {	  /* Now create appropriate value record	*/
    case globus_rls_attr_type_date:
      snprintf(buf, BIGBUF,
	       "insert into %s (obj_id, attr_id, value) values(%d, %d, %s)",
	       valtab, (int) obj_id, (int) attr_id, todate(sval, dbuf));
      break;
    case globus_rls_attr_type_flt:
      snprintf(buf, BIGBUF,
	       "insert into %s (obj_id, attr_id, value) values(%d, %d, %f)",
	       valtab, (int) obj_id, (int) attr_id, atof(sval));
      break;
    case globus_rls_attr_type_int:
      snprintf(buf, BIGBUF,
	      "insert into %s (obj_id, attr_id, value) values(%d, %d, %d)",
	       valtab, (int) obj_id, (int) attr_id, atoi(sval));
      break;
    case globus_rls_attr_type_str:
      snprintf(buf, BIGBUF,
	       "insert into %s (obj_id, attr_id, value) values(%d, %d, '%s')",
	       valtab, (int) obj_id, (int) attr_id, sval);
      break;
  }
  r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
  if (!SQLOK(r)) {
    rc = seterr_insertattr(h, r, valtab, (int) obj_id, (int) attr_id, 
            GLOBUS_RLS_ATTR_EXIST, errmsg);
    goto error;
  }

  return endtran(h, SQL_COMMIT, errmsg, NULL);

 error:
  endtran(h, SQL_ROLLBACK, NULL, NULL);
  return rc;
}

int
db_attr_create(void *hv, char *name, globus_rls_obj_type_t objtype,
	       globus_rls_attr_type_t type, char *errmsg)

{
  db_handle_t	*h = (db_handle_t *) hv;
  char		buf[BIGBUF];
  SQLRETURN	r;
  int		rc;

  if (loglevel > 1)
    logit(LOG_DEBUG, "db_attr_create: %s %d %d", name, objtype, type);

  if (strlen(name) + 1 > MAXDBSTR) {
    strncpy(errmsg, name, MAXERRMSG);
    return GLOBUS_RLS_BADARG;
  }

  begintran(h, NULL);

  snprintf(buf, BIGBUF,
	   "insert into t_attribute (name,objtype,type) values ('%s',%d,%d)",
	   name, objtype, type);
  r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
  if (!SQLOK(r)) {
    rc = seterr_insert(h,r,"t_attribute",name,GLOBUS_RLS_ATTR_EXIST,errmsg);
    goto error;
  }

  return endtran(h, SQL_COMMIT, errmsg, NULL);

 error:
  endtran(h, SQL_ROLLBACK, NULL, NULL);
  return rc;
}

int
db_attr_delete(void *hv, char *name, globus_rls_obj_type_t objtype,
	       globus_bool_t clearvalues, char *errmsg)

{
  db_handle_t			*h = (db_handle_t *) hv;
  char				buf[BIGBUF];
  SQLRETURN			r;
  int				rc;
  long				attr_id = 0;
  long              type    = 0;
  SQLLEN			len;
  char				*valtab;
  long				count   = 0;

  if (loglevel > 1)
    logit(LOG_DEBUG, "db_attr_delete: %s %d", name, objtype);

  rc = GLOBUS_RLS_SUCCESS;
  begintran(h, NULL);
  snprintf(buf, BIGBUF, "select id, type from t_attribute where name = '%s' and objtype = %d",
	   name, objtype);
  r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
  if (SQLOK(r))
    r = SQLBindCol(h->stmt, 1, SQL_C_LONG, &attr_id, sizeof(attr_id), &len);
  if (SQLOK(r))
    r = SQLBindCol(h->stmt, 2, SQL_C_LONG, &type, sizeof(type), &len);
  if (SQLOK(r))
    r = SQLFetch(h->stmt);
  SQLFreeStmt(h->stmt, SQL_CLOSE);
  SQLFreeStmt(h->stmt, SQL_UNBIND);
  if (r == SQL_NO_DATA) {
    rc = GLOBUS_RLS_ATTR_NEXIST;
    strncpy(errmsg, name, MAXERRMSG);
    goto error;
  } else if (!SQLOK(r))
    goto error;

  if ((valtab = valuetype2table((globus_rls_attr_type_t) type)) == NULL) {
    sprintf(errmsg, "%d", (int) type);
    rc = GLOBUS_RLS_INV_ATTR_TYPE;
    goto error;
  }
  if (clearvalues) {
    snprintf(buf,BIGBUF,"delete from %s where attr_id = %d", valtab, (int) attr_id);
    r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
    if (!SQLNODATAOK(r))
      goto error;
  } else {
    snprintf(buf, BIGBUF, "select count(*) from %s where attr_id = %d",
	     valtab, (int) attr_id);
    r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
    if (!SQLOK(r))
      goto error;
    r = SQLBindCol(h->stmt, 1, SQL_C_LONG, &count, sizeof(count), &len);
    if (SQLOK(r))
      r = SQLFetch(h->stmt);
    SQLFreeStmt(h->stmt, SQL_CLOSE);
    SQLFreeStmt(h->stmt, SQL_UNBIND);
    if (!SQLOK(r))
      goto error;
    if (count) {
      logit(LOG_INFO, "db_attr_delete(%s): Attribute in use in %d records",
	    name, (int) count);
      strncpy(errmsg, name, MAXERRMSG);
      rc = GLOBUS_RLS_ATTR_INUSE;
      goto error;
    }
  }

  snprintf(buf, BIGBUF,
	   "delete from t_attribute where name = '%s' and objtype = %d",
	   name, objtype);
  r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
  if (!SQLOK(r)) {
    rc = seterr_insert(h,r,"t_attribute",name,GLOBUS_RLS_ATTR_EXIST,errmsg);
    goto error;
  }

  return endtran(h, SQL_COMMIT, errmsg, NULL);

 error:
  endtran(h, SQL_ROLLBACK, NULL, NULL);
  if (rc == GLOBUS_RLS_SUCCESS) {
    odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
    rc = GLOBUS_RLS_DBERROR;
  }
  return rc;
}

int
db_attr_get(void *hv, char *name, globus_rls_obj_type_t objtype,
	    db_str2_callback_t cb, void *a, char *errmsg)

{
  db_handle_t	*h = (db_handle_t *) hv;
  char		buf[BIGBUF];

  if (loglevel > 1)
    logit(LOG_DEBUG, "db_attr_get: %s %d", PS(name), objtype);
  if (name && *name)
    snprintf(buf, BIGBUF, "select name, type from t_attribute where name = '%s' and objtype = %d", name, objtype);
  else
    snprintf(buf, BIGBUF, "select name, type from t_attribute where objtype = %d", objtype);
  return query_ret2(h, buf, name, cb, a, errmsg);
}

int
db_attr_modify(void *hv, char *key, char *name, globus_rls_obj_type_t objtype,
	       globus_rls_attr_type_t type, char *sval, char *errmsg)

{
  db_handle_t			*h = (db_handle_t *) hv;
  char				buf[BIGBUF];
  SQLRETURN			r;
  int				rc;
  SQLINTEGER		attr_id;
  globus_rls_attr_type_t	atype;
  SQLINTEGER		obj_id;
  char				*valtab;
  char				dbuf[100];

  if (loglevel > 1)
    logit(LOG_DEBUG, "db_attr_modify: %s %s %d %d %s", key, name, objtype,
	  type, sval);

  begintran(h, NULL);

  rc = attrinfo(h, objtype, name, key, &attr_id, &atype, &obj_id, &valtab,
		errmsg);
  if (rc != GLOBUS_RLS_SUCCESS)
    goto error;

  if (atype != type) {
    sprintf(errmsg, "specified %d should be %d", type, atype);
    rc = GLOBUS_RLS_INV_ATTR_TYPE;
    goto error;
  }

  switch (type) {
    case globus_rls_attr_type_date:
      snprintf(buf, BIGBUF,
	       "update %s set value = %s where obj_id = %d and attr_id = %d",
	       valtab, todate(sval, dbuf), (int) obj_id, (int) attr_id);
      break;
    case globus_rls_attr_type_flt:
      snprintf(buf, BIGBUF,
	       "update %s set value = %f where obj_id = %d and attr_id = %d",
	       valtab, atof(sval), (int) obj_id, (int) attr_id);
      break;
    case globus_rls_attr_type_int:
      snprintf(buf, BIGBUF,
	      "update %s set value = %d where obj_id = %d and attr_id = %d",
	       valtab, atoi(sval), (int) obj_id, (int) attr_id);
      break;
    case globus_rls_attr_type_str:
      snprintf(buf, BIGBUF,
	       "update %s set value = '%s' where obj_id = %d and attr_id = %d",
	       valtab, sval, (int) obj_id, (int) attr_id);
      break;
  }
  r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
  if (!SQLNODATAOK(r)) {
    odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
    rc = GLOBUS_RLS_DBERROR;
    goto error;
  }
  /* At this point, don't check the rowcount, as it may be zero
     if the new value is the same as the current value. */

  return endtran(h, SQL_COMMIT, errmsg, NULL);

 error:
  endtran(h, SQL_ROLLBACK, NULL, NULL);
  return rc;
}

int
db_attr_remove(void *hv, char *key, char *name, globus_rls_obj_type_t objtype,
	       char *errmsg)

{
  db_handle_t			*h = (db_handle_t *) hv;
  char				buf[BIGBUF];
  SQLRETURN			r;
  int				rc;
  SQLINTEGER		attr_id;
  globus_rls_attr_type_t	type;
  SQLINTEGER		obj_id;
  char				*valtab;
  SQLLEN		rows    = 0;

  if (loglevel > 1)
    logit(LOG_DEBUG, "db_attr_remove: %s %s %d", key, name, objtype);

  begintran(h, NULL);

  rc = attrinfo(h, objtype, name, key, &attr_id, &type, &obj_id, &valtab,
		errmsg);
  if (rc != GLOBUS_RLS_SUCCESS)
    goto error;

  snprintf(buf, BIGBUF, "delete from %s where obj_id = %d and attr_id = %d",
	   valtab, (int) obj_id, (int) attr_id);
  r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
  if (!SQLNODATAOK(r)) {
    odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
    rc = GLOBUS_RLS_DBERROR;
    goto error;
  }
  if (SQLOK(r)) {
    SQLRowCount(h->stmt, &rows);
  }
  else {
    rows = 0;
  }
  if (rows != 1) {
    snprintf(errmsg, MAXERRMSG, "%s: %s", key, name);
    rc = GLOBUS_RLS_ATTR_NEXIST;
    goto error;
  }

  return endtran(h, SQL_COMMIT, errmsg, NULL);

 error:
  endtran(h, SQL_ROLLBACK, NULL, NULL);
  return rc;
}

int
db_attr_search(void *hv, char *name, globus_rls_obj_type_t objtype,
	       globus_rls_attr_op_t op,  char *op1, char *op2, int offset,
	       int reslimit, db_str3_callback_t cb, void *a, char *errmsg)

{
  db_handle_t			*h = (db_handle_t *) hv;
  int				rc;
  char				buf[BIGBUF];
  int				nferr;
  SQLINTEGER		attr_id;
  char				*objtab;
  char				*valtab;
  globus_rls_attr_type_t	type;
  char				wherebuf[BIGBUF];
  char				*qc;
  char				rlbuf[100];
  char				op1buf[100];
  char				op2buf[100];

  if (loglevel > 1)
    logit(LOG_DEBUG, "db_attr_search: %s %d %d %s %s %d %d", name, objtype,
	 op, PS(op1), PS(op2), offset, reslimit);

  if ((rc = attrinfo(h, objtype, name, NULL, &attr_id, &type, NULL, &valtab,
		     errmsg)) != GLOBUS_RLS_SUCCESS)
    return rc;
  if ((objtab = objtype2table(objtype, &nferr)) == NULL) {
    sprintf(errmsg, "%d", objtype);
    return GLOBUS_RLS_INV_OBJ_TYPE;
  }
  if (type == globus_rls_attr_type_str || type == globus_rls_attr_type_date)
    qc = "'";
  else
    qc = "";
  if (type == globus_rls_attr_type_date) {
    op1 = todate(op1, op1buf);
    if (op == globus_rls_attr_op_btw)
      op2 = todate(op2, op2buf);
  }
  switch (op) {
    case globus_rls_attr_op_all:
      *wherebuf = '\0';
      break;
    case globus_rls_attr_op_eq:
      snprintf(wherebuf,BIGBUF," and %s.value = %s%s%s", valtab, qc, op1, qc);
      break;
    case globus_rls_attr_op_ne:
      snprintf(wherebuf,BIGBUF," and %s.value != %s%s%s", valtab, qc, op1, qc);
      break;
    case globus_rls_attr_op_gt:
      snprintf(wherebuf,BIGBUF," and %s.value > %s%s%s", valtab, qc, op1, qc);
      break;
    case globus_rls_attr_op_ge:
      snprintf(wherebuf,BIGBUF," and %s.value >= %s%s%s", valtab, qc, op1, qc);
      break;
    case globus_rls_attr_op_lt:
      snprintf(wherebuf,BIGBUF," and %s.value < %s%s%s", valtab, qc, op1, qc);
      break;
    case globus_rls_attr_op_le:
      snprintf(wherebuf,BIGBUF," and %s.value <= %s%s%s", valtab, qc, op1, qc);
      break;
    case globus_rls_attr_op_btw:
      snprintf(wherebuf,BIGBUF," and %s.value between %s%s%s and %s%s%s",
	       valtab, qc, op1, qc, qc, op2, qc);
      break;
    case globus_rls_attr_op_like:
      snprintf(wherebuf,BIGBUF," and %s.value like %s%s%s",
	       valtab, qc, op1, qc);
      break;
    default:
      sprintf(errmsg, "%d", op);
      return GLOBUS_RLS_INV_ATTR_OP;
  }
  snprintf(buf, BIGBUF, "select %s.name, t_attribute.type, %s.value from %s, t_attribute, %s where t_attribute.id = %d and %s.attr_id = %d and %s.obj_id = %s.id %s%s",
	   objtab, valtab, objtab, valtab, (int) attr_id, valtab, (int) attr_id, valtab,
	   objtab, wherebuf, sqllimit(offset, reslimit, rlbuf));
  return query_ret3(h, buf, name, cb, a, errmsg);
}

int
db_attr_value_get(void *hv, char *key, char *name,
		  globus_rls_obj_type_t objtype, db_str3_callback_t cb,
		  void *a, char *errmsg)

{
  db_handle_t			*h = (db_handle_t *) hv;
  int				rc;
  char				buf[BIGBUF];
  int				nferr;
  char				*objtab;
  char				*valtab;
  SQLINTEGER		attr_id;
  SQLINTEGER		obj_id;
  globus_rls_attr_type_t	type;
  char				dbuf[100];

  if (loglevel > 1)
    logit(LOG_DEBUG,"db_attr_value_get: %s %s %d", PS(key), PS(name), objtype);

  if (name) {
    if ((rc = attrinfo(h, objtype, name, key, &attr_id, &type, &obj_id,
		       &valtab, errmsg)) != GLOBUS_RLS_SUCCESS)
      return rc;
    if (type == globus_rls_attr_type_date)
      snprintf(buf, BIGBUF, "select t_attribute.name, t_attribute.type, %s from %s, t_attribute where %s.obj_id = %d and t_attribute.id = %d and %s.attr_id = t_attribute.id and t_attribute.objtype = %d",
	       tochar("t_date_attr.value", dbuf), valtab, valtab, (int) obj_id,
	       (int) attr_id, valtab, objtype);
    else
      snprintf(buf, BIGBUF, "select t_attribute.name, t_attribute.type, %s.value from %s, t_attribute where %s.obj_id = %d and t_attribute.id = %d and %s.attr_id = t_attribute.id and t_attribute.objtype = %d",
	       valtab, valtab, valtab, (int) obj_id, (int) attr_id, valtab, objtype);
    return query_ret3(h, buf, key, cb, a, errmsg);
  }

  /* Specific attribute not specified so return all attributes for key	*/
  if ((objtab = objtype2table(objtype, &nferr)) == NULL) {
    sprintf(errmsg, "%d", objtype);
    return GLOBUS_RLS_INV_OBJ_TYPE;
  }
  for (type = globus_rls_attr_type_date;
       type <= globus_rls_attr_type_str;
       type++) {
    if ((valtab = valuetype2table(type)) == NULL)
      continue;
    if (type == globus_rls_attr_type_date)
      snprintf(buf, BIGBUF, "select t_attribute.name, t_attribute.type, %s from %s, %s, t_attribute where %s.name = '%s' and %s.id = %s.obj_id and %s.attr_id = t_attribute.id and t_attribute.objtype = %d",
	       tochar("t_date_attr.value", dbuf), valtab, objtab,
	       objtab, key, objtab, valtab, valtab, objtype);
    else
      snprintf(buf, BIGBUF, "select t_attribute.name, t_attribute.type, %s.value from %s, %s, t_attribute where %s.name = '%s' and %s.id = %s.obj_id and %s.attr_id = t_attribute.id and t_attribute.objtype = %d",
	       valtab, objtab, valtab, objtab, key, objtab, valtab, valtab,
	       objtype);
    if ((rc = query_ret3(h, buf, key, cb, a, errmsg)) != GLOBUS_RLS_SUCCESS)
      return rc;
  }
  return GLOBUS_RLS_SUCCESS;
}

int
db_attr_value_get_bulk(void *hv, char *key, char *name,
		       globus_rls_obj_type_t objtype, db_str4_callback_t cb,
		       void *a, char *errmsg)

{
  db_handle_t			*h = (db_handle_t *) hv;
  int				rc;
  char				buf[BIGBUF];
  int				nferr;
  char				*objtab;
  char				*valtab;
  SQLINTEGER		attr_id;
  SQLINTEGER		obj_id;
  globus_rls_attr_type_t	type;
  char				dbuf[100];

  if (loglevel > 1)
    logit(LOG_DEBUG,"db_attr_value_get_bulk: %s %s %d", PS(key), PS(name), objtype);

  if (name) {
    if ((rc = attrinfo(h, objtype, name, key, &attr_id, &type, &obj_id,
		       &valtab, errmsg)) != GLOBUS_RLS_SUCCESS)
      return rc;
    if (type == globus_rls_attr_type_date)
      snprintf(buf, BIGBUF, "select '%s', t_attribute.name, t_attribute.type, %s from %s, t_attribute where %s.obj_id = %d and t_attribute.id = %d and %s.attr_id = t_attribute.id and t_attribute.objtype = %d",
	       key, tochar("t_date_attr.value", dbuf), valtab, valtab,
	       (int) obj_id, (int) attr_id, valtab, objtype);
    else
      snprintf(buf, BIGBUF, "select '%s',t_attribute.name, t_attribute.type, %s.value from %s, t_attribute where %s.obj_id = %d and t_attribute.id = %d and %s.attr_id = t_attribute.id and t_attribute.objtype = %d",
	       key, valtab, valtab, valtab, (int) obj_id, (int) attr_id, valtab, objtype);
    return query_ret4(h, buf, key, cb, a, errmsg);
  }

  /* Specific attribute not specified so return all attributes for key	*/
  if ((objtab = objtype2table(objtype, &nferr)) == NULL) {
    sprintf(errmsg, "%d", objtype);
    return GLOBUS_RLS_INV_OBJ_TYPE;
  }
  for (type = globus_rls_attr_type_date;
       type <= globus_rls_attr_type_str;
       type++) {
    if ((valtab = valuetype2table(type)) == NULL)
      continue;
    if (type == globus_rls_attr_type_date)
      snprintf(buf, BIGBUF, "select '%s', t_attribute.name, t_attribute.type, %s from %s, %s, t_attribute where %s.name = '%s' and %s.id = %s.obj_id and %s.attr_id = t_attribute.id and t_attribute.objtype = %d",
	       key, tochar("t_date_attr.value", dbuf), objtab, valtab, objtab,
	       key, objtab, valtab, valtab, objtype);
    else
      snprintf(buf, BIGBUF, "select '%s', t_attribute.name, t_attribute.type, %s.value from %s, %s, t_attribute where %s.name = '%s' and %s.id = %s.obj_id and %s.attr_id = t_attribute.id and t_attribute.objtype = %d",
	       key, valtab, objtab, valtab, objtab, key, objtab, valtab, valtab,
	       objtype);
    if ((rc = query_ret4(h, buf, key, cb, a, errmsg)) != GLOBUS_RLS_SUCCESS)
      return rc;
  }
  return GLOBUS_RLS_SUCCESS;
}

int
db_exists(void *hv, char *key, globus_rls_obj_type_t objtype, char *errmsg)

{
  db_handle_t	*h = (db_handle_t *) hv;
  char		buf[BIGBUF];
  char		*objtab;
  int		nferr;
  long       	rows    = 0;
  SQLRETURN	r;

  if (loglevel > 1)
    logit(LOG_DEBUG, "db_exists: %s %d", key, objtype);
  if ((objtab = objtype2table(objtype, &nferr)) == NULL) {
    sprintf(errmsg, "%d", objtype);
    return GLOBUS_RLS_INV_OBJ_TYPE;
  }
  snprintf(buf, BIGBUF, "select count(*) from %s where name = '%s'",
	   objtab, key);
  r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
  if (!SQLOK(r)) {
    odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
    return GLOBUS_RLS_DBERROR;
  }
  /*
   * Use to use SQLRowCount() to see if we got a result, but some ODBC
   * drivers (including Easysoft's Oracle driver) don't support this on
   * after a select stmt.
   */
  r = SQLBindCol(h->stmt, 1, SQL_C_LONG, &rows, 0, NULL);
  if (SQLOK(r))
    r = SQLFetch(h->stmt);
  if (!SQLOK(r))
    goto error;
  SQLFreeStmt(h->stmt, SQL_CLOSE);
  SQLFreeStmt(h->stmt, SQL_UNBIND);
  if (rows != 1) {
    snprintf(errmsg, MAXERRMSG, "%s,%d", key, objtype);
    return nferr;
  }
  return GLOBUS_RLS_SUCCESS;

 error:
  SQLFreeStmt(h->stmt, SQL_CLOSE);
  SQLFreeStmt(h->stmt, SQL_UNBIND);
  odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
  return GLOBUS_RLS_DBERROR;
}

int
db_lrc_add(void *hv, char *lfn, char *pfn, char *errmsg)

{
  db_handle_t	*h = (db_handle_t *) hv;
  char		buf[BIGBUF];
  SQLINTEGER	lfn_id;
  SQLINTEGER	pfn_id;
  int		rc;
  SQLRETURN	r;
  int		adjcount[T_NUM];

  if (loglevel > 1)
    logit(LOG_DEBUG, "db_lrc_add: %s %s", lfn, pfn);

  if (strlen(pfn) + 1 > MAXDBSTR) {
    strncpy(errmsg, pfn, MAXERRMSG);
    return GLOBUS_RLS_BADARG;
  }

  begintran(h, adjcount);

  /* Update ref counts for lfn, pfn, if pfn doesn't exist it's inserted	*/
  if ((rc = updateref(h, T_LRCLFN, lfn, 1, 0, &lfn_id, NULL,
		      errmsg, adjcount)) != GLOBUS_RLS_SUCCESS)
    goto error;
  if ((rc = updateref(h, T_LRCPFN, pfn, 1, 1, &pfn_id, NULL,
		      errmsg, adjcount)) != GLOBUS_RLS_SUCCESS)
    goto error;

  /* Add mapping for lfn_id, pfn_id	*/
  snprintf(buf, BIGBUF, "insert into t_map (lfn_id,pfn_id) values (%d,%d)",
	   (int) lfn_id, (int) pfn_id);
  r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
  if (!SQLOK(r)) {
    rc = seterr_insertmap(h, r, "lfn_id", (int) lfn_id, "pfn_id", (int) pfn_id,
			  NULL, 0, errmsg);
    goto error;
  }
  adjcount[T_LRCMAP]++;

  return endtran(h, SQL_COMMIT, errmsg, adjcount);

error:
  endtran(h, SQL_ROLLBACK, NULL, NULL);
  return rc;
}

int
db_lrc_renamelfn(void *hv, char *oldname, char *newname, char *errmsg)

{
  db_handle_t	*h = (db_handle_t *) hv;
  char		buf[BIGBUF];
  SQLINTEGER lfn_id;
  int		rc;
  SQLRETURN	r;
  int		adjcount[T_NUM];

  if (loglevel > 1)
    logit(LOG_DEBUG, "db_lrc_renamelfn: %s %s", oldname, newname);

  if (strlen(newname) + 1 > MAXDBSTR) {
    strncpy(errmsg, newname, MAXERRMSG);
    return GLOBUS_RLS_BADARG;
  }

  begintran(h, adjcount);

  /* Ensure that LFN DOES exist for oldname */
  if ((rc = getid(h, oldname, "t_lfn", GLOBUS_RLS_LFN_NEXIST, &lfn_id,
		  errmsg)) != GLOBUS_RLS_SUCCESS) {
    rc = GLOBUS_RLS_LFN_NEXIST;
    strncpy(errmsg, oldname, MAXERRMSG);
    goto error;
  }

  /* Ensure that LFN does NOT exist for newname */
  if ((rc = getid(h, newname, "t_lfn", GLOBUS_RLS_LFN_NEXIST, &lfn_id,
		  errmsg)) == GLOBUS_RLS_SUCCESS) {
    rc = GLOBUS_RLS_LFN_EXIST;
    strncpy(errmsg, newname, MAXERRMSG);
    goto error;
  }

  /* Rename lfn	*/
  snprintf(buf, BIGBUF, "update t_lfn set name = '%s' where name = '%s'",
	   newname, oldname);
  r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
  if (!SQLOK(r)) {
    odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
    rc = GLOBUS_RLS_DBERROR;
    goto error;
  }

  /* Commit changes */
  rc = endtran(h, SQL_COMMIT, errmsg, adjcount);
  return rc;

error:
  endtran(h, SQL_ROLLBACK, NULL, NULL);
  return rc;
}

int
db_lrc_renamepfn(void *hv, char *oldname, char *newname, char *errmsg)

{
  db_handle_t	*h = (db_handle_t *) hv;
  char		buf[BIGBUF];
  SQLINTEGER pfn_id;
  int		rc;
  SQLRETURN	r;
  int		adjcount[T_NUM];

  if (loglevel > 1)
    logit(LOG_DEBUG, "db_lrc_renamepfn: %s %s", oldname, newname);

  if (strlen(newname) + 1 > MAXDBSTR) {
    strncpy(errmsg, newname, MAXERRMSG);
    return GLOBUS_RLS_BADARG;
  }

  begintran(h, adjcount);

  /* Ensure that PFN DOES exist for oldname */
  if ((rc = getid(h, oldname, "t_pfn", GLOBUS_RLS_PFN_NEXIST, &pfn_id,
		  errmsg)) != GLOBUS_RLS_SUCCESS) {
    rc = GLOBUS_RLS_PFN_NEXIST;
    strncpy(errmsg, oldname, MAXERRMSG);
    goto error;
  }

  /* Ensure that PFN does NOT exist for newname */
  if ((rc = getid(h, newname, "t_pfn", GLOBUS_RLS_PFN_NEXIST, &pfn_id,
		  errmsg)) == GLOBUS_RLS_SUCCESS) {
    rc = GLOBUS_RLS_PFN_EXIST;
    strncpy(errmsg, newname, MAXERRMSG);
    goto error;
  }

  /* Rename pfn	*/
  snprintf(buf, BIGBUF, "update t_pfn set name = '%s' where name = '%s'",
	   newname, oldname);
  r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
  if (!SQLOK(r)) {
    odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
    rc = GLOBUS_RLS_DBERROR;
    goto error;
  }

  return endtran(h, SQL_COMMIT, errmsg, adjcount);

error:
  endtran(h, SQL_ROLLBACK, NULL, NULL);
  return rc;
}

int
db_lrc_alllfn(void *hv, db_str1_callback_t cb, void *a, char *errmsg)

{
  db_handle_t	*h = (db_handle_t *) hv;

  if (loglevel > 1)
    logit(LOG_DEBUG, "db_lrc_alllfn:");
  return query_ret1(h, "select t_lfn.name from t_lfn", "all", cb, a, errmsg);
}

int
db_lrc_clear(void *hv, char *errmsg)

{
  db_handle_t	*h = (db_handle_t *) hv;
  SQLRETURN	r;
  int		adjcount[T_NUM];

  if (loglevel > 1)
    logit(LOG_DEBUG, "db_lrc_clear:");

  begintran(h, adjcount);

  r = SQLExecDirect(h->stmt, (SQLCHAR *) "delete from t_date_attr", SQL_NTS);
  if (SQLNODATAOK(r))
    r = SQLExecDirect(h->stmt, (SQLCHAR *) "delete from t_flt_attr", SQL_NTS);
  if (SQLNODATAOK(r))
    r = SQLExecDirect(h->stmt, (SQLCHAR *) "delete from t_int_attr", SQL_NTS);
  if (SQLNODATAOK(r))
    r = SQLExecDirect(h->stmt, (SQLCHAR *) "delete from t_str_attr", SQL_NTS);
  if (SQLNODATAOK(r))
    r = SQLExecDirect(h->stmt, (SQLCHAR *) "delete from t_map", SQL_NTS);
  if (SQLNODATAOK(r))
    r = SQLExecDirect(h->stmt, (SQLCHAR *) "delete from t_pfn", SQL_NTS);
  if (SQLNODATAOK(r))
    r = SQLExecDirect(h->stmt, (SQLCHAR *) "delete from t_lfn", SQL_NTS);

  if (!SQLNODATAOK(r)) {
    endtran(h, SQL_ROLLBACK, NULL, NULL);
    return GLOBUS_RLS_DBERROR;
  }

  adjcount[T_LRCLFN] = -table[T_LRCLFN].count;
  adjcount[T_LRCPFN] = -table[T_LRCPFN].count;
  adjcount[T_LRCMAP] = -table[T_LRCMAP].count;
  return endtran(h, SQL_COMMIT, errmsg, adjcount);
}

int
db_lrc_create(void *hv, char *lfn, char *pfn, char *errmsg)

{
  db_handle_t	*h = (db_handle_t *) hv;
  char		buf[BIGBUF];
  SQLINTEGER	lfn_id;
  SQLINTEGER	pfn_id;
  int		rc;
  SQLRETURN	r;
  int		adjcount[T_NUM];

  if (loglevel > 1)
    logit(LOG_DEBUG, "db_lrc_create: %s %s", lfn, pfn);

  if (strlen(lfn) + 1 > MAXDBSTR) {
    strncpy(errmsg, lfn, MAXERRMSG);
    return GLOBUS_RLS_BADARG;
  }
  if (strlen(pfn) + 1 > MAXDBSTR) {
    strncpy(errmsg, pfn, MAXERRMSG);
    return GLOBUS_RLS_BADARG;
  }

  begintran(h, adjcount);

  /* Create LFN, error if already exists	*/
  snprintf(buf, BIGBUF, "insert into t_lfn (name, ref) values ('%s', 1)", lfn);
  r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
  if (!SQLOK(r)) {
    rc = seterr_insert(h, r, "t_lfn", lfn, GLOBUS_RLS_LFN_EXIST, errmsg);
    goto error;
  }
  adjcount[T_LRCLFN]++;
  if ((rc = getid(h, lfn, "t_lfn", GLOBUS_RLS_LFN_NEXIST, &lfn_id,
		  errmsg)) != GLOBUS_RLS_SUCCESS)
    goto error;
  /* Update ref count for pfn or create if it doesn't exist	*/
  if ((rc = updateref(h, T_LRCPFN, pfn, 1, 1, &pfn_id, NULL,
		      errmsg, adjcount)) != GLOBUS_RLS_SUCCESS)
    goto error;

  /* Add mapping for lfn_id, pfn_id	*/
  snprintf(buf, BIGBUF, "insert into t_map (lfn_id,pfn_id) values (%d,%d)",
	   (int) lfn_id, (int) pfn_id);
  r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
  if (!SQLOK(r)) {
    rc = seterr_insertmap(h, r, "lfn_id", (int) lfn_id, "pfn_id", (int) pfn_id, 
            NULL, 0, errmsg);
    goto error;
  }
  adjcount[T_LRCMAP]++;

  rc = endtran(h, SQL_COMMIT, errmsg, adjcount);
  return rc;

 error:
  endtran(h, SQL_ROLLBACK, NULL, NULL);
  return rc;
}

int
db_lrc_delete(void *hv, char *lfn, char *pfn, int *lfndeleted, char *errmsg)

{
  db_handle_t	*h = (db_handle_t *) hv;
  char		buf[BIGBUF];
  SQLINTEGER	lfn_id;
  SQLINTEGER	lfn_ref;
  SQLINTEGER	pfn_id;
  SQLINTEGER	pfn_ref;
  int		rc;
  SQLRETURN	r;
  SQLLEN	rows    = 0;
  int		adjcount[T_NUM];

  if (loglevel > 1)
    logit(LOG_DEBUG, "db_lrc_delete: %s %s", lfn, pfn);

  begintran(h, adjcount);

  /* Decrement ref counts on lfn, pfn	*/
  if ((rc = updateref(h, T_LRCLFN, lfn, -1, 0, &lfn_id, &lfn_ref,
		      errmsg, adjcount)) != GLOBUS_RLS_SUCCESS)
    goto error;
  if ((rc = updateref(h, T_LRCPFN, pfn, -1, 0, &pfn_id, &pfn_ref,
		      errmsg, adjcount)) != GLOBUS_RLS_SUCCESS)
    goto error;

  /* Delete t_map map record	*/
  snprintf(buf, BIGBUF, "delete from t_map where lfn_id = %d and pfn_id = %d",
	   (int) lfn_id, (int) pfn_id);
  r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
  if (!SQLNODATAOK(r)) {
    odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
    rc = GLOBUS_RLS_DBERROR;
    goto error;
  }
  if (SQLOK(r)) {
    SQLRowCount(h->stmt, &rows);
  }
  else {
    rows = 0;
  }
  if (rows != 1) {
    snprintf(errmsg, MAXERRMSG, "%s,%s", lfn, pfn);
    rc = GLOBUS_RLS_MAPPING_NEXIST;
    goto error;
  }
  adjcount[T_LRCMAP]--;

  *lfndeleted = 0;
  if (lfn_ref == 0) {	/* Delete lfn (and attrs) if ref count is 0	*/
    r = deleteattrvals(h, globus_rls_obj_lrc_lfn, lfn_id);
    if (SQLNODATAOK(r)) {
      snprintf(buf, BIGBUF, "delete from t_lfn where id=%d and ref=0", (int) lfn_id);
      r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
    }
    if (!SQLNODATAOK(r)) {
      odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
      rc = GLOBUS_RLS_DBERROR;
      goto error;
    }
    if (SQLOK(r)) {
      SQLRowCount(h->stmt, &rows);
    }
    else {
      rows = 0;
    }
    if (rows) {
      adjcount[T_LRCLFN]--;
      *lfndeleted = 1;
    }
  }

  if (pfn_ref == 0) {	/* Delete pfn (and attrs) if ref count is 0	*/
    r = deleteattrvals(h, globus_rls_obj_lrc_pfn, pfn_id);
    if (SQLNODATAOK(r)) {
      snprintf(buf, BIGBUF, "delete from t_pfn where id=%d and ref=0", (int) pfn_id);
      r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
    }
    if (!SQLNODATAOK(r)) {
      odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
      rc = GLOBUS_RLS_DBERROR;
      goto error;
    }
    if (SQLOK(r)) {
      SQLRowCount(h->stmt, &rows);
      if (rows)
        adjcount[T_LRCPFN]--;
    }
  }

  rc = endtran(h, SQL_COMMIT, errmsg, adjcount);
  return rc;

 error:
  endtran(h, SQL_ROLLBACK, NULL, NULL);
  return rc;
}

int
db_lrc_getlfn(void *hv, char *pfn, globus_bool_t wc, int offset, int reslimit,
	      db_str2_callback_t cb, void *a,char *errmsg)

{
  db_handle_t	*h = (db_handle_t *) hv;
  char		buf[BIGBUF];
  char		rlbuf[100];

  if (loglevel > 1)
    logit(LOG_DEBUG, "db_lrc_getlfn: %s%s %d %d", pfn, wc ? " wildcard" : "",
	  offset, reslimit);
  snprintf(buf, BIGBUF, "select t_lfn.name, t_pfn.name from t_lfn, t_pfn, t_map where t_pfn.name %s '%s' and t_lfn.id = t_map.lfn_id and t_pfn.id = t_map.pfn_id order by t_lfn.name,t_pfn.name%s", wc ? "like" : "=", pfn, sqllimit(offset, reslimit, rlbuf));

  return query_ret2(h, buf, pfn, cb, a, errmsg);
}

int
db_lrc_getpfn(void *hv, char *lfn, globus_bool_t wc, int offset, int reslimit,
	      db_str2_callback_t cb, void *a,char *errmsg)

{
  db_handle_t	*h = (db_handle_t *) hv;
  char		buf[BIGBUF];
  char		rlbuf[100];

  if (loglevel > 1)
    logit(LOG_DEBUG, "db_lrc_getpfn: %s%s %d %d", lfn, wc ? " wildcard" : "",
	  offset, reslimit);
  snprintf(buf, BIGBUF, "select t_lfn.name, t_pfn.name from t_lfn, t_pfn, t_map where t_lfn.name %s '%s' and t_lfn.id = t_map.lfn_id and t_pfn.id = t_map.pfn_id order by t_lfn.name,t_pfn.name%s",
	   wc ? "like" : "=", lfn, sqllimit(offset, reslimit, rlbuf));

  return query_ret2(h, buf, lfn, cb, a, errmsg);
}

int
db_lrc_mapping_exists(void *hv, char *lfn, char *pfn, char *errmsg)

{
  db_handle_t	*h = (db_handle_t *) hv;
  char		buf[BIGBUF];
  long		rows    = 0;
  SQLLEN	len;
  SQLRETURN	r;

  if (loglevel > 1)
    logit(LOG_DEBUG, "db_lrc_mapping_exists: %s %s", lfn, pfn);
  snprintf(buf, BIGBUF, "select count(*) from t_map,t_lfn,t_pfn where t_lfn.name = '%s' and t_pfn.name = '%s' and t_map.lfn_id = t_lfn.id and t_map.pfn_id = t_pfn.id" , lfn, pfn);
  r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
  if (!SQLOK(r)) {
    odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
    return GLOBUS_RLS_DBERROR;
  }
  /*
   * Use to use SQLRowCount() to see if we got a result, but some ODBC
   * drivers (including Easysoft's Oracle driver) don't support this on
   * after a select stmt.
   */
  r = SQLBindCol(h->stmt, 1, SQL_C_LONG, &rows, sizeof(rows), &len);
  if (SQLOK(r))
    r = SQLFetch(h->stmt);
  if (!SQLOK(r))
    goto error;
  SQLFreeStmt(h->stmt, SQL_CLOSE);
  SQLFreeStmt(h->stmt, SQL_UNBIND);
  if (rows != 1) {
    snprintf(errmsg, MAXERRMSG, "%s,%s", lfn, pfn);
    return GLOBUS_RLS_MAPPING_NEXIST;
  }
  return GLOBUS_RLS_SUCCESS;

 error:
  SQLFreeStmt(h->stmt, SQL_CLOSE);
  SQLFreeStmt(h->stmt, SQL_UNBIND);
  odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
  return GLOBUS_RLS_DBERROR;
}

int
db_rli_alllfn(void *hv, db_str2_callback_t cb, void *a, char *errmsg)

{
  db_handle_t	*h = (db_handle_t *) hv;

  if (loglevel > 1)
    logit(LOG_DEBUG, "db_rli_alllfn:");
  return query_ret2(h, "select t_lfn.name, t_lrc.name from t_lfn, t_lrc, t_map where t_lfn.id = t_map.lfn_id and t_map.lrc_id = t_lrc.id order by t_lrc.name", "all", cb, a, errmsg);
}

int
db_rli_delete(void *hv, char *lfn, char *lrc, char *sender, int *lfndeleted,
	      char *errmsg)

{
  db_handle_t	*h = (db_handle_t *) hv;
  char		buf[BIGBUF];
  SQLINTEGER	lfn_id;
  SQLINTEGER	lrc_id;
  SQLINTEGER	sender_id;
  int		rc;
  SQLRETURN	r;
  SQLLEN	rows    = 0;
  int		adjcount[T_NUM];

  if (loglevel > 1)
    logit(LOG_DEBUG, "db_rli_delete: %s %s %s", lfn, lrc, sender);

  globus_mutex_lock(&rlidbmtx);
  begintran(h, adjcount);

  /* Decrement ref counts on lfn, lrc, sender	*/
  if ((rc = updateref(h, T_RLILFN, lfn, -1, 0, &lfn_id, NULL,
		      errmsg, adjcount)) != GLOBUS_RLS_SUCCESS)
    goto error;
  if ((rc = updateref(h, T_RLILRC, lrc, -1, 0, &lrc_id, NULL,
		      errmsg, adjcount)) != GLOBUS_RLS_SUCCESS)
    goto error;
  if ((rc = updateref(h, T_RLISENDER, sender, -1, 0, &sender_id, NULL,
		      errmsg, adjcount)) != GLOBUS_RLS_SUCCESS)
    goto error;

  /* Delete t_map map record	*/
  snprintf(buf, BIGBUF, "delete from t_map where lfn_id = %d and lrc_id = %d and sender_id = %d",
	   (int) lfn_id, (int) lrc_id, (int) sender_id);
  r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
  if (!SQLNODATAOK(r)) {
    odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
    rc = GLOBUS_RLS_DBERROR;
    goto error;
  }
  if (SQLOK(r)) {
    SQLRowCount(h->stmt, &rows);
  }
  else {
    rows = 0;
  }
  if (rows != 1) {
    snprintf(errmsg, MAXERRMSG, "%s,%s,%s", lfn, lrc, sender);
    rc = GLOBUS_RLS_MAPPING_NEXIST;
    goto error;
  }
  adjcount[T_RLIMAP]--;

  /* Delete lfn, lrc and sender if ref count is 0	*/
  snprintf(buf, BIGBUF, "delete from t_lfn where id = %d and ref = 0", (int) lfn_id);
  r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
  if (!SQLNODATAOK(r)) {
    odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
    rc = GLOBUS_RLS_DBERROR;
    goto error;
  }
  if (SQLOK(r)) {
    SQLRowCount(h->stmt, &rows);
  }
  else {
    rows = 0;
  }
  if (rows) {
    *lfndeleted = 1;
    adjcount[T_RLILFN]--;
  } else
    *lfndeleted = 0;

  snprintf(buf, BIGBUF, "delete from t_lrc where id = %d and ref = 0", (int) lrc_id);
  r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
  if (!SQLNODATAOK(r)) {
    odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
    rc = GLOBUS_RLS_DBERROR;
    goto error;
  }
  if (SQLOK(r)) {
    SQLRowCount(h->stmt, &rows);
  }
  else {
    rows = 0;
  }
  if (rows)
    adjcount[T_RLILRC]--;

  snprintf(buf, BIGBUF, "delete from t_sender where id = %d and ref = 0", (int) sender_id);
  r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
  if (!SQLNODATAOK(r)) {
    odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
    rc = GLOBUS_RLS_DBERROR;
    goto error;
  }
  if (SQLOK(r)) {
    SQLRowCount(h->stmt, &rows);
  }
  else {
    rows = 0;
  }
  if (rows)
    adjcount[T_RLISENDER]--;

  globus_mutex_unlock(&rlidbmtx);
  return endtran(h, SQL_COMMIT, errmsg, adjcount);

 error:
  globus_mutex_unlock(&rlidbmtx);
  endtran(h, SQL_ROLLBACK, NULL, NULL);
  return rc;
}

int
db_rli_expire(void *hv, char *sender, time_t rlistaleint, char *errmsg)

{
  db_handle_t	*h = (db_handle_t *) hv;
  char		buf[BIGBUF];
  char		tbuf[100];
  long  	lfn_id  = 0;
  long  	lrc_id  = 0;
  SQLINTEGER	sender_id;
  SQLLEN	len;
  IDLIST	*lfnids = NULL;
  IDLIST	*lrcids = NULL;
  IDLIST	*p;
  int		i;
  SQLRETURN	r;
  int		rc;
  int		mapdelcnt;
  SQLLEN	rows    = 0;
  int		adjcount[T_NUM];
  char		dbuf[100];

  if (loglevel)
    logit(LOG_DEBUG, "db_rli_expire: %s %d", sender, rlistaleint);

  if ((rc = getid(h, sender, "t_sender", GLOBUS_RLS_LRC_NEXIST, &sender_id,
		  errmsg)) != GLOBUS_RLS_SUCCESS)
    return rc;

  globus_mutex_lock(&rlidbmtx);
  begintran(h, adjcount);

  /* Get lfn and lrc ids we're going to expire		*/
  mycftime(tbuf, 100, "%Y-%m-%d %H:%M:%S", time(0) - rlistaleint);
  snprintf(buf, BIGBUF, "select lfn_id, lrc_id from t_map where updatetime < %s and sender_id = %d", todate(tbuf, dbuf), (int) sender_id);
  r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
  if (!SQLOK(r))
    goto error;

  r = SQLBindCol(h->stmt, 1, SQL_C_LONG, &lfn_id, sizeof(lfn_id), &len);
  if (SQLOK(r))
    r = SQLBindCol(h->stmt, 2, SQL_C_LONG, &lrc_id, sizeof(lrc_id), &len);
  if (!SQLOK(r)) {
    SQLFreeStmt(h->stmt, SQL_CLOSE);
    SQLFreeStmt(h->stmt, SQL_UNBIND);
    goto error;
  }

  /* Save ids in lfnids list	*/
  while (1) {
    if ((r = SQLFetch(h->stmt)) == SQL_NO_DATA)
      break;
    if (!SQLOK(r)) {
      SQLFreeStmt(h->stmt, SQL_CLOSE);
      SQLFreeStmt(h->stmt, SQL_UNBIND);
      goto error;
    }
    addid(&lfnids, (SQLINTEGER) lfn_id, 0);
    addid(&lrcids, (SQLINTEGER) lrc_id, 0);
  }
  SQLFreeStmt(h->stmt, SQL_CLOSE);
  SQLFreeStmt(h->stmt, SQL_UNBIND);

  /* Remove stale map entries		*/
  snprintf(buf, BIGBUF, "delete from t_map where updatetime < %s and sender_id = %d", todate(tbuf, dbuf), (int) sender_id);
  r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
  if (!SQLNODATAOK(r))
    goto error;
  if (SQLOK(r)) {
    SQLRowCount(h->stmt, &rows);
  }
  else {
    rows = 0;
  }
  mapdelcnt = rows;
  adjcount[T_RLIMAP] -= mapdelcnt;

  /* Update ref counts on lfns referenced by deleted mappings	*/
  for (p = lfnids; p; p = p->nxt) {
    for (i = 0; i < p->n; i++) {
      snprintf(buf, BIGBUF, "update t_lfn set ref = ref - 1 where id = %d",
	       (int) p->ids[i]);
      r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
      if (!SQLNODATAOK(r))
	goto error;
    }
  }
  snprintf(buf, BIGBUF,"delete from t_lfn where ref = 0");
  r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
  if (!SQLNODATAOK(r))
    goto error;
  if (SQLOK(r)) {
    SQLRowCount(h->stmt, &rows);
    if (rows)
      adjcount[T_RLILFN] -= rows;
  }

  /* Update ref counts on lrcs referenced by deleted mappings	*/
  for (p = lrcids; p; p = p->nxt) {
    for (i = 0; i < p->n; i++) {
      snprintf(buf, BIGBUF, "update t_lrc set ref = ref - 1 where id = %d",
	       (int) p->ids[i]);
      r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
      if (!SQLNODATAOK(r))
	goto error;
    }
  }
  snprintf(buf, BIGBUF,"delete from t_lrc where ref = 0");
  r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
  if (!SQLNODATAOK(r))
    goto error;
  if (SQLOK(r)) {
    SQLRowCount(h->stmt, &rows);
    if (rows)
      adjcount[T_RLILRC] -= rows;
  }

  /* Update ref count on sender table		*/
  if (mapdelcnt) {
    /* Decrement ref count on sender record	*/
    snprintf(buf, BIGBUF, "update t_sender set ref = ref - %d where id = %d",
	     mapdelcnt, (int) sender_id);
    r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
    if (!SQLNODATAOK(r))
      goto error;
    snprintf(buf, BIGBUF,"delete from t_sender where id = %d and ref = 0", (int) sender_id);
    r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
    if (!SQLNODATAOK(r))
      goto error;
    if (SQLOK(r)) {
      SQLRowCount(h->stmt, &rows);
      if (rows)
        adjcount[T_RLISENDER]--;
    }
  }

  globus_mutex_unlock(&rlidbmtx);
  freeidlist(lfnids);
  freeidlist(lrcids);
  return endtran(h, SQL_COMMIT, errmsg, adjcount);

 error:
  endtran(h, SQL_ROLLBACK, NULL, NULL);
  globus_mutex_unlock(&rlidbmtx);
  odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
  freeidlist(lfnids);
  freeidlist(lrcids);
  return GLOBUS_RLS_DBERROR;
}

int
db_rli_getlrc(void *hv, char *lfn, globus_bool_t wc, int offset, int reslimit,
	      db_str2_callback_t cb, void *a, char *errmsg)

{
  db_handle_t	*h = (db_handle_t *) hv;
  char		buf[BIGBUF];
  char		rlbuf[100];

  if (loglevel > 1)
    logit(LOG_DEBUG, "db_rli_getlrc: %s%s %d %d", lfn, wc ? " wildcard":"",
	  offset, reslimit);
  snprintf(buf, BIGBUF, "select distinct t_lfn.name, t_lrc.name from t_lfn, t_lrc, t_map where t_lfn.name %s '%s' and t_lfn.id = t_map.lfn_id and t_lrc.id = t_map.lrc_id order by t_lfn.name%s",
	   wc ? "like" : "=", lfn, sqllimit(offset, reslimit, rlbuf));

  return query_ret2(h, buf, lfn, cb, a, errmsg);
}

int
db_rli_sender_list(void *hv, db_str2_callback_t cb, void *a, char *errmsg)

{
  db_handle_t	*h = (db_handle_t *) hv;
  char		buf[BIGBUF];
  char		dbuf[100];

  if (loglevel > 1)
    logit(LOG_DEBUG, "db_rli_sender_list:");
  snprintf(buf, BIGBUF, "select t_sender.name, %s from t_sender, t_map where t_sender.id = t_map.sender_id group by name",
	   tochar("max(t_map.updatetime)", dbuf));
  return query_ret2(h, buf, "all", cb, a, errmsg);
}

int
db_rli_mapping_exists(void *hv, char *lfn, char *lrc, char *errmsg)

{
  db_handle_t	*h = (db_handle_t *) hv;
  char		buf[BIGBUF];
  long		rows    = 0;
  SQLLEN	len;
  SQLRETURN	r;

  if (loglevel > 1)
    logit(LOG_DEBUG, "db_lrc_mapping_exists: %s %s", lfn, lrc);
  snprintf(buf, BIGBUF, "select count(*) from t_map,t_lfn,t_lrc where t_lfn.name = '%s' and t_lrc.name = '%s' and t_map.lfn_id = t_lfn.id and t_map.lrc_id = t_lrc.id" , lfn, lrc);
  r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
  if (!SQLOK(r)) {
    odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
    return GLOBUS_RLS_DBERROR;
  }
  /*
   * Use to use SQLRowCount() to see if we got a result, but some ODBC
   * drivers (including Easysoft's Oracle driver) don't support this on
   * after a select stmt.
   */
  r = SQLBindCol(h->stmt, 1, SQL_C_LONG, &rows, sizeof(rows), &len);
  if (SQLOK(r))
    r = SQLFetch(h->stmt);
  if (!SQLOK(r))
    goto error;
  SQLFreeStmt(h->stmt, SQL_CLOSE);
  SQLFreeStmt(h->stmt, SQL_UNBIND);
  if (rows != 1) {
    snprintf(errmsg, MAXERRMSG, "%s,%s", lfn, lrc);
    return GLOBUS_RLS_MAPPING_NEXIST;
  }
  return GLOBUS_RLS_SUCCESS;

 error:
  SQLFreeStmt(h->stmt, SQL_CLOSE);
  SQLFreeStmt(h->stmt, SQL_UNBIND);
  odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
  return GLOBUS_RLS_DBERROR;
}

int
db_rli_update(void *hv, char *lfn, char *lrc, char *sender, char *ts,
	      int *lfnadded, char *errmsg)

{
  db_handle_t	*h = (db_handle_t *) hv;
  int		rc;
  char		buf[BIGBUF];
  long  	lfn_id      = 0;
  long  	lrc_id      = 0;
  long  	sender_id   = 0;
  SQLRETURN	r;
  SQLLEN	len;
  int		adjcount[T_NUM];
  char		dbuf[100];

  if (loglevel > 1)
    logit(LOG_DEBUG, "db_rli_update: %s %s %s", lfn, lrc, sender);

  globus_mutex_lock(&rlidbmtx);
  begintran(h, adjcount);

  /*
   * See if mapping exists, if so just update timestamp.  Note we could do this
   * in a single update if mysql supported nested subqueries.
   * *** should lock lfn, lrc, sender records ***
   */
  snprintf(buf, BIGBUF, "select t_lfn.id, t_lrc.id, t_sender.id from t_lfn, t_lrc, t_sender, t_map where t_lfn.name = '%s' and t_lrc.name = '%s' and t_sender.name = '%s' and t_lfn.id = t_map.lfn_id and t_lrc.id = t_map.lrc_id and t_sender.id = t_map.sender_id", lfn, lrc, sender);
  r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
  if (!SQLOK(r)) {
    rc = GLOBUS_RLS_DBERROR;
    goto error;
  }

  r = SQLBindCol(h->stmt, 1, SQL_C_LONG, &lfn_id, sizeof(lfn_id), &len);
  if (SQLOK(r))
    r = SQLBindCol(h->stmt, 2, SQL_C_LONG, &lrc_id, sizeof(lrc_id), &len);
  if (SQLOK(r))
    r = SQLBindCol(h->stmt, 3, SQL_C_LONG, &sender_id, sizeof(sender_id), &len);
  if (!SQLOK(r)) {
    SQLFreeStmt(h->stmt, SQL_CLOSE);
    SQLFreeStmt(h->stmt, SQL_UNBIND);
    rc = GLOBUS_RLS_DBERROR;
    goto error;
  }

  r = SQLFetch(h->stmt);
  SQLFreeStmt(h->stmt, SQL_CLOSE);
  SQLFreeStmt(h->stmt, SQL_UNBIND);
  if (SQLOK(r)) {
    snprintf(buf, BIGBUF, "update t_map set updatetime = %s where lfn_id = %d and lrc_id = %d and sender_id = %d",
	    todate(ts, dbuf), (int) lfn_id, (int) lrc_id, (int) sender_id);
    r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
    if (!SQLNODATAOK(r)) {
      rc = GLOBUS_RLS_DBERROR;
      goto error;
    }
  } else if (r == SQL_NO_DATA) {
    /*
     * We're adding a new mapping, so increment ref counts on lfn, lrc and
     * sender, or create if they don't exist.
     */
    if ((rc = updateref(h, T_RLILFN, lfn, 1, 1, (SQLINTEGER *) &lfn_id, NULL,
			errmsg, adjcount)) != GLOBUS_RLS_SUCCESS)
      goto error;
    if ((rc = updateref(h, T_RLILRC, lrc, 1, 1, (SQLINTEGER *) &lrc_id, NULL,
			errmsg, adjcount)) != GLOBUS_RLS_SUCCESS)
      goto error;
    if ((rc = updateref(h, T_RLISENDER, sender, 1, 1, (SQLINTEGER *) &sender_id, NULL,
			errmsg, adjcount)) != GLOBUS_RLS_SUCCESS)
      goto error;

    snprintf(buf, BIGBUF, "insert into t_map (lfn_id,lrc_id,sender_id,updatetime) values (%d,%d,%d,%s)",
	     (int) lfn_id, (int) lrc_id, (int) sender_id, todate(ts, dbuf));
    r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
    if (!SQLOK(r)) {
      rc = seterr_insertmap(h, r, "lfn_id", (int) lfn_id, "lrc_id", 
              (int) lrc_id, "sender_id", (int) sender_id, errmsg);
      goto error;
    }
    adjcount[T_RLIMAP]++;
  } else
    goto error;

  globus_mutex_unlock(&rlidbmtx);
  *lfnadded= adjcount[T_RLILFN];
  return endtran(h, SQL_COMMIT, errmsg, adjcount);

error:
  endtran(h, SQL_ROLLBACK, NULL, NULL);
  globus_mutex_unlock(&rlidbmtx);
  odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
  return rc;
}

/*
 * Turn autocommit mode off.  Used to do this once at connect time, but
 * a query, followed by an update of that data in another thread, followed
 * by the same query in the original thread, would read the old data and
 * never see the update.  So now we do it at start of each transaction,
 * and turn autocommit mode back on at end of transactions.  Errors are
 * logged and ignored.  Not sure why autocommit needs to be on for
 * queries to see the most recent updates.
 */
static void
begintran(db_handle_t *h, int *adjcount)

{
  SQLRETURN	r;
  char		errmsg[MAXERRMSG];
  int		i;

  r = SQLSetConnectAttr(h->dbc, SQL_ATTR_AUTOCOMMIT, SQL_AUTOCOMMIT_OFF, 
          SQL_IS_UINTEGER);
  if (!SQLOK(r))
    logit(LOG_WARNING, "begintran: autocommit(off): %s",
	  odbcerr(r, SQL_HANDLE_DBC, h->dbc, errmsg));
  if (adjcount)
    for (i = 0; i < T_NUM; i++)
      adjcount[i] = 0;
}

/*
 * End a transaction, turn autocommit mode back on (see comment for begintran).
 */
static int
endtran(db_handle_t *h, SQLSMALLINT comptype, char *errmsg, int *adjcount)

{
  SQLRETURN	r;
  int		rc;
  char		buf[MAXERRMSG];
  char		*msgp;
  int		i;

  msgp = errmsg ? errmsg : buf;

  r = SQLEndTran(SQL_HANDLE_DBC, h->dbc, comptype);
  if (!SQLOK(r)) {
    logit(LOG_WARNING, "endtran(%d): %s", comptype,
	  odbcerr(r, SQL_HANDLE_DBC, h->dbc, msgp));
    rc = GLOBUS_RLS_DBERROR;
  } else {
    if (adjcount) {
      globus_mutex_lock(&tablemtx);
      for (i = 0; i < T_NUM; i++)
	table[i].count += adjcount[i];
      globus_mutex_unlock(&tablemtx);
    }
    rc = GLOBUS_RLS_SUCCESS;
  }

  r = SQLSetConnectAttr(h->dbc, SQL_ATTR_AUTOCOMMIT,
			(SQLPOINTER) SQL_AUTOCOMMIT_ON, SQL_IS_UINTEGER);
  if (!SQLOK(r))
    logit(LOG_WARNING, "endtran(%d): autocommit(on): %s", comptype,
	  odbcerr(r, SQL_HANDLE_DBC, h->dbc, msgp));

  return rc;
}

/*
 * Perform query in buf, which returns 1 string column.  Invoke callback
 * for each result.
 */
static int
query_ret1(db_handle_t *h, char *buf, char *key,
	   db_str1_callback_t cb, void *a, char *errmsg)

{
  SQLRETURN	r;
  SQLCHAR		buf1[BUFLEN];
  SQLLEN	len;

  r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
  if (!SQLOK(r)) {
    odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
    return GLOBUS_RLS_DBERROR;
  }
  r = SQLBindCol(h->stmt, 1, SQL_C_CHAR, buf1, BUFLEN, &len);
  if (!SQLOK(r))
    goto error;

  while (1) {
    *buf1 = '\0';
    if ((r = SQLFetch(h->stmt)) == SQL_NO_DATA)
      break;
    if (!SQLOK(r))
      goto error;
    if (!cb(a, buf1))
      break;
  }
  SQLFreeStmt(h->stmt, SQL_CLOSE);
  SQLFreeStmt(h->stmt, SQL_UNBIND);

  return GLOBUS_RLS_SUCCESS;

 error:
  odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
  SQLFreeStmt(h->stmt, SQL_CLOSE);
  SQLFreeStmt(h->stmt, SQL_UNBIND);
  return GLOBUS_RLS_DBERROR;
}

/*
 * Perform query in buf, which returns 2 string columns.  Invoke callback
 * for each result.
 */
static int
query_ret2(db_handle_t *h, char *buf, char *key, db_str2_callback_t cb,
	   void *a, char *errmsg)

{
  SQLRETURN	r;
  SQLCHAR		buf1[BUFLEN];
  SQLCHAR		buf2[BUFLEN];
  SQLLEN	len;

  r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
  if (!SQLOK(r)) {
    odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
    return GLOBUS_RLS_DBERROR;
  }
  r = SQLBindCol(h->stmt, 1, SQL_C_CHAR, buf1, BUFLEN, &len);
  if (SQLOK(r))
    r = SQLBindCol(h->stmt, 2, SQL_C_CHAR, buf2, BUFLEN, &len);
  if (!SQLOK(r))
    goto error;

  while (1) {
    *buf1 = '\0';
    *buf2 = '\0';
    if ((r = SQLFetch(h->stmt)) == SQL_NO_DATA)
      break;
    if (!SQLOK(r))
      goto error;
    if (!cb(a, buf1, buf2))
      break;
  }
  SQLFreeStmt(h->stmt, SQL_CLOSE);
  SQLFreeStmt(h->stmt, SQL_UNBIND);

  return GLOBUS_RLS_SUCCESS;

 error:
  odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
  SQLFreeStmt(h->stmt, SQL_CLOSE);
  SQLFreeStmt(h->stmt, SQL_UNBIND);
  return GLOBUS_RLS_DBERROR;
}

/*
 * Perform query in buf, which returns 3 string columns.  Invoke callback
 * for each result.
 */
static int
query_ret3(db_handle_t *h, char *buf, char *key, db_str3_callback_t cb,
	   void *a, char *errmsg)

{
  SQLRETURN	r;
  SQLCHAR		buf1[BUFLEN];
  SQLCHAR		buf2[BUFLEN];
  SQLCHAR		buf3[BUFLEN];
  SQLLEN	len;

  r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
  if (!SQLOK(r)) {
    odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
    return GLOBUS_RLS_DBERROR;
  }
  r = SQLBindCol(h->stmt, 1, SQL_C_CHAR, buf1, BUFLEN, &len);
  if (SQLOK(r))
    r = SQLBindCol(h->stmt, 2, SQL_C_CHAR, buf2, BUFLEN, &len);
  if (SQLOK(r))
    r = SQLBindCol(h->stmt, 3, SQL_C_CHAR, buf3, BUFLEN, &len);
  if (!SQLOK(r))
    goto error;

  while (1) {
    *buf1 = '\0';
    *buf2 = '\0';
    *buf3 = '\0';
    if ((r = SQLFetch(h->stmt)) == SQL_NO_DATA)
      break;
    if (!SQLOK(r))
      goto error;
    if (!cb(a, buf1, buf2, buf3))
      break;
  }
  SQLFreeStmt(h->stmt, SQL_CLOSE);
  SQLFreeStmt(h->stmt, SQL_UNBIND);

  return GLOBUS_RLS_SUCCESS;

 error:
  odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
  SQLFreeStmt(h->stmt, SQL_CLOSE);
  SQLFreeStmt(h->stmt, SQL_UNBIND);
  return GLOBUS_RLS_DBERROR;
}

/*
 * Perform query in buf, which returns 3 string columns.  Invoke callback
 * for each result.
 */
static int
query_ret4(db_handle_t *h, char *buf, char *key, db_str4_callback_t cb,
	   void *a, char *errmsg)

{
  SQLRETURN	r;
  SQLCHAR		buf1[BUFLEN];
  SQLCHAR		buf2[BUFLEN];
  SQLCHAR		buf3[BUFLEN];
  SQLCHAR		buf4[BUFLEN];
  SQLLEN	len;

  r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
  if (!SQLOK(r)) {
    odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
    return GLOBUS_RLS_DBERROR;
  }
  r = SQLBindCol(h->stmt, 1, SQL_C_CHAR, buf1, BUFLEN, &len);
  if (SQLOK(r))
    r = SQLBindCol(h->stmt, 2, SQL_C_CHAR, buf2, BUFLEN, &len);
  if (SQLOK(r))
    r = SQLBindCol(h->stmt, 3, SQL_C_CHAR, buf3, BUFLEN, &len);
  if (SQLOK(r))
    r = SQLBindCol(h->stmt, 4, SQL_C_CHAR, buf4, BUFLEN, &len);
  if (!SQLOK(r))
    goto error;

  while (1) {
    *buf1 = '\0';
    *buf2 = '\0';
    *buf3 = '\0';
    *buf4 = '\0';
    if ((r = SQLFetch(h->stmt)) == SQL_NO_DATA)
      break;
    if (!SQLOK(r))
      goto error;
    if (!cb(a, buf1, buf2, buf3, buf4))
      break;
  }
  SQLFreeStmt(h->stmt, SQL_CLOSE);
  SQLFreeStmt(h->stmt, SQL_UNBIND);

  return GLOBUS_RLS_SUCCESS;

 error:
  odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
  SQLFreeStmt(h->stmt, SQL_CLOSE);
  SQLFreeStmt(h->stmt, SQL_UNBIND);
  return GLOBUS_RLS_DBERROR;
}

/*
 * Add adjust to ref count for record matching name in table.  If
 * the record is not found then nferr is returned if insert is not set,
 * else the record is inserted.  The id and ref columns of the record are
 * returned in *id, *ref.
 */
static int
updateref(db_handle_t *h, int tidx, char *name, int adjust, int insert,
	  SQLINTEGER *id, SQLINTEGER *ref, char *errmsg, int *adjcount)

{
  char		buf[BIGBUF];
  SQLLEN	rows    = 0;
  SQLLEN	len;
  SQLRETURN	r;
  SQLINTEGER	tref;
  long      idbuf   = 0;
  long      refbuf  = 0;

  snprintf(buf, BIGBUF, "update %s set ref = ref + %d where name = '%s'",
	   table[tidx].name, adjust, name);
  r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
  if (!SQLNODATAOK(r)) {
    odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
    return GLOBUS_RLS_DBERROR;
  }
  if (SQLOK(r)) {
    SQLRowCount(h->stmt, &rows);
    if (rows > 0)
      goto ok;
  }
  if (!insert) {
    strncpy(errmsg, name, MAXERRMSG);
    return table[tidx].nferr;
  }
  snprintf(buf, BIGBUF, "insert into %s (name, ref) values ('%s', 1)",
	   table[tidx].name, name);
  r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
  if (!SQLOK(r)) {
    odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
    return GLOBUS_RLS_DBERROR;
  }
  adjcount[tidx]++;

 ok:	/* Get ID of record we just updated/inserted	*/
  if (!ref)
    ref = &tref;
  snprintf(buf, BIGBUF, "select id, ref from %s where name = '%s'",
	   table[tidx].name, name);
  r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
  if (!SQLOK(r)) {
    odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
    return GLOBUS_RLS_DBERROR;
  }

  r = SQLBindCol(h->stmt, 1, SQL_C_LONG, &idbuf, sizeof(idbuf), &len);
  if (SQLOK(r))
    r = SQLBindCol(h->stmt, 2, SQL_C_LONG, &refbuf, sizeof(refbuf), &len);
  if (!SQLOK(r))
    goto error;

  r = SQLFetch(h->stmt);
  SQLFreeStmt(h->stmt, SQL_CLOSE);
  SQLFreeStmt(h->stmt, SQL_UNBIND);
  if (r == SQL_NO_DATA) {
    strncpy(errmsg, name, MAXERRMSG);
    return table[tidx].nferr;
  }
  if (!SQLOK(r)) {
    odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
    return GLOBUS_RLS_DBERROR;
  }
  *id = (SQLINTEGER) idbuf;
  *ref = (SQLINTEGER) refbuf;
  return GLOBUS_RLS_SUCCESS;

 error:
  odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
  SQLFreeStmt(h->stmt, SQL_CLOSE);
  SQLFreeStmt(h->stmt, SQL_UNBIND);
  return GLOBUS_RLS_DBERROR;
}

/*
 * Get size of table.  table[] should already be locked.
 */
static void
gettablecount(db_handle_t *h, int tidx)

{
  SQLRETURN	r;
  SQLLEN	len;
  char		buf[BIGBUF];
  char		errmsg[MAXERRMSG];
  long      lbuf    = 0;

  snprintf(buf, BIGBUF, "select count(*) from %s", table[tidx].name);
  r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
  if (!SQLOK(r)) {
    logit(LOG_WARNING, "gettablecount(%d): %s", tidx,
	  odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg));
    return;
  }
  r = SQLBindCol(h->stmt, 1, SQL_C_LONG, &lbuf, sizeof(lbuf), &len);
  if (!SQLOK(r)) {
    SQLFreeStmt(h->stmt, SQL_CLOSE);
    SQLFreeStmt(h->stmt, SQL_UNBIND);
    logit(LOG_WARNING, "gettablecount(%d): %s", tidx,
	  odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg));
    return;
  }
  r = SQLFetch(h->stmt);
  SQLFreeStmt(h->stmt, SQL_CLOSE);
  SQLFreeStmt(h->stmt, SQL_UNBIND);
  if (!SQLOK(r)) {
    logit(LOG_WARNING, "gettablecount(%d): %s", tidx,
	  odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg));
    return;
  }
  table[tidx].count = (int) lbuf;
}

static int
getid(db_handle_t *h, char *s, char *tab, int nferr, SQLINTEGER *id, 
        char *errmsg)

{
  char		buf[BIGBUF];
  SQLLEN	len;
  SQLRETURN	r;
  long      lbuf    = 0;

  snprintf(buf, BIGBUF, "select id from %s where name = '%s'", tab, s);
  r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
  if (!SQLOK(r)) {
    odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
    return GLOBUS_RLS_DBERROR;
  }
  r = SQLBindCol(h->stmt, 1, SQL_C_LONG, &lbuf, sizeof(lbuf), &len);
  if (!SQLOK(r))
    goto error;
  r = SQLFetch(h->stmt);
  SQLFreeStmt(h->stmt, SQL_CLOSE);
  SQLFreeStmt(h->stmt, SQL_UNBIND);
  if (r == SQL_NO_DATA) {
    strncpy(errmsg, s, MAXERRMSG);
    return nferr;
  }
  if (!SQLOK(r)) {
    odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
    return GLOBUS_RLS_DBERROR;
  }
  *id = (SQLINTEGER) lbuf;
  return GLOBUS_RLS_SUCCESS;

 error:
  odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
  SQLFreeStmt(h->stmt, SQL_CLOSE);
  SQLFreeStmt(h->stmt, SQL_UNBIND);
  return GLOBUS_RLS_DBERROR;
}

/*
 * Delete attribute values for object (which is being deleted).  Should
 * already be in a transaction.  This is made much more complex by
 * MySQLs lack of sub-selects.  We'd like to write:
 *
 *  delete from t_xxx_attr
 *    where obj_id = X and
 *	  attr_id in (select attr_id from t_attribute
 *		        where objtype = Y and type = xxx)
 *
 * But we can't.  So for each value type get attr_ids for this objtype
 * and delete individually.
 */
static SQLRETURN
deleteattrvals(db_handle_t *h, globus_rls_obj_type_t objtype, SQLINTEGER obj_id)

{
  IDLIST	*attrids = NULL;
  IDLIST	*p;
  char		buf[BIGBUF];
  int		i;
  SQLRETURN	r;
  SQLLEN	len;
  long		attr_id = 0;
  long		type    = 0;
  char		*valtab;

  /*
   * Get list of attr_ids for objtype.
   */
  snprintf(buf, BIGBUF, "select id, type from t_attribute where objtype = %d",
	   objtype);
  r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
  if (SQLOK(r))
    r = SQLBindCol(h->stmt, 1, SQL_C_LONG, &attr_id, sizeof(attr_id), &len);
  if (SQLOK(r))
    r = SQLBindCol(h->stmt, 2, SQL_C_LONG, &type, sizeof(type), &len);
  if (!SQLOK(r))
    goto error;
  while (1) {
    if ((r = SQLFetch(h->stmt)) == SQL_NO_DATA)
      break;
    if (!SQLOK(r))
      goto error;
    addid(&attrids, (SQLINTEGER) attr_id, (SQLINTEGER) type);
  }
  SQLFreeStmt(h->stmt, SQL_CLOSE);
  SQLFreeStmt(h->stmt, SQL_UNBIND);

  /*
   * Remove value records for obj_id.
   */
  for (p = attrids; p; p = p->nxt)
    for (i = 0; i < p->n; i++) {
      if ((valtab = valuetype2table(p->vals[i])) == NULL)
	continue;
      snprintf(buf,BIGBUF,"delete from %s where obj_id = %d and attr_id = %d",
	       valtab, (int) obj_id, (int) p->ids[i]);
      r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
      if (!SQLNODATAOK(r))
        goto error;
    }
  freeidlist(attrids);

  return SQL_SUCCESS;

 error:
  SQLFreeStmt(h->stmt, SQL_CLOSE);
  SQLFreeStmt(h->stmt, SQL_UNBIND);
  freeidlist(attrids);
  return r;
}

/*
 * Get ids and other values needed to add or remove an attribute to/from an
 * object.
 */
static int
attrinfo(db_handle_t *h, globus_rls_obj_type_t objtype, char *name, char *key,
	 SQLINTEGER *attr_id, globus_rls_attr_type_t *type, SQLINTEGER *obj_id,
	 char **valtab, char *errmsg)

{
  char		buf[BIGBUF];
  SQLRETURN	r;
  int		rc;
  SQLLEN	len;
  char		*table;
  int		nferr;
  long      attr_id_buf = 0;
  long      type_buf    = 0;

  /* First get attribute id and type			*/
  snprintf(buf, BIGBUF, "select id, type from t_attribute where name = '%s' and objtype = %d", name, objtype);
  r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
  if (SQLOK(r))
    r = SQLBindCol(h->stmt, 1, SQL_C_LONG, &attr_id_buf, sizeof(attr_id_buf), &len);
  if (SQLOK(r))
    r = SQLBindCol(h->stmt, 2, SQL_C_LONG, &type_buf, sizeof(type_buf), &len);
  if (SQLOK(r))
    r = SQLFetch(h->stmt);
  SQLFreeStmt(h->stmt, SQL_CLOSE);
  SQLFreeStmt(h->stmt, SQL_UNBIND);
  if (r == SQL_NO_DATA) {
    rc = GLOBUS_RLS_ATTR_NEXIST;
    strncpy(errmsg, name, MAXERRMSG);
    return GLOBUS_RLS_ATTR_NEXIST;
  } else if (!SQLOK(r)) {
    odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
    return GLOBUS_RLS_DBERROR;
  }
  *attr_id = (SQLINTEGER) attr_id_buf;
  *type = (globus_rls_attr_type_t) type_buf;
  if ((*valtab = valuetype2table(*type)) == NULL) {
    sprintf(errmsg, "%d", *type);
    return GLOBUS_RLS_INV_ATTR_TYPE;
  }

  if (key) {
    if ((table = objtype2table(objtype, &nferr)) == NULL) {
      sprintf(errmsg, "%d", objtype);
      return GLOBUS_RLS_INV_OBJ_TYPE;
    }
    return getid(h, key, table, nferr, obj_id, errmsg);
  } else
    return GLOBUS_RLS_SUCCESS;
}

/*
 * newidlist and addid are for remembering lfn_ids or sender_ids of entries
 * that have been expired.  After the map entries are removed from
 * t_map the t_lfn and t_sender records are removed (if there are no
 * other map entries for them).  If we run out of memory it's logged but
 * otherwise ignored.
 */
static IDLIST *
newidlist(IDLIST *list)

{
  IDLIST	*p;

  if ((p = (IDLIST *) globus_libc_malloc(sizeof(IDLIST))) == NULL) {
    logit(LOG_WARNING, "newidlist: No memory");
    return NULL;
  }
  p->n = 0;
  p->nxt = list;
  return p;
}

/*
 * Add id to list.
 */
static void
addid(IDLIST **listp, SQLINTEGER id, SQLINTEGER val)

{
  if (!*listp || (*listp)->n >= IDLISTSIZE)
    if ((*listp = newidlist(*listp)) == NULL)
      return;
  (*listp)->ids[(*listp)->n] = id;
  (*listp)->vals[(*listp)->n++] = val;
}

static void
freeidlist(IDLIST *list)

{
  IDLIST	*p;

  while ((p = list)) {
    list = list->nxt;
    globus_libc_free(p);
  }
}

/*
 * An insert statement failed, check if it failed because key alreay exists
 * in table (in which case return err), else return generic error code
 * GLOBUS_RLS_DBERROR.  Also fills in errmsg.
 */
static int
seterr_insert(db_handle_t *h, SQLRETURN r, char *table, char *name, int err,
	      char *errmsg)

{
  char		buf[BIGBUF];
  long		rows    = 0;
  SQLLEN	len;
  
  odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
  snprintf(buf, BIGBUF, "select count(*) from %s where name = '%s'", table, name);
  r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
  if (!SQLOK(r))
    return GLOBUS_RLS_DBERROR;
  /*
   * Use to use SQLRowCount() to see if we got a result, but some ODBC
   * drivers (including Easysoft's Oracle driver) don't support this on
   * after a select stmt.
   */
  r = SQLBindCol(h->stmt, 1, SQL_C_LONG, &rows, sizeof(rows), &len);
  if (SQLOK(r))
    r = SQLFetch(h->stmt);
  if (!SQLOK(r))
    goto error;
  SQLFreeStmt(h->stmt, SQL_CLOSE);
  SQLFreeStmt(h->stmt, SQL_UNBIND);
  if (rows > 0) {
    strncpy(errmsg, name, MAXERRMSG);
    return err;
  }
  return GLOBUS_RLS_DBERROR;

 error:
  SQLFreeStmt(h->stmt, SQL_CLOSE);
  SQLFreeStmt(h->stmt, SQL_UNBIND);
  odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
  return GLOBUS_RLS_DBERROR;
}

/*
 * An attribute insert statement failed, check if it failed because attr alreay
 * exists (in which case return err), else return generic error code
 * GLOBUS_RLS_DBERROR.  Also fills in errmsg.
 */
static int
seterr_insertattr(db_handle_t *h, SQLRETURN r, char *table, int obj_id,
		  int attr_id, int err, char *errmsg)

{
  char		buf[BIGBUF];
  long		rows    = 0;
  SQLLEN	len;
  
  odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
  snprintf(buf, BIGBUF, "select count(*) from %s where obj_id = %d and attr_id = %d",
	   table, obj_id, attr_id);
  r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
  if (!SQLOK(r))
    return GLOBUS_RLS_DBERROR;
  /*
   * Use to use SQLRowCount() to see if we got a result, but some ODBC
   * drivers (including Easysoft's Oracle driver) don't support this on
   * after a select stmt.
   */
  r = SQLBindCol(h->stmt, 1, SQL_C_LONG, &rows, sizeof(rows), &len);
  if (SQLOK(r))
    r = SQLFetch(h->stmt);
  if (!SQLOK(r))
    goto error;
  SQLFreeStmt(h->stmt, SQL_CLOSE);
  SQLFreeStmt(h->stmt, SQL_UNBIND);
  if (rows > 0)
    return err;
  return GLOBUS_RLS_DBERROR;

 error:
  SQLFreeStmt(h->stmt, SQL_CLOSE);
  SQLFreeStmt(h->stmt, SQL_UNBIND);
  odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
  return GLOBUS_RLS_DBERROR;
}

/*
 * An insert into a map table failed, check if it failed because key alreay
 * exists in table (in which case return err), else return generic error code
 * GLOBUS_RLS_DBERROR.  Also fills in errmsg.
 */
static int
seterr_insertmap(db_handle_t *h, SQLRETURN r, char *c1, int id1, char *c2,
	 	 int id2, char *c3, int id3, char *errmsg)

{
  char		buf[BIGBUF];
  long		rows    = 0;
  SQLLEN	len;
  
  odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
  if (c3)
    snprintf(buf, BIGBUF, "select count(*) from t_map where %s = %d and %s = %d and %s = %d",
	     c1, id1, c2, id2, c3, id3);
  else
    snprintf(buf, BIGBUF, "select count(*) from t_map where %s = %d and %s = %d",
	     c1, id1, c2, id2);
  r = SQLExecDirect(h->stmt, (SQLCHAR *) buf, SQL_NTS);
  if (!SQLOK(r))
    return GLOBUS_RLS_DBERROR;
  /*
   * Use to use SQLRowCount() to see if we got a result, but some ODBC
   * drivers (including Easysoft's Oracle driver) don't support this on
   * after a select stmt.
   */
  r = SQLBindCol(h->stmt, 1, SQL_C_LONG, &rows, sizeof(rows), &len);
  if (SQLOK(r))
    r = SQLFetch(h->stmt);
  if (!SQLOK(r))
    goto error;
  SQLFreeStmt(h->stmt, SQL_CLOSE);
  SQLFreeStmt(h->stmt, SQL_UNBIND);
  if (rows > 0)
    return GLOBUS_RLS_MAPPING_EXIST;
  return GLOBUS_RLS_DBERROR;

 error:
  SQLFreeStmt(h->stmt, SQL_CLOSE);
  SQLFreeStmt(h->stmt, SQL_UNBIND);
  odbcerr(r, SQL_HANDLE_STMT, h->stmt, errmsg);
  return GLOBUS_RLS_DBERROR;
}

static char *
odbcerr(SQLRETURN rc, SQLSMALLINT htype, SQLHANDLE h, char *errmsg)

{
  SQLRETURN	r;
  SQLSMALLINT	len;
  SQLCHAR	state[10];
  SQLINTEGER	nativeerr;

  r = SQLGetDiagRec(htype, h, 1, state, &nativeerr, (SQLCHAR *) errmsg,
		    MAXERRMSG, &len);
  if (!SQLOK(r)) {
    logit(LOG_WARNING, "odbc_error: SQLGetDiagRec failed: %d", r);
    return "Can't get odbc error string";
  }
  return errmsg;
}

/*
 * Map object type to table it exists in, also set *nferr to "not found"
 * error appropriate for table.
 */
static char *
objtype2table(globus_rls_obj_type_t objtype, int *nferr)

{
  switch (objtype) {
    case globus_rls_obj_lrc_lfn:
      *nferr = GLOBUS_RLS_LFN_NEXIST;
      return "t_lfn";
    case globus_rls_obj_lrc_pfn:
      *nferr = GLOBUS_RLS_PFN_NEXIST;
      return "t_pfn";
    case globus_rls_obj_rli_lfn:
      *nferr = GLOBUS_RLS_LFN_NEXIST;
      return "t_lfn";
    case globus_rls_obj_rli_lrc:
      *nferr = GLOBUS_RLS_LRC_NEXIST;
      return "t_lrc";
    default:
      logit(LOG_WARNING, "objtype2table(%d): Unknown object type", objtype);
      return NULL;
  }
}

static char *
valuetype2table(globus_rls_attr_type_t type)

{
  switch (type) {
    case globus_rls_attr_type_date:	return "t_date_attr";
    case globus_rls_attr_type_flt:	return "t_flt_attr";
    case globus_rls_attr_type_int:	return "t_int_attr";
    case globus_rls_attr_type_str:	return "t_str_attr";
    default:
	logit(LOG_INFO, "valuetype2table(%d): Bad type", type);
	return NULL;
  }
}

/*
 * Set result limit if user wants to limit number of results.  Don't
 * know if this works for DBs other than MySQL.
 */
static char *
sqllimit(int offset, int reslimit, char *buf)

{
  if (reslimit > 0) {
    if (ismysql)
      sprintf(buf, " limit %d,%d", offset, reslimit);
    else
      sprintf(buf, " limit %d offset %d", reslimit, offset);
  } else
    *buf = '\0';
  return buf;
}

/*
 * For dbms' that don't support a default date format of YYYY-MM-DD HH:MM:SS
 * (oracle) use to_date, to_char functions to convert date strings to date
 * and vice versa.
 */
static char *
todate(char *datestr, char *buf)

{
  if (isoracle)
    sprintf(buf, "to_date('%s', 'YYYY-MM-DD HH24:MI:SS')", datestr);
  else
    sprintf(buf, "'%s'", datestr);
  return buf;
}

static char *
tochar(char *field, char *buf)

{
  if (isoracle) {
    sprintf(buf, "to_char(%s, 'YYYY-MM-DD HH24:MI:SS')", field);
    return buf;
  } else
    return field;
}

#ifdef COUNTDB
#undef SQLExecDirect
static SQLRETURN
mysqlexecdirect(SQLHSTMT s, SQLCHAR *b, int f)

{
  static globus_mutex_t	mtx;

  if (dbstatements == 0)
    globus_mutex_init(&mtx, GLOBUS_NULL);
  globus_mutex_lock(&mtx);
  dbstatements++;
  globus_mutex_unlock(&mtx);
  if (loglevel > 2)
    logit(LOG_DEBUG, "SQL: %s", b);
  return SQLExecDirect(s, b, f);
}
#endif
