// Copyright 2009 The Go Authors. All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

package os

import (
	"errors"
)

// Portable analogs of some common system call errors.
var (
	ErrInvalid    = errors.New("invalid argument")
	ErrPermission = errors.New("permission denied")
	ErrExist      = errors.New("file already exists")
	ErrNotExist   = errors.New("file does not exist")
)

// PathError records an error and the operation and file path that caused it.
type PathError struct {
	Op   string
	Path string
	Err  error
}

func (e *PathError) Error() string { return e.Op + " " + e.Path + ": " + e.Err.Error() }

// SyscallError records an error from a specific system call.
type SyscallError struct {
	Syscall string
	Err     error
}

func (e *SyscallError) Error() string { return e.Syscall + ": " + e.Err.Error() }

// NewSyscallError returns, as an error, a new SyscallError
// with the given system call name and error details.
// As a convenience, if err is nil, NewSyscallError returns nil.
func NewSyscallError(syscall string, err error) error {
	if err == nil {
		return nil
	}
	return &SyscallError{syscall, err}
}

// IsExist returns whether the error is known to report that a file or directory
// already exists. It is satisfied by ErrExist as well as some syscall errors.
func IsExist(err error) bool {
	return isExist(err)
}

// IsNotExist returns whether the error is known to report that a file or directory
// does not exist. It is satisfied by ErrNotExist as well as some syscall errors.
func IsNotExist(err error) bool {
	return isNotExist(err)
}

// IsPermission returns whether the error is known to report that permission is denied.
// It is satisfied by ErrPermission as well as some syscall errors.
func IsPermission(err error) bool {
	return isPermission(err)
}
