# - Find Feel
# This module looks for Feel (Library for the Finite Element Method) support
# it will define the following values
#  FEELPP_INCLUDE_DIR = where feel/feelcore/feel.hpp can be found
#  FEELPP_LIBRARY    = the library to link in

#should check the version of gcc for -std=c++0x ou -std=c++11
set( CMAKE_CXX_FLAGS "${CMAKE_CXX_FLAGS} -std=c++0x" )
IF("${CMAKE_CXX_COMPILER_ID}" MATCHES "Clang")
  set( CMAKE_CXX_FLAGS "${CMAKE_CXX_FLAGS}  -std=c++11 -stdlib=libstdc++" )
ENDIF()

LIST(REMOVE_DUPLICATES CMAKE_CXX_FLAGS)
LIST(REMOVE_DUPLICATES CMAKE_CXX_FLAGS_DEBUG)
LIST(REMOVE_DUPLICATES CMAKE_CXX_FLAGS_RELEASE)

INCLUDE(CheckIncludeFile)
INCLUDE(CheckIncludeFiles)
INCLUDE(CheckIncludeFileCXX)
INCLUDE(CheckFunctionExists)
INCLUDE(CheckSymbolExists)
INCLUDE(CheckCXXSourceCompiles)
INCLUDE(CheckLibraryExists)

OPTION(FEELPP_ENABLE_SYSTEM_EIGEN3 "enable system eigen3 support" ON)



OPTION(FEELPP_ENABLE_MOVE_SEMANTICS "enable move semantics(elision)" ON )
OPTION(FEELPP_ENABLE_INSTANTIATION_MODE "Instantiation mode" ON )
OPTION(FEELPP_ENABLE_MPI_MODE "Instantiation mode" ON )
OPTION(FEELPP_ENABLE_SCHED_SLURM "Enable Feel++ slurm submission scripts generation" OFF)
OPTION(FEELPP_ENABLE_SCHED_CCC "Enable Feel++ tgcc/ccc submission scripts generation" OFF)
OPTION(FEELPP_ENABLE_SCHED_LOADLEVELER "Enable Feel++ ibm(supermuc) submission scripts generation" OFF)
OPTION(FEELPP_ENABLE_TBB "enable feel++ TBB support" OFF)
OPTION(FEELPP_ENABLE_SLEPC "enable feel++ SLEPc support" ON)
OPTION(FEELPP_ENABLE_TRILINOS "enable feel++ Trilinos support" OFF)
OPTION(FEELPP_ENABLE_EXODUS "enable feel++ Exodus support" OFF)
if ( APPLE )
  OPTION(FEELPP_ENABLE_OPENTURNS "enable feel++ OpenTURNS support" OFF)
else()
  OPTION(FEELPP_ENABLE_OPENTURNS "enable feel++ OpenTURNS support" ON)
endif()
OPTION(FEELPP_ENABLE_OCTAVE "Enable Feel++/Octave interface" OFF)

OPTION(FEELPP_ENABLE_OPENGL "enable feel++ OpenGL support" ON)
OPTION(FEELPP_DISABLE_EIGEN_ALIGNMENT "disable alignement (hence vectorization) in Eigen" OFF)

# enable mpi mode
IF ( FEELPP_ENABLE_MPI_MODE )
  SET( FEELPP_ENABLE_MPI_MODE 1 )
ENDIF()

# disable alignement
MARK_AS_ADVANCED(FEELPP_DISABLE_EIGEN_ALIGNMENT)
if ( FEELPP_DISABLE_EIGEN_ALIGNMENT )
  add_definitions(-DEIGEN_DONT_ALIGN=1 -DEIGEN_DONT_VECTORIZE=1)
  message(STATUS "Disabling alignment and vectorisation in Feel++/Eigen")
endif()


# enable move semantics
MARK_AS_ADVANCED(FEELPP_ENABLE_MOVE_SEMANTICS)
IF ( FEELPP_ENABLE_MOVE_SEMANTICS )
  SET( BOOST_UBLAS_MOVE_SEMANTICS 1 CACHE STRING "Enable Boost Ublas move semantics" FORCE )
  ADD_DEFINITIONS( -DBOOST_UBLAS_MOVE_SEMANTICS )
ENDIF( FEELPP_ENABLE_MOVE_SEMANTICS )

# enable instantiation
MARK_AS_ADVANCED(FEELPP_ENABLE_INSTANTIATION_MODE)
IF ( FEELPP_ENABLE_INSTANTIATION_MODE )
  SET( FEELPP_INSTANTIATION_MODE 1 )
ENDIF()
SET(FEELPP_MESH_MAX_ORDER "5" CACHE STRING "maximum geometrical order in templates to instantiate" )

if ( FEELPP_ENABLE_TBB )
  FIND_PACKAGE(TBB)
  IF ( TBB_FOUND )
    INCLUDE_DIRECTORIES( ${TBB_INCLUDE_DIR} )
    SET(FEELPP_LIBRARIES ${TBB_LIBRARIES})
    SET(FEELPP_ENABLED_OPTIONS "${FEELPP_ENABLED_OPTIONS} Tbb" )
  ENDIF (TBB_FOUND )
endif()

FIND_PACKAGE(OpenMP)

# on APPLE enfore the use of macports openmpi version
if ( APPLE )
  if ( EXISTS /opt/local/lib/openmpi/bin/mpic++ )
    set(MPI_COMPILER /opt/local/lib/openmpi/bin/mpic++)
  endif()

  #  set(MPI_LIBRARY "MPI_LIBRARY-NOTFOUND" )
  MESSAGE(STATUS "Use mpi compiler ${MPI_COMPILER}")

endif( APPLE )
FIND_PACKAGE(MPI REQUIRED)
IF ( MPI_FOUND )
  SET(CMAKE_REQUIRED_INCLUDES "${MPI_INCLUDE_PATH};${CMAKE_REQUIRED_INCLUDES}")
  SET( FEELPP_HAS_MPI 1 )
  SET( FEELPP_HAS_MPI_H 1 )
  ADD_DEFINITIONS( -DFEELPP_HAS_MPI=1 -DFEELPP_HAS_MPI_H=1 )
  SET(FEELPP_BOOST_MPI mpi)
  SET(FEELPP_LIBRARIES ${MPI_LIBRARIES} ${FEELPP_LIBRARIES})
  INCLUDE_DIRECTORIES(${MPI_INCLUDE_PATH})
  SET(FEELPP_ENABLED_OPTIONS "${FEELPP_ENABLED_OPTIONS} Mpi" )
ENDIF()



Check_Include_File_CXX(dlfcn.h FEELPP_HAS_DLFCN_H)
if ( FEELPP_HAS_DLFCN_H )
  add_definitions(-DFEELPP_HAS_DLFCN_H)
endif()
CHECK_LIBRARY_EXISTS (dl dlopen "" FEELPP_HAS_LIBDL)
IF (FEELPP_HAS_LIBDL)
  set(DL_LIBRARY dl)
  SET (FEELPP_HAS_DLOPEN 1)
  add_definitions(-DFEELPP_HAS_DLOPEN)
  SET(FEELPP_LIBRARIES ${DL_LIBRARY} ${FEELPP_LIBRARIES})
ELSE ()
  CHECK_FUNCTION_EXISTS (dlopen FEELPP_HAS_DLOPEN)
ENDIF (FEELPP_HAS_LIBDL)


#
# Blas and Lapack
#
if (APPLE)
  FIND_LIBRARY(ATLAS_LIBRARY
    NAMES
    atlas
    PATHS
    /opt/local/lib/lib
    NO_DEFAULT_PATH
    )
  message(STATUS "ATLAS: ${ATLAS_LIBRARY}" )
  IF( ATLAS_LIBRARY )
    SET(FEELPP_LIBRARIES ${ATLAS_LIBRARY} ${FEELPP_LIBRARIES})
  ENDIF()
  FIND_PACKAGE(LAPACK )
else (APPLE)
  FIND_PACKAGE(LAPACK REQUIRED)
endif (APPLE)
SET(FEELPP_LIBRARIES  ${LAPACK_LIBRARIES} ${FEELPP_LIBRARIES})

FIND_PACKAGE(Boost COMPONENTS date_time filesystem system program_options unit_test_framework signals  ${FEELPP_BOOST_MPI} regex  serialization)
if(Boost_FOUND)
  IF(Boost_MAJOR_VERSION EQUAL "1" AND Boost_MINOR_VERSION GREATER "51")
    add_definitions(-DBOOST_RESULT_OF_USE_TR1)
    message(STATUS "added -DBOOST_RESULT_OF_USE_TR1" )
  ENDIF()
  IF("${CMAKE_CXX_COMPILER_ID}" MATCHES "Clang")
    # ensures that boost.signals2 compiles with clang++ >= 3.1
    IF(Boost_MAJOR_VERSION EQUAL "1" AND Boost_MINOR_VERSION GREATER "52")
      add_definitions(-DBOOST_NO_CXX11_VARIADIC_TEMPLATES)
      message(STATUS "added -DBOOST_NO_CXX11_VARIADIC_TEMPLATES" )
    ELSE()
      add_definitions(-DBOOST_NO_VARIADIC_TEMPLATES)
      message(STATUS "added -DBOOST_NO_VARIADIC_TEMPLATES" )
    ENDIF()
  ENDIF()
endif()

OPTION(BOOST_ENABLE_TEST_DYN_LINK "enable boost test with dynamic lib" ON)
MARK_AS_ADVANCED(BOOST_ENABLE_TEST_DYN_LINK)

set(Boost_ADDITIONAL_VERSIONS "1.39" "1.40" "1.41" "1.42" "1.43" "1.44" "1.45" "1.46" "1.47" "1.48" "1.49" "1.50" "1.51" "1.52" "1.53")
set( BOOST_PARAMETER_MAX_ARITY 20 )
#set( BOOST_FILESYSTEM_VERSION 2)
set( BOOST_FILESYSTEM_VERSION 3)
if (BOOST_ENABLE_TEST_DYN_LINK)
  add_definitions( -DBOOST_PARAMETER_MAX_ARITY=${BOOST_PARAMETER_MAX_ARITY} -DBOOST_TEST_DYN_LINK -DBOOST_FILESYSTEM_VERSION=${BOOST_FILESYSTEM_VERSION})
else (BOOST_ENABLE_TEST_DYN_LINK)
  add_definitions( -DBOOST_PARAMETER_MAX_ARITY=${BOOST_PARAMETER_MAX_ARITY} -DBOOST_FILESYSTEM_VERSION=${BOOST_FILESYSTEM_VERSION})
endif (BOOST_ENABLE_TEST_DYN_LINK)

# this fix an issue with boost filesystem: boost is usually no compiled with
# std=c++0x and we compile with it, this causes problems with the macro
# BOOST_SCOPED_ENUM macros whose behavior differs in both case and would
# generate different c++ codes and undefined references at link time.
# in a short future, this should not be necessary anymore
ADD_DEFINITIONS(-DBOOST_NO_SCOPED_ENUMS)
IF(Boost_MAJOR_VERSION EQUAL "1" AND Boost_MINOR_VERSION GREATER "51")
  ADD_DEFINITIONS(-DBOOST_NO_CXX11_SCOPED_ENUMS)
endif()

INCLUDE_DIRECTORIES(${Boost_INCLUDE_DIR}   ${BOOST_INCLUDE_PATH})

SET(FEELPP_LIBRARIES ${Boost_LIBRARIES} ${FEELPP_LIBRARIES})

set(INCLUDE_INSTALL_DIR ${CMAKE_INSTALL_PREFIX}/include/feel)

INCLUDE_DIRECTORIES(BEFORE contrib/)

#FIND_PACKAGE(GINAC)
#IF( GINAC_FOUND )
#  set( FEELPP_HAS_GINAC 1 )
#  INCLUDE_DIRECTORIES( GINAC_INCLUDE_DIRS )
#  SET(FEELPP_LIBRARIES ${GINAC_LIBRARIES} ${FEELPP_LIBRARIES})
#  SET(FEELPP_ENABLED_OPTIONS "${FEELPP_ENABLED_OPTIONS} GINAC" )
#ENDIF()

add_definitions(-DHAVE_LIBDL)
# cln and ginac
if ( EXISTS ${CMAKE_CURRENT_SOURCE_DIR}/feel AND EXISTS ${CMAKE_CURRENT_SOURCE_DIR}/contrib )
  add_definitions(-DIN_GINAC -DHAVE_LIBDL)
  include_directories(${FEELPP_BUILD_DIR}/contrib/cln/include ${FEELPP_SOURCE_DIR}/contrib/ginac/ ${FEELPP_BUILD_DIR}/contrib/ginac/ ${FEELPP_SOURCE_DIR}/contrib/ginac/ginac ${FEELPP_BUILD_DIR}/contrib/ginac/ginac )
  SET(FEELPP_LIBRARIES feelpp_ginac ${CLN_LIBRARIES} ${FEELPP_LIBRARIES} ${CMAKE_DL_LIBS} )
  set(DL_LIBS ${CMAKE_DL_LIBS})
  add_subdirectory(contrib/ginac)
endif()

if ( FEELPP_ENABLE_EXODUS )
  include_directories(${FEELPP_SOURCE_DIR}/contrib/exodus-5.24/exodus/cbind/include/)
  add_subdirectory(contrib/exodus-5.24/exodus)
  #add_subdirectory(contrib/exodus-5.24/nemesis)
  set(FEELPP_HAS_EXODUS 1)
  SET(FEELPP_ENABLED_OPTIONS "${FEELPP_ENABLED_OPTIONS} Exodus" )
endif()

#
# Eigen
#
if ( FEELPP_ENABLE_SYSTEM_EIGEN3 )
FIND_PACKAGE(Eigen3)
endif()
if (NOT EIGEN3_FOUND AND EXISTS ${CMAKE_CURRENT_SOURCE_DIR}/feel AND EXISTS ${CMAKE_CURRENT_SOURCE_DIR}/contrib )
  option(EIGEN_BUILD_PKGCONFIG "Build pkg-config .pc file for Eigen" OFF)
  set(EIGEN_INCLUDE_INSTALL_DIR ${CMAKE_INSTALL_PREFIX}/include/feel)
  add_subdirectory(contrib/eigen)
  INCLUDE_DIRECTORIES( ${FEELPP_SOURCE_DIR}/contrib/eigen )
  SET(FEELPP_ENABLED_OPTIONS "${FEELPP_ENABLED_OPTIONS} Eigen3/Contrib" )
elseif( EIGEN3_FOUND )
  INCLUDE_DIRECTORIES( ${EIGEN3_INCLUDE_DIR} )
  SET(FEELPP_ENABLED_OPTIONS "${FEELPP_ENABLED_OPTIONS} Eigen3/System" )
endif()
message(STATUS "Eigen3: ${EIGEN3_INCLUDE_DIR}" )

#FIND_PACKAGE(Eigen2 REQUIRED)
#INCLUDE_DIRECTORIES( ${Eigen2_INCLUDE_DIR} )
#add_subdirectory(contrib/eigen)
#INCLUDE_DIRECTORIES( ${FEELPP_SOURCE_DIR}/contrib/eigen )
#add_definitions( -DEIGEN_NO_STATIC_ASSERT )

#
# Scotch
#
#CheckIncludeFileCXX( ptscotch.h FEELPP_HAS_PTSCOTCH_H )
#CheckIncludeFileCXX( scotch.h FEELPP_HAS_SCOTCH_H )
#
# Metis
#
FIND_PACKAGE(Metis)
if ( METIS_FOUND )
  INCLUDE_DIRECTORIES(${METIS_INCLUDE_DIR})
  #  LINK_DIRECTORIES(${METIS_LIBRARIES})
  SET(FEELPP_LIBRARIES ${METIS_LIBRARY} ${FEELPP_LIBRARIES})
  SET(FEELPP_ENABLED_OPTIONS "${FEELPP_ENABLED_OPTIONS} Metis" )
endif( METIS_FOUND )

#
# Ann
#
FIND_PACKAGE(ANN)
if ( ANN_FOUND )
  set(FEELPP_HAS_ANN_H 1)
  INCLUDE_DIRECTORIES( ${ANN_INCLUDE_DIR} )
  SET(FEELPP_LIBRARIES ${ANN_LIBRARIES} ${FEELPP_LIBRARIES})
  SET(FEELPP_ENABLED_OPTIONS "${FEELPP_ENABLED_OPTIONS} ANN" )
endif()

#
# GLPK
#
FIND_PACKAGE(GLPK)
if ( GLPK_FOUND )
  set(FEELPP_HAS_GLPK_H 1)
  INCLUDE_DIRECTORIES( ${GLPK_INCLUDE_DIR} )
  SET(FEELPP_LIBRARIES ${GLPK_LIBRARIES} ${FEELPP_LIBRARIES})
  SET(FEELPP_ENABLED_OPTIONS "${FEELPP_ENABLED_OPTIONS} GLPK" )
endif()

# google perf tools
option(FEELPP_ENABLE_GOOGLEPERFTOOLS "Enable Google Perf Tools (tcmalloc, stracktrace and profiler)" OFF)
if ( FEELPP_ENABLE_GOOGLEPERFTOOLS )
  find_package(GooglePerfTools)
  if ( GOOGLE_PERFTOOLS_FOUND )
    set(FEELPP_HAS_GPERFTOOLS 1 )
    message(STATUS "Google PerfTools: ${TCMALLOC_LIBRARIES} ${STACKTRACE_LIBRARIES} ${PROFILER_LIBRARIES}")
    include_directories(${GOOGLE_PERFTOOLS_INCLUDE_DIR})
    SET(FEELPP_LIBRARIES  ${FEELPP_LIBRARIES} ${TCMALLOC_LIBRARIES})
    SET(FEELPP_ENABLED_OPTIONS "${FEELPP_ENABLED_OPTIONS} GooglePerfTools" )
  endif()
endif( FEELPP_ENABLE_GOOGLEPERFTOOLS )

option(FEELPP_ENABLE_DDT "Enable DDT support" OFF)
if ( FEELPP_ENABLE_DDT )
  find_package(DDT)
  if ( DDT_FOUND )
    message(STATUS "DDT: ${DDT_LIBRARIES}")
    SET(FEELPP_LIBRARIES  ${FEELPP_LIBRARIES} ${DDT_LIBRARIES})
    SET(FEELPP_ENABLED_OPTIONS "${FEELPP_ENABLED_OPTIONS} DDT" )
  endif()
endif( FEELPP_ENABLE_DDT )

# google gflags
find_package(GFLAGS REQUIRED)

INCLUDE_DIRECTORIES( ${GFLAGS_INCLUDE_DIR} )

SET(FEELPP_LIBRARIES ${GFLAGS_LIBRARIES} ${FEELPP_LIBRARIES})
if ( ${GFLAGS_INCLUDE_DIR} MATCHES "/contrib/" )
  SET(FEELPP_ENABLED_OPTIONS "${FEELPP_ENABLED_OPTIONS} GFLAGS/Contrib" )
else()
  SET(FEELPP_ENABLED_OPTIONS "${FEELPP_ENABLED_OPTIONS} GFLAGS/System" )
endif()

# google glog
find_package(GLOG REQUIRED)

INCLUDE_DIRECTORIES( ${GLOG_INCLUDE_DIR} )
SET(FEELPP_LIBRARIES ${GLOG_LIBRARIES} ${FEELPP_LIBRARIES})
if ( ${GLOG_INCLUDE_DIR} MATCHES "/contrib/" )
  SET(FEELPP_ENABLED_OPTIONS "${FEELPP_ENABLED_OPTIONS} GLOG/Contrib" )
else()
  SET(FEELPP_ENABLED_OPTIONS "${FEELPP_ENABLED_OPTIONS} GLOG/System" )
endif()


# xml
find_package(LibXml2 2.6.27)

# Python libs
FIND_PACKAGE(PythonLibs)
if ( PYTHONLIBS_FOUND )
  message(STATUS "PythonLibs: ${PYTHON_INCLUDE_DIRS} ${PYTHON_LIBRARIES}")
  include_directories(${PYTHON_INCLUDE_DIRS})
  #   SET(FEELPP_LIBRARIES ${PYTHON_LIBRARIES} ${FEELPP_LIBRARIES})
  SET(FEELPP_ENABLED_OPTIONS "${FEELPP_ENABLED_OPTIONS} Python" )
endif()

#
# Python interp
#
FIND_PACKAGE(PythonInterp REQUIRED)
if(PYTHONINTERP_FOUND)
  execute_process(COMMAND
	${PYTHON_EXECUTABLE}
	-c "import sys; print sys.version[0:3]"
	OUTPUT_VARIABLE PYTHON_VERSION
	OUTPUT_STRIP_TRAILING_WHITESPACE)

  message(STATUS "Found python version ${PYTHON_VERSION}")
endif()

# metis
FIND_LIBRARY(METIS_LIBRARY
  NAMES
  metis
  PATHS
  $ENV{PETSC_DIR}/lib
  $ENV{PETSC_DIR}/$ENV{PETSC_ARCH}/lib
  #    "/opt/local/lib"
  )
message(STATUS "Metis: ${METIS_LIBRARY}" )
IF( METIS_LIBRARY )
  SET(FEELPP_LIBRARIES ${METIS_LIBRARY} ${FEELPP_LIBRARIES})
ENDIF()

FIND_LIBRARY(PARMETIS_LIBRARY
  NAMES
  parmetis
  PATHS
  $ENV{PETSC_DIR}/lib
  $ENV{PETSC_DIR}/$ENV{PETSC_ARCH}/lib
  )


IF( PARMETIS_LIBRARY )
  message(STATUS "Parmetis: ${PARMETIS_LIBRARY}" )
  SET(FEELPP_LIBRARIES ${PARMETIS_LIBRARY} ${FEELPP_LIBRARIES})
ENDIF()

FIND_LIBRARY(PTSCOTCHERREXIT_LIBRARY
  NAMES
  ptscotcherrexit
  PATHS
  $ENV{PETSC_DIR}/lib
  $ENV{PETSC_DIR}/$ENV{PETSC_ARCH}/lib
  /opt/local/lib/petsc/lib
  $ENV{PTSCOTCH_DIR}/lib
  )
message(STATUS "PTScotcherrexit: ${PTSCOTCHERREXIT_LIBRARY}" )
IF( PTSCOTCHERREXIT_LIBRARY )
  #message(STATUS "PTScotch: ${PTSCOTCH_LIBRARY}" )
  SET(FEELPP_LIBRARIES ${PTSCOTCHERREXIT_LIBRARY} ${FEELPP_LIBRARIES})
ENDIF()

FIND_LIBRARY(PTSCOTCHERR_LIBRARY
  NAMES
  ptscotcherr
  PATHS
  $ENV{PETSC_DIR}/lib
  $ENV{PETSC_DIR}/$ENV{PETSC_ARCH}/lib
  /opt/local/lib/petsc/lib
  $ENV{PTSCOTCH_DIR}/lib
  )
message(STATUS "PTScotcherr: ${PTSCOTCHERR_LIBRARY}" )
IF( PTSCOTCHERR_LIBRARY )
  #message(STATUS "PTScotch: ${PTSCOTCH_LIBRARY}" )
  SET(FEELPP_LIBRARIES ${PTSCOTCHERR_LIBRARY} ${FEELPP_LIBRARIES})
ENDIF()

FIND_LIBRARY(PTSCOTCH_LIBRARY
  NAMES
  ptscotch
  PATHS
  $ENV{PETSC_DIR}/lib
  $ENV{PETSC_DIR}/$ENV{PETSC_ARCH}/lib
  /opt/local/lib/petsc/lib
  $ENV{PTSCOTCH_DIR}/lib
  )
message(STATUS "PTScotch: ${PTSCOTCH_LIBRARY}" )

IF( PTSCOTCH_LIBRARY )
  SET(FEELPP_LIBRARIES ${PTSCOTCH_LIBRARY} ${FEELPP_LIBRARIES})
ENDIF()

FIND_LIBRARY(PTESMUMPS_LIBRARY
  NAMES
  ptesmumps
  PATHS
  $ENV{PETSC_DIR}/lib
  $ENV{PETSC_DIR}/$ENV{PETSC_ARCH}/lib
  /opt/local/lib/petsc/lib
  )
message(STATUS "PTESMUMPS: ${PTESMUMPS_LIBRARY}" )
IF( PTESMUMPS_LIBRARY )
  set( FEELPP_HAS_MUMPS 1 )
  SET(FEELPP_LIBRARIES ${PTESMUMPS_LIBRARY} ${FEELPP_LIBRARIES})
ENDIF()


FIND_LIBRARY(ML_LIBRARY
  NAMES
  ml
  PATHS
  $ENV{PETSC_DIR}/lib
  )
message(STATUS "ML: ${ML_LIBRARY}" )
IF ( ML_LIBRARY )
  SET(FEELPP_LIBRARIES ${ML_LIBRARY} ${FEELPP_LIBRARIES})
ENDIF()

if ( NOT GFORTRAN_LIBRARY )
  FIND_LIBRARY(GFORTRAN_LIBRARY
    NAMES
    gfortran
    PATHS
    $ENV{LIBRARY_PATH}
    /opt/local/lib
    /usr/lib/gcc/x86_64-linux-gnu/
    PATH_SUFFIXES
    gcc47 gcc46 gcc45 gcc44 4.7 4.6 4.5 4.4
    )
endif()

message(STATUS "gfortran lib: ${GFORTRAN_LIBRARY} ")
if ( GFORTRAN_LIBRARY )
  set( FEELPP_LIBRARIES ${GFORTRAN_LIBRARY} ${FEELPP_LIBRARIES})
endif()
FIND_PACKAGE(MUMPS)
if ( GFORTRAN_LIBRARY AND MUMPS_FOUND )
  set( FEELPP_HAS_MUMPS 1 )
  set( FEELPP_LIBRARIES ${MUMPS_LIBRARIES} ${FEELPP_LIBRARIES} )
endif()
FIND_LIBRARY(SUITESPARSECONFIG_LIBRARY
  NAMES
  suitesparseconfig
  PATHS
  $ENV{PETSC_DIR}/lib
  $ENV{PETSC_DIR}/$ENV{PETSC_ARCH}/lib
  $ENV{SUITESPARSE_DIR}/lib
  )
IF ( SUITESPARSECONFIG_LIBRARY )
  SET(FEELPP_LIBRARIES  ${SUITESPARSECONFIG_LIBRARY} ${FEELPP_LIBRARIES})
endif()
FIND_LIBRARY(AMD_LIBRARY
  NAMES
  amd
  PATHS
  $ENV{PETSC_DIR}/lib
  $ENV{PETSC_DIR}/$ENV{PETSC_ARCH}/lib
  $ENV{SUITESPARSE_DIR}/lib
  )
IF ( AMD_LIBRARY )
  SET(FEELPP_LIBRARIES  ${AMD_LIBRARY} ${FEELPP_LIBRARIES})
endif()

FIND_LIBRARY(COLAMD_LIBRARY
  NAMES
  colamd
  PATHS
  $ENV{PETSC_DIR}/lib
  $ENV{PETSC_DIR}/$ENV{PETSC_ARCH}/lib
  $ENV{SUITESPARSE_DIR}/lib
  )
IF ( COLAMD_LIBRARY )
  SET(FEELPP_LIBRARIES  ${COLAMD_LIBRARY} ${FEELPP_LIBRARIES})
endif()

FIND_LIBRARY(CHOLMOD_LIBRARY
  NAMES
  cholmod
  PATHS
  $ENV{PETSC_DIR}/lib
  $ENV{PETSC_DIR}/$ENV{PETSC_ARCH}/lib
  $ENV{SUITESPARSE_DIR}/lib
  )

FIND_LIBRARY(UMFPACK_LIBRARY
  NAMES
  umfpack
  PATHS
  $ENV{PETSC_DIR}/lib
  $ENV{PETSC_DIR}/$ENV{PETSC_ARCH}/lib
  $ENV{SUITESPARSE_DIR}/lib
  )
message(STATUS "SuiteSparseConfig: ${SUITESPARSECONFIG_LIBRARY}" )
message(STATUS "Amd: ${AMD_LIBRARY}" )
message(STATUS "ColAmd: ${COLAMD_LIBRARY}" )
message(STATUS "Cholmod: ${CHOLMOD_LIBRARY}" )
message(STATUS "Umfpack: ${UMFPACK_LIBRARY}" )
if ( AMD_LIBRARY AND CHOLMOD_LIBRARY AND UMFPACK_LIBRARY )
  SET(FEELPP_LIBRARIES ${UMFPACK_LIBRARY} ${CHOLMOD_LIBRARY} ${FEELPP_LIBRARIES})
endif()

FIND_LIBRARY(YAML_LIBRARY
  NAMES
  yaml
  PATHS
  $ENV{PETSC_DIR}/lib
  $ENV{PETSC_DIR}/$ENV{PETSC_ARCH}/lib
  $ENV{SUITESPARSE_DIR}/lib
  /opt/local/lib
  )
if ( YAML_LIBRARY )
  SET(FEELPP_LIBRARIES ${YAML_LIBRARY} ${FEELPP_LIBRARIES})
endif()
FIND_LIBRARY(HDF5_LIBRARY
  NAMES
  hdf5
  PATHS
  $ENV{PETSC_DIR}/lib
  $ENV{PETSC_DIR}/$ENV{PETSC_ARCH}/lib
  $ENV{SUITESPARSE_DIR}/lib
  /opt/local/lib
  )
if ( HDF5_LIBRARY )
  SET(FEELPP_LIBRARIES ${HDF5_LIBRARY} ${FEELPP_LIBRARIES})
endif()

#
# Petsc
#
FIND_PACKAGE( PETSc REQUIRED)
if ( PETSC_FOUND )
  SET(CMAKE_REQUIRED_INCLUDES "${PETSC_INCLUDES};${CMAKE_REQUIRED_INCLUDES}")
  SET(FEELPP_LIBRARIES ${PETSC_LIBRARIES} ${FEELPP_LIBRARIES})
  SET(BACKEND_PETSC petsc)
  INCLUDE_DIRECTORIES(
    ${PETSC_INCLUDE_DIR}
    ${PETSC_INCLUDE_CONF}
    )
  SET(FEELPP_ENABLED_OPTIONS "${FEELPP_ENABLED_OPTIONS} PETSc" )

endif( PETSC_FOUND )

#
# parpack
#
FIND_LIBRARY(PARPACK_LIBRARY NAMES parpack)
if (PARPACK_LIBRARY)
  SET(PARPACK_LIBRARIES ${PARPACK_LIBRARY})
  SET(FEELPP_LIBRARIES ${PARPACK_LIBRARIES} ${FEELPP_LIBRARIES})
endif()
MARK_AS_ADVANCED( PARPACK_LIBRARY )


#
# SLEPc
#
if (FEELPP_ENABLE_SLEPC)
  FIND_PACKAGE( SLEPc )
  if ( SLEPC_FOUND )
    SET(CMAKE_REQUIRED_INCLUDES "${SLEPC_INCLUDES};${CMAKE_REQUIRED_INCLUDES}")
    INCLUDE_DIRECTORIES( ${SLEPC_INCLUDE_DIR} )
    SET(FEELPP_LIBRARIES ${SLEPC_LIBRARIES} ${FEELPP_LIBRARIES})
    SET(FEELPP_ENABLED_OPTIONS "${FEELPP_ENABLED_OPTIONS} SLEPc" )
  endif(SLEPC_FOUND)
endif(FEELPP_ENABLE_SLEPC)


#
# Trilinos
#
if (FEELPP_ENABLE_TRILINOS)
  FIND_PACKAGE(Trilinos)
  if ( TRILINOS_FOUND )
    INCLUDE_DIRECTORIES(${TRILINOS_INCLUDE_DIR})
    SET(FEELPP_LIBRARIES ${TRILINOS_LIBRARIES} ${FEELPP_LIBRARIES})
    SET(BACKEND_TRILINOS trilinos)
    SET(FEELPP_ENABLED_OPTIONS "${FEELPP_ENABLED_OPTIONS} Trilinos" )
  endif( TRILINOS_FOUND )
endif (FEELPP_ENABLE_TRILINOS)

#
# OpenTURNS
#
IF ( FEELPP_ENABLE_OPENTURNS )
  FIND_PACKAGE( OpenTURNS )
  if ( OPENTURNS_FOUND )
    MESSAGE(STATUS "OpenTURNS Libraries: ${OpenTURNS_LIBRARIES}")
    MESSAGE(STATUS "OpenTURNS Headers: ${OpenTURNS_INCLUDE_DIRS}")
    INCLUDE_DIRECTORIES(${OpenTURNS_INCLUDE_DIRS})
    #SET(FEELPP_LIBRARIES ${OpenTURNS_LIBRARIES} ${FEELPP_LIBRARIES})
    # now OpenTURNS_LIBRARIES are used in crb_add_python_module
    SET(FEELPP_ENABLED_OPTIONS "${FEELPP_ENABLED_OPTIONS} OpenTURNS" )
  endif( OPENTURNS_FOUND )
endif()

#
# VTK
#
FIND_PACKAGE(VTK)
if ( VTK_FOUND )
  set(FEELPP_HAS_VTK 1)
  SET(VTK_LIBRARIES "-lvtkRendering -lvtkGraphics -lvtkImaging  -lvtkFiltering -lvtkCommon -lvtksys" )
  INCLUDE_DIRECTORIES(${VTK_INCLUDE_DIRS})
  MARK_AS_ADVANCED( VTK_DIR )
  SET(FEELPP_LIBRARIES ${VTK_LIBRARIES} ${FEELPP_LIBRARIES})
  SET(FEELPP_ENABLED_OPTIONS "${FEELPP_ENABLED_OPTIONS} VTK" )
endif()

#
# Octave
#
if ( FEELPP_ENABLE_OCTAVE )
  FIND_PACKAGE(Octave)
  if ( OCTAVE_FOUND )

    # find octave-config and get variables from it
    FIND_PROGRAM(OCTAVE_CONFIG octave-config)
    IF(NOT OCTAVE_CONFIG)
      MESSAGE(FATAL_ERROR "Octave is required, but octave-config was not found.  Please install Octave and rerun cmake.")
    ENDIF(NOT OCTAVE_CONFIG)

    EXECUTE_PROCESS(COMMAND ${OCTAVE_CONFIG} --oct-site-dir
      OUTPUT_VARIABLE OCT_SITE_DIR OUTPUT_STRIP_TRAILING_WHITESPACE)
    EXECUTE_PROCESS(COMMAND ${OCTAVE_CONFIG} --m-site-dir
      OUTPUT_VARIABLE M_SITE_DIR OUTPUT_STRIP_TRAILING_WHITESPACE)
    EXECUTE_PROCESS(COMMAND ${OCTAVE_CONFIG} -p OCTINCLUDEDIR
      OUTPUT_VARIABLE OCTINCLUDEDIR OUTPUT_STRIP_TRAILING_WHITESPACE)
    EXECUTE_PROCESS(COMMAND ${OCTAVE_CONFIG} -p OCTLIBDIR
      OUTPUT_VARIABLE OCTLIBDIR OUTPUT_STRIP_TRAILING_WHITESPACE)

    # Make the values accessible from other CMakeLists.txt files
    # Also, this allows packagers to override the default values
    SET(FEELPP_OCT_DIR ${OCT_SITE_DIR}/feel++ CACHE PATH ".oct files from Feel++")
    SET(FEELPP_M_DIR ${M_SITE_DIR}/feel++ CACHE PATH ".m files from Feel++")

    message(STATUS "oct dir: ${FEELPP_OCT_DIR}" )
    message(STATUS "m dir: ${FEELPP_M_DIR}" )
    message(STATUS "include dir: ${OCTINCLUDEDIR}" )


    INCLUDE_DIRECTORIES( ${OCTINCLUDEDIR} )

    SET(FEELPP_ENABLED_OPTIONS "${FEELPP_ENABLED_OPTIONS} Octave" )
  endif( OCTAVE_FOUND )
endif( FEELPP_ENABLE_OCTAVE)

#
# Gmsh
#
FIND_PACKAGE(Gmsh REQUIRED)
if ( GMSH_FOUND )
  ADD_DEFINITIONS( -DFEELPP_HAS_GMSH=1 -D_FEELPP_HAS_GMSH_ -DGMSH_EXECUTABLE=${GMSH_EXECUTABLE} )
  if ( GL2PS_LIBRARY )
    if ( GL_LIBRARY AND FEELPP_ENABLE_OPENGL )
      SET(FEELPP_LIBRARIES ${GMSH_LIBRARY} ${GL2PS_LIBRARY} ${GL_LIBRARY} ${FEELPP_LIBRARIES})
    else()
      SET(FEELPP_LIBRARIES ${GMSH_LIBRARY} ${GL2PS_LIBRARY} ${FEELPP_LIBRARIES})
    endif()
  else()
    SET(FEELPP_LIBRARIES ${GMSH_LIBRARY} ${FEELPP_LIBRARIES})
  endif()
  include_directories(${GMSH_INCLUDE_DIR})
  SET(FEELPP_ENABLED_OPTIONS "${FEELPP_ENABLED_OPTIONS} Gmsh" )
endif()


#
# if Feel++ has been installed on the system
#
if ( NOT EXISTS ${CMAKE_SOURCE_DIR}/feel OR NOT EXISTS ${CMAKE_SOURCE_DIR}/contrib )
  include(feelpp.macros)
  FIND_PATH(FEELPP_INCLUDE_DIR feel/feelconfig.h  PATHS $ENV{FEELPP_DIR}/include/ /usr/include /opt/local/include PATH_SUFFIXES feel )

  #  FIND_LIBRARY(FEELPP_GFLAGS_LIBRARY feelpp_gflags PATHS $ENV{FEELPP_DIR}/lib /usr/lib /usr/lib/feel/lib /opt/feel/lib /usr/ljk/lib )
  #  FIND_LIBRARY(FEELPP_GLOG_LIBRARY feelpp_glog PATHS $ENV{FEELPP_DIR}/lib /usr/lib /usr/lib/feel/lib /opt/feel/lib /usr/ljk/lib )
  #  FIND_LIBRARY(FEELPP_CLN_LIBRARY feelpp_cln PATHS $ENV{FEELPP_DIR}/lib /usr/lib /usr/lib/feel/lib /opt/feel/lib /usr/ljk/lib )
  FIND_LIBRARY(FEELPP_GINAC_LIBRARY feelpp_ginac PATHS $ENV{FEELPP_DIR}/lib /usr/lib /usr/lib/feel/lib /opt/feel/lib /usr/ljk/lib )
  FIND_LIBRARY(FEELPP_LIBRARY feelpp PATHS $ENV{FEELPP_DIR}/lib /usr/lib /usr/lib/feel/lib /opt/feel/lib /usr/ljk/lib )

  INCLUDE_DIRECTORIES ( ${FEELPP_INCLUDE_DIR} ${FEELPP_INCLUDE_DIR}/feel )
  FIND_PACKAGE_HANDLE_STANDARD_ARGS (Feel DEFAULT_MSG
    FEELPP_INCLUDE_DIR  FEELPP_LIBRARY
    )



  if ( FEELPP_FOUND )
    message(STATUS "Feel++ includes: ${FEELPP_INCLUDE_DIR}")
    message(STATUS "Feel++ library: ${FEELPP_LIBRARY}")
  endif()

  MARK_AS_ADVANCED(
    FEELPP_INCLUDE_DIR
    FEELPP_LIBRARY
    )
  SET(FEELPP_LIBRARIES ${FEELPP_LIBRARY} ${FEELPP_GINAC_LIBRARY}  ${FEELPP_LIBRARIES})
else()
  INCLUDE_DIRECTORIES (
    ${FEELPP_BUILD_DIR}/
    ${FEELPP_SOURCE_DIR}/
    ${FEELPP_SOURCE_DIR}/contrib/gmm/include
    )
  SET(FEELPP_LIBRARIES feelpp  ${FEELPP_LIBRARIES})
endif()



LINK_DIRECTORIES(
  ${VTK_LIBRARY_DIRS}
  ${BOOST_LIBRARY_PATH}
  ${MPI_LIBRARY_PATH}
  )


MARK_AS_ADVANCED(FEELPP_LIBRARIES)
