/*
 * $Id: atapi.c,v 1.1 2012-07-03 07:09:10 vrsieh Exp $ 
 *
 * Copyright (C) 2004-2012 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "build_config.h"
#include "compiler.h"

/* ==================== RUNTIME_RM ==================== */
#if defined(RUNTIME_RM)

#define IDE_ERR_STAT    0x01
#define IDE_INDEX_STAT  0x02
#define IDE_ECC_STAT    0x04
#define IDE_DRQ_STAT    0x08
#define IDE_SEEK_STAT   0x10
#define IDE_WRERR_STAT  0x20
#define IDE_READY_STAT  0x40
#define IDE_BUSY_STAT   0x80

extern const uint16_t ide_port_table[];

#endif /* RUNTIME_RM */
/* ==================== RUNTIME_RM ==================== */
#ifdef RUNTIME_RM

CODE16;

#include "assert.h"
#include "fixme.h"
#include "stdio.h"
#include "string.h"
#include "in.h"

#include "io.h"
#include "const.h"
#include "cmos.h"
#include "var.h"
#include "el_torito.h"
#include "ptrace.h"
#include "disk.h"

#define ATAPI_TEST_UNIT_READY	0x00
#define ATAPI_REQUEST_SENSE	0x03
#define ATAPI_READ_SECTOR	0x28

#define ATAPI_NO_SENSE		0
#define ATAPI_NOT_READY		2
#define ATAPI_UNIT_ATTENTION	6

struct atapi_compacket {
	uint8_t opcode;
	uint8_t reserved0;
	uint32_t lba;
	uint8_t reserved1;
	union {
		uint16_t transfer_length;
		uint16_t paramlist_length;
		uint16_t allocation_length;
	} misc;
	uint8_t reserved2;
	uint16_t reserved3;
} __attribute__((__packed__));

struct atapi_request_sense {
	/* Byte 0: */
	/*	Bit 0-6: error code */
	/*	Bit 7: valid */
	uint8_t error_code;
	/* Byte 1: */
	uint8_t segment_number;
	/* Byte 2: */
	/*	Bit 0-3: sense_key */
	/*	Bit 4: reserved */
	/*	Bit 5: ili */
	/*	Bit 6-7: reserved */
	uint8_t sense_key;
	/* Byte 3-6: */
	uint8_t information[4];
	/* Byte 7: */
	uint8_t add_sense_len;
	/* Byte 8-11: */
	uint8_t command_info[4];
	/* Byte 12: */
	uint8_t asc;
	/* Byte 13: */
	uint8_t ascq;
	/* Byte 14: */
	uint8_t fruc;
	/* Byte 15-17: */
	uint8_t sks[3];
} __attribute__((__packed__));

static void
outsw(uint16_t port, uint16_t *addr, uint32_t count)
{
	for ( ; 0 < count; count--) {
		outw(*addr++, port);
	}
}

static uint8_t
send_atapi_pc(
	uint16_t unit,
	uint16_t port,
	struct atapi_compacket *command
)
{
	uint8_t status;

	/* Select drive. */
	outb(unit << 4, port + 6);

	/* Wait while busy. */
	do {
		status = inb(port + 0x206);
	} while (status & IDE_BUSY_STAT);

	/* Send packet command. */
	outb(0x00, port + 1);		/* No DMA, no overlay */
	outb(2048 % 256, port + 4);	/* 2048 bytes transfer */
	outb(2048 / 256, port + 5);
	outb(0xa0, port + 7);		/* Packet command */

	/* Wait while busy. */
	do {
		status = inb(port + 0x206);
	} while (status & IDE_BUSY_STAT);
	if (status & IDE_ERR_STAT) {
		/* Use sense command - FIXME MARCEL */
		return 1;
	}
	assert(status & IDE_DRQ_STAT);

	assert(((inb(port + 2) >> 0) & 1) == 1); /* Command */
	assert(((inb(port + 2) >> 1) & 1) == 0); /* Out */
	assert(inb(port + 4) == 2048 % 256);
	assert(inb(port + 5) == 2048 / 256);

	/* Send command packet. */
	outsw(port, (void *) command, sizeof(struct atapi_compacket) / 2);

	/* Wait while busy. */
	do {
		status = inb(port + 0x206);
	} while (status & IDE_BUSY_STAT);
	if (status & IDE_ERR_STAT) {
		return 1;
	}

	return 0;
}

static uint8_t
atapi_request_sense(
	uint16_t unit,
	uint16_t port,
	struct atapi_request_sense *rs
)
{
	uint16_t *buf = (uint16_t *) rs;
	uint8_t command[12];
	uint8_t status;
	uint16_t count;

	for (count = 0; count < sizeof(command) / sizeof(command[0]); count++) {
		command[count] = 0;
	}
	command[0] = ATAPI_REQUEST_SENSE;
	command[4] = sizeof(*rs);

	/*
	 * Send ATAPI command.
	 */
	if (send_atapi_pc(unit, port, (struct atapi_compacket *) command)) {
		return 1;
	}

	/*
	 * Wait for result.
	 */
	do {
		status = inb(port + 0x206);
	} while (status & IDE_BUSY_STAT);
	if (status & IDE_ERR_STAT) {
		return 1;
	}

	/*
	 * Get result.
	 */
	for (count = 0; count < sizeof(*rs) / 2; count++) {
		assert(status & IDE_DRQ_STAT);
		*buf++ = inw(port);
	}

	return 0;
}

uint8_t
read_sector_lba_atapi(
	uint8_t device,
	uint16_t num,
	uint32_t /* long long */ block,
	uint16_t buffer_seg,
	uint16_t buffer_off,
	uint16_t sector_size
)
{
	uint16_t port;
	uint16_t atapi_num;
	uint16_t start_dummy;
	uint16_t end_dummy;
	uint16_t ret;
	uint16_t count;
	uint16_t sector_count;
	uint16_t data;
	union {
		struct atapi_compacket command;
		struct atapi_request_sense rs;
	} u;
	uint8_t status;

again:	;
	assert(/* 0 <= device / 2 && */ device / 2 < 6);
	assert(/* 0 <= device % 2 && */ device % 2 < 2);

	/* Get port addresses. */
	port = const_get(ide_port_table[device / 2]);

	/*
	 * Send read command.
	 */
	u.command.opcode = ATAPI_READ_SECTOR;
	u.command.reserved0 = 0;
	u.command.reserved1 = 0;
	u.command.reserved2 = 0;
	u.command.reserved3 = 0;
	if (sector_size == 2048) {
		/* CDROM mode */
		u.command.lba = htonl(block);
		start_dummy = 0;
		end_dummy = 0;
		atapi_num = num;

	} else {
		/* floppy emulation mode */
		uint32_t slba;
		uint32_t elba;

		assert(sector_size == 512);

		slba = block;
		elba = block + num;

		u.command.lba = htonl(slba / 4);
		start_dummy = slba % 4;
		if (elba % 4 == 0) {
			end_dummy = 0;
		} else {
			end_dummy = 4 - elba % 4;
		}
		atapi_num = elba / 4 - slba / 4;
		if (elba % 4 != 0) {
			atapi_num++;
		}
	}
	u.command.misc.transfer_length = htons(atapi_num);

	ret = send_atapi_pc(device % 2, port, &u.command);
	if (ret != 0) {
		ret = atapi_request_sense(device % 2, port, &u.rs);
		if (ret != 0) {
			return ret;
		}
		if ((u.rs.sense_key & 0xf) == ATAPI_UNIT_ATTENTION) {
			goto again;
		}
		return 1;
	}

	/* read dummy data */
	for (sector_count = 0; sector_count < start_dummy; sector_count++) {
		do {
			status = inb(port + 0x206);
		} while (status & IDE_BUSY_STAT);
		if (status & IDE_ERR_STAT) {
			return 1;
		}
		assert(status & IDE_DRQ_STAT);
		for (count = 0; count < sector_size / 2; count++) {
			data = inw(port);
		}
	}
	/*
	 * Get wanted sectors.
	 */
	for (sector_count = 0; sector_count < num; sector_count++) {
		do {
			status = inb(port + 0x206);
		} while (status & IDE_BUSY_STAT);
		if (status & IDE_ERR_STAT) {
			return 1;
		}
		assert(status & IDE_DRQ_STAT);
		for (count = 0; count < sector_size / 2; count++) {
			data = inw(port);
			put_word(buffer_seg, buffer_off, data);
			buffer_off += 2;
			if (16 <= buffer_off) {
				buffer_seg++;
				buffer_off -= 16;
			}
		}
	}
	/* read dummy data */
	for (sector_count = 0; sector_count < end_dummy; sector_count++) {
		do {
			status = inb(port + 0x206);
		} while (status & IDE_BUSY_STAT);
		if (status & IDE_ERR_STAT) {
			return 1;
		}
		assert(status & IDE_DRQ_STAT);
		for (count = 0; count < sector_size / 2; count++) {
			data = inw(port);
		}
	}

	return 0;
}

#endif /* RUNTIME_RM */
