//===- CIRTypes.cpp - MLIR CIR Types --------------------------------------===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
//
//===----------------------------------------------------------------------===//
//
// This file defines the types in the CIR dialect.
//
//===----------------------------------------------------------------------===//

#include "clang/CIR/Dialect/IR/CIRTypes.h"

#include "mlir/IR/DialectImplementation.h"
#include "clang/CIR/Dialect/IR/CIRDialect.h"
#include "llvm/ADT/TypeSwitch.h"

//===----------------------------------------------------------------------===//
// CIR Custom Parser/Printer Signatures
//===----------------------------------------------------------------------===//

static mlir::ParseResult
parseFuncTypeArgs(mlir::AsmParser &p, llvm::SmallVector<mlir::Type> &params,
                  bool &isVarArg);
static void printFuncTypeArgs(mlir::AsmPrinter &p,
                              mlir::ArrayRef<mlir::Type> params, bool isVarArg);

//===----------------------------------------------------------------------===//
// Get autogenerated stuff
//===----------------------------------------------------------------------===//

#define GET_TYPEDEF_CLASSES
#include "clang/CIR/Dialect/IR/CIROpsTypes.cpp.inc"

using namespace mlir;
using namespace cir;

//===----------------------------------------------------------------------===//
// General CIR parsing / printing
//===----------------------------------------------------------------------===//

Type CIRDialect::parseType(DialectAsmParser &parser) const {
  llvm::SMLoc typeLoc = parser.getCurrentLocation();
  llvm::StringRef mnemonic;
  Type genType;

  // Try to parse as a tablegen'd type.
  OptionalParseResult parseResult =
      generatedTypeParser(parser, &mnemonic, genType);
  if (parseResult.has_value())
    return genType;

  // TODO(CIR) Attempt to parse as a raw C++ type.
  parser.emitError(typeLoc) << "unknown CIR type: " << mnemonic;
  return Type();
}

void CIRDialect::printType(Type type, DialectAsmPrinter &os) const {
  // Try to print as a tablegen'd type.
  if (generatedTypePrinter(type, os).succeeded())
    return;

  // TODO(CIR) Attempt to print as a raw C++ type.
  llvm::report_fatal_error("printer is missing a handler for this type");
}

//===----------------------------------------------------------------------===//
// IntType Definitions
//===----------------------------------------------------------------------===//

Type IntType::parse(mlir::AsmParser &parser) {
  mlir::MLIRContext *context = parser.getBuilder().getContext();
  llvm::SMLoc loc = parser.getCurrentLocation();
  bool isSigned;
  unsigned width;

  if (parser.parseLess())
    return {};

  // Fetch integer sign.
  llvm::StringRef sign;
  if (parser.parseKeyword(&sign))
    return {};
  if (sign == "s")
    isSigned = true;
  else if (sign == "u")
    isSigned = false;
  else {
    parser.emitError(loc, "expected 's' or 'u'");
    return {};
  }

  if (parser.parseComma())
    return {};

  // Fetch integer size.
  if (parser.parseInteger(width))
    return {};
  if (width < IntType::minBitwidth() || width > IntType::maxBitwidth()) {
    parser.emitError(loc, "expected integer width to be from ")
        << IntType::minBitwidth() << " up to " << IntType::maxBitwidth();
    return {};
  }

  if (parser.parseGreater())
    return {};

  return IntType::get(context, width, isSigned);
}

void IntType::print(mlir::AsmPrinter &printer) const {
  char sign = isSigned() ? 's' : 'u';
  printer << '<' << sign << ", " << getWidth() << '>';
}

llvm::TypeSize
IntType::getTypeSizeInBits(const mlir::DataLayout &dataLayout,
                           mlir::DataLayoutEntryListRef params) const {
  return llvm::TypeSize::getFixed(getWidth());
}

uint64_t IntType::getABIAlignment(const mlir::DataLayout &dataLayout,
                                  mlir::DataLayoutEntryListRef params) const {
  return (uint64_t)(getWidth() / 8);
}

uint64_t
IntType::getPreferredAlignment(const ::mlir::DataLayout &dataLayout,
                               ::mlir::DataLayoutEntryListRef params) const {
  return (uint64_t)(getWidth() / 8);
}

mlir::LogicalResult
IntType::verify(llvm::function_ref<mlir::InFlightDiagnostic()> emitError,
                unsigned width, bool isSigned) {
  if (width < IntType::minBitwidth() || width > IntType::maxBitwidth()) {
    emitError() << "IntType only supports widths from "
                << IntType::minBitwidth() << " up to "
                << IntType::maxBitwidth();
    return mlir::failure();
  }
  return mlir::success();
}

//===----------------------------------------------------------------------===//
// Floating-point type definitions
//===----------------------------------------------------------------------===//

const llvm::fltSemantics &SingleType::getFloatSemantics() const {
  return llvm::APFloat::IEEEsingle();
}

llvm::TypeSize
SingleType::getTypeSizeInBits(const mlir::DataLayout &dataLayout,
                              mlir::DataLayoutEntryListRef params) const {
  return llvm::TypeSize::getFixed(getWidth());
}

uint64_t
SingleType::getABIAlignment(const mlir::DataLayout &dataLayout,
                            mlir::DataLayoutEntryListRef params) const {
  return (uint64_t)(getWidth() / 8);
}

uint64_t
SingleType::getPreferredAlignment(const ::mlir::DataLayout &dataLayout,
                                  ::mlir::DataLayoutEntryListRef params) const {
  return (uint64_t)(getWidth() / 8);
}

const llvm::fltSemantics &DoubleType::getFloatSemantics() const {
  return llvm::APFloat::IEEEdouble();
}

llvm::TypeSize
DoubleType::getTypeSizeInBits(const mlir::DataLayout &dataLayout,
                              mlir::DataLayoutEntryListRef params) const {
  return llvm::TypeSize::getFixed(getWidth());
}

uint64_t
DoubleType::getABIAlignment(const mlir::DataLayout &dataLayout,
                            mlir::DataLayoutEntryListRef params) const {
  return (uint64_t)(getWidth() / 8);
}

uint64_t
DoubleType::getPreferredAlignment(const ::mlir::DataLayout &dataLayout,
                                  ::mlir::DataLayoutEntryListRef params) const {
  return (uint64_t)(getWidth() / 8);
}

const llvm::fltSemantics &FP16Type::getFloatSemantics() const {
  return llvm::APFloat::IEEEhalf();
}

llvm::TypeSize
FP16Type::getTypeSizeInBits(const mlir::DataLayout &dataLayout,
                            mlir::DataLayoutEntryListRef params) const {
  return llvm::TypeSize::getFixed(getWidth());
}

uint64_t FP16Type::getABIAlignment(const mlir::DataLayout &dataLayout,
                                   mlir::DataLayoutEntryListRef params) const {
  return (uint64_t)(getWidth() / 8);
}

uint64_t
FP16Type::getPreferredAlignment(const ::mlir::DataLayout &dataLayout,
                                ::mlir::DataLayoutEntryListRef params) const {
  return (uint64_t)(getWidth() / 8);
}

const llvm::fltSemantics &BF16Type::getFloatSemantics() const {
  return llvm::APFloat::BFloat();
}

llvm::TypeSize
BF16Type::getTypeSizeInBits(const mlir::DataLayout &dataLayout,
                            mlir::DataLayoutEntryListRef params) const {
  return llvm::TypeSize::getFixed(getWidth());
}

uint64_t BF16Type::getABIAlignment(const mlir::DataLayout &dataLayout,
                                   mlir::DataLayoutEntryListRef params) const {
  return (uint64_t)(getWidth() / 8);
}

uint64_t
BF16Type::getPreferredAlignment(const ::mlir::DataLayout &dataLayout,
                                ::mlir::DataLayoutEntryListRef params) const {
  return (uint64_t)(getWidth() / 8);
}

const llvm::fltSemantics &FP80Type::getFloatSemantics() const {
  return llvm::APFloat::x87DoubleExtended();
}

llvm::TypeSize
FP80Type::getTypeSizeInBits(const mlir::DataLayout &dataLayout,
                            mlir::DataLayoutEntryListRef params) const {
  // Though only 80 bits are used for the value, the type is 128 bits in size.
  return llvm::TypeSize::getFixed(128);
}

uint64_t FP80Type::getABIAlignment(const mlir::DataLayout &dataLayout,
                                   mlir::DataLayoutEntryListRef params) const {
  return 16;
}

uint64_t
FP80Type::getPreferredAlignment(const ::mlir::DataLayout &dataLayout,
                                ::mlir::DataLayoutEntryListRef params) const {
  return 16;
}

const llvm::fltSemantics &FP128Type::getFloatSemantics() const {
  return llvm::APFloat::IEEEquad();
}

llvm::TypeSize
FP128Type::getTypeSizeInBits(const mlir::DataLayout &dataLayout,
                             mlir::DataLayoutEntryListRef params) const {
  return llvm::TypeSize::getFixed(getWidth());
}

uint64_t FP128Type::getABIAlignment(const mlir::DataLayout &dataLayout,
                                    mlir::DataLayoutEntryListRef params) const {
  return 16;
}

uint64_t
FP128Type::getPreferredAlignment(const ::mlir::DataLayout &dataLayout,
                                 ::mlir::DataLayoutEntryListRef params) const {
  return 16;
}

const llvm::fltSemantics &LongDoubleType::getFloatSemantics() const {
  return mlir::cast<cir::CIRFPTypeInterface>(getUnderlying())
      .getFloatSemantics();
}

llvm::TypeSize
LongDoubleType::getTypeSizeInBits(const mlir::DataLayout &dataLayout,
                                  mlir::DataLayoutEntryListRef params) const {
  return mlir::cast<mlir::DataLayoutTypeInterface>(getUnderlying())
      .getTypeSizeInBits(dataLayout, params);
}

uint64_t
LongDoubleType::getABIAlignment(const mlir::DataLayout &dataLayout,
                                mlir::DataLayoutEntryListRef params) const {
  return mlir::cast<mlir::DataLayoutTypeInterface>(getUnderlying())
      .getABIAlignment(dataLayout, params);
}

uint64_t LongDoubleType::getPreferredAlignment(
    const ::mlir::DataLayout &dataLayout,
    mlir::DataLayoutEntryListRef params) const {
  return mlir::cast<mlir::DataLayoutTypeInterface>(getUnderlying())
      .getPreferredAlignment(dataLayout, params);
}

LogicalResult
LongDoubleType::verify(function_ref<InFlightDiagnostic()> emitError,
                       mlir::Type underlying) {
  if (!mlir::isa<DoubleType, FP80Type, FP128Type>(underlying)) {
    emitError() << "invalid underlying type for long double";
    return failure();
  }

  return success();
}

//===----------------------------------------------------------------------===//
// Floating-point type helpers
//===----------------------------------------------------------------------===//

bool cir::isAnyFloatingPointType(mlir::Type t) {
  return isa<cir::SingleType, cir::DoubleType, cir::LongDoubleType,
             cir::FP80Type, cir::BF16Type, cir::FP16Type, cir::FP128Type>(t);
}

//===----------------------------------------------------------------------===//
// FuncType Definitions
//===----------------------------------------------------------------------===//

FuncType FuncType::clone(TypeRange inputs, TypeRange results) const {
  assert(results.size() == 1 && "expected exactly one result type");
  return get(llvm::to_vector(inputs), results[0], isVarArg());
}

mlir::ParseResult parseFuncTypeArgs(mlir::AsmParser &p,
                                    llvm::SmallVector<mlir::Type> &params,
                                    bool &isVarArg) {
  isVarArg = false;
  // `(` `)`
  if (succeeded(p.parseOptionalRParen()))
    return mlir::success();

  // `(` `...` `)`
  if (succeeded(p.parseOptionalEllipsis())) {
    isVarArg = true;
    return p.parseRParen();
  }

  // type (`,` type)* (`,` `...`)?
  mlir::Type type;
  if (p.parseType(type))
    return mlir::failure();
  params.push_back(type);
  while (succeeded(p.parseOptionalComma())) {
    if (succeeded(p.parseOptionalEllipsis())) {
      isVarArg = true;
      return p.parseRParen();
    }
    if (p.parseType(type))
      return mlir::failure();
    params.push_back(type);
  }

  return p.parseRParen();
}

void printFuncTypeArgs(mlir::AsmPrinter &p, mlir::ArrayRef<mlir::Type> params,
                       bool isVarArg) {
  llvm::interleaveComma(params, p,
                        [&p](mlir::Type type) { p.printType(type); });
  if (isVarArg) {
    if (!params.empty())
      p << ", ";
    p << "...";
  }
  p << ')';
}

llvm::ArrayRef<mlir::Type> FuncType::getReturnTypes() const {
  return static_cast<detail::FuncTypeStorage *>(getImpl())->returnType;
}

bool FuncType::isVoid() const { return mlir::isa<VoidType>(getReturnType()); }

//===----------------------------------------------------------------------===//
// PointerType Definitions
//===----------------------------------------------------------------------===//

llvm::TypeSize
PointerType::getTypeSizeInBits(const ::mlir::DataLayout &dataLayout,
                               ::mlir::DataLayoutEntryListRef params) const {
  // FIXME: improve this in face of address spaces
  return llvm::TypeSize::getFixed(64);
}

uint64_t
PointerType::getABIAlignment(const ::mlir::DataLayout &dataLayout,
                             ::mlir::DataLayoutEntryListRef params) const {
  // FIXME: improve this in face of address spaces
  return 8;
}

uint64_t PointerType::getPreferredAlignment(
    const ::mlir::DataLayout &dataLayout,
    ::mlir::DataLayoutEntryListRef params) const {
  // FIXME: improve this in face of address spaces
  return 8;
}

mlir::LogicalResult
PointerType::verify(llvm::function_ref<mlir::InFlightDiagnostic()> emitError,
                    mlir::Type pointee) {
  // TODO(CIR): Verification of the address space goes here.
  return mlir::success();
}

//===----------------------------------------------------------------------===//
// CIR Dialect
//===----------------------------------------------------------------------===//

void CIRDialect::registerTypes() {
  // Register tablegen'd types.
  addTypes<
#define GET_TYPEDEF_LIST
#include "clang/CIR/Dialect/IR/CIROpsTypes.cpp.inc"
      >();

  // Register raw C++ types.
  // TODO(CIR) addTypes<StructType>();
}
